/*
 * Copyright (c) 2016, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jdk.internal.jshell.tool;

import java.util.*;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jdk.internal.jshell.tool.JShellTool.CompletionProvider;

import static java.util.stream.Collectors.*;
import static jdk.internal.jshell.tool.ContinuousCompletionProvider.PERFECT_MATCHER;
import static jdk.internal.jshell.tool.JShellTool.EMPTY_COMPLETION_PROVIDER;
import static jdk.internal.jshell.tool.Selector.SelectorKind;
import static jdk.internal.jshell.tool.Selector.SelectorInstanceWithDoc;
import static jdk.internal.jshell.tool.Selector.SelectorBuilder;
import static jdk.internal.jshell.tool.Selector.FormatAction;
import static jdk.internal.jshell.tool.Selector.FormatCase;
import static jdk.internal.jshell.tool.Selector.FormatErrors;
import static jdk.internal.jshell.tool.Selector.FormatResolve;
import static jdk.internal.jshell.tool.Selector.FormatUnresolved;
import static jdk.internal.jshell.tool.Selector.FormatWhen;


/**
 * Feedback customization support
 *
 * @author Robert Field
 */
class Feedback {

    // Patern for substituted fields within a customized format string
    private static final Pattern FIELD_PATTERN = Pattern.compile("\\{(.*?)\\}");

    // Internal field name for truncation length
    private static final String TRUNCATION_FIELD = "<truncation>";

    // For encoding to Properties String
    private static final String RECORD_SEPARATOR = "\u241E";

    // Selector for truncation of var value
    private static final Selector VAR_VALUE_ADD_SELECTOR = new Selector(
            FormatCase.VARVALUE,
            FormatAction.ADDED,
            FormatWhen.PRIMARY,
            FormatResolve.OK,
            FormatUnresolved.UNRESOLVED0,
            FormatErrors.ERROR0);

    // Selector for typeKind record
    private static final Selector RECORD_TYPEKIND_SELECTOR = new Selector(
            EnumSet.of(FormatCase.RECORD),
            EnumSet.noneOf(FormatAction.class),
            EnumSet.noneOf(FormatWhen.class),
            EnumSet.noneOf(FormatResolve.class),
            EnumSet.noneOf(FormatUnresolved.class),
            EnumSet.noneOf(FormatErrors.class));

    // Current mode -- initial value is placeholder during start-up
    private Mode mode = new Mode("");

    // Retained current mode -- for checks
    private Mode retainedCurrentMode = null;

    // Mapping of mode name to mode
    private final Map<String, Mode> modeMap = new HashMap<>();

    // Mapping of mode names to encoded retained mode
    private final Map<String, String> retainedMap = new HashMap<>();

    public boolean shouldDisplayCommandFluff() {
        return mode.commandFluff;
    }

    public String getPre() {
        return mode.format("pre", Selector.ANY);
    }

    public String getPost() {
        return mode.format("post", Selector.ANY);
    }

    public String getErrorPre() {
        return mode.format("errorpre", Selector.ANY);
    }

    public String getErrorPost() {
        return mode.format("errorpost", Selector.ANY);
    }

    public String format(FormatCase fc, FormatAction fa, FormatWhen fw,
                         FormatResolve fr, FormatUnresolved fu, FormatErrors fe,
                         String name, String type, String value, String unresolved, List<String> errorLines) {
        return mode.format(fc, fa, fw, fr, fu, fe,
                name, type, value, unresolved, errorLines);
    }

    public String format(String field, FormatCase fc, FormatAction fa, FormatWhen fw,
                         FormatResolve fr, FormatUnresolved fu, FormatErrors fe,
                         String name, String type, String value, String unresolved, List<String> errorLines) {
        return mode.format(field, fc, fa, fw, fr, fu, fe,
                name, type, value, unresolved, errorLines);
    }

    public String truncateVarValue(String value) {
        return mode.truncateVarValue(value);
    }

    public String getPrompt(String nextId) {
        return mode.getPrompt(nextId);
    }

    public String getContinuationPrompt(String nextId) {
        return mode.getContinuationPrompt(nextId);
    }

    public boolean setFeedback(MessageHandler messageHandler, ArgTokenizer at, Consumer<String> retainer) {
        return new Setter(messageHandler, at).setFeedback(retainer);
    }

    public boolean setFormat(MessageHandler messageHandler, ArgTokenizer at) {
        return new Setter(messageHandler, at).setFormat();
    }

    public boolean setTruncation(MessageHandler messageHandler, ArgTokenizer at) {
        return new Setter(messageHandler, at).setTruncation();
    }

    public boolean setMode(MessageHandler messageHandler, ArgTokenizer at, Consumer<String> retainer) {
        return new Setter(messageHandler, at).setMode(retainer);
    }

    public boolean setPrompt(MessageHandler messageHandler, ArgTokenizer at) {
        return new Setter(messageHandler, at).setPrompt();
    }

    public boolean restoreEncodedModes(MessageHandler messageHandler, String encoded) {
        return new Setter(messageHandler, new ArgTokenizer("<init>", "")).restoreEncodedModes(encoded);
    }

    public void markModesReadOnly() {
        modeMap.values()
                .forEach(m -> m.readOnly = true);
    }

    JShellTool.CompletionProvider modeCompletions() {
        return modeCompletions(EMPTY_COMPLETION_PROVIDER);
    }

    JShellTool.CompletionProvider modeCompletions(CompletionProvider successor) {
        return new ContinuousCompletionProvider(
                () -> modeMap.keySet().stream()
                        .collect(toMap(Function.identity(), m -> successor)),
                PERFECT_MATCHER);
    }

    /**
     * Holds all the context of a mode mode
     */
    private static class Mode {

        // Name of mode
        final String name;

        // Display command verification/information
        boolean commandFluff;

        // Setting (including format) by field
        final Map<String, List<Setting>> byField;

        boolean readOnly = false;

        String prompt = "\n-> ";
        String continuationPrompt = ">> ";

        static class Setting {

            final String format;
            final Selector selector;

            Setting(String format, Selector selector) {
                this.format = format;
                this.selector = selector;
            }

            @Override
            public boolean equals(Object o) {
                if (o instanceof Setting) {
                    Setting ing = (Setting) o;
                    return format.equals(ing.format)
                            && selector.equals(ing.selector);
                } else {
                    return false;
                }
            }

            @Override
            public int hashCode() {
                int hash = 7;
                hash = 67 * hash + Objects.hashCode(this.selector);
                hash = 67 * hash + Objects.hashCode(this.format);
                return hash;
            }

            @Override
            public String toString() {
                return "Setting(" + format + "," + selector.toString() + ")";
            }
        }

        /**
         * Set up an empty mode.
         *
         * @param name
         */
        Mode(String name) {
            this.name = name;
            this.byField = new HashMap<>();
            set("name", "%1$s", Selector.ALWAYS);
            set("type", "%2$s", Selector.ALWAYS);
            set("value", "%3$s", Selector.ALWAYS);
            set("unresolved", "%4$s", Selector.ALWAYS);
            set("errors", "%5$s", Selector.ALWAYS);
            set("err", "%6$s", Selector.ALWAYS);

            set("errorline", "    {err}%n", Selector.ALWAYS);

            set("pre", "|  ", Selector.ALWAYS);
            set("post", "%n", Selector.ALWAYS);
            set("errorpre", "|  ", Selector.ALWAYS);
            set("errorpost", "%n", Selector.ALWAYS);
        }

        private Mode(String name, boolean commandFluff, String prompt, String continuationPrompt) {
            this.name = name;
            this.commandFluff = commandFluff;
            this.prompt = prompt;
            this.continuationPrompt = continuationPrompt;
            this.byField = new HashMap<>();
        }

        /**
         * Set up a copied mode.
         *
         * @param name
         * @param m    Mode to copy, or null for no fresh
         */
        Mode(String name, Mode m) {
            this(name, m.commandFluff, m.prompt, m.continuationPrompt);
            m.byField.forEach((fieldName, settingList) ->
                    settingList.forEach(setting -> set(fieldName, setting.format, setting.selector)));

        }

        @Override
        public boolean equals(Object o) {
            if (o instanceof Mode) {
                Mode m = (Mode) o;
                return name.equals((m.name))
                        && commandFluff == m.commandFluff
                        && prompt.equals((m.prompt))
                        && continuationPrompt.equals((m.continuationPrompt))
                        && byField.equals((m.byField));
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(name);
        }

        /**
         * Set if this mode displays informative/confirmational messages on
         * commands.
         *
         * @param fluff the value to set
         */
        void setCommandFluff(boolean fluff) {
            commandFluff = fluff;
        }

        /**
         * Encodes the mode into a String so it can be saved in Preferences.
         *
         * @return the string representation
         */
        String encode() {
            List<String> el = new ArrayList<>();
            el.add(name);
            el.add(String.valueOf(commandFluff));
            el.add(prompt);
            el.add(continuationPrompt);
            for (Entry<String, List<Setting>> es : byField.entrySet()) {
                el.add(es.getKey());
                el.add("(");
                for (Setting ing : es.getValue()) {
                    el.add(ing.selector.toString());
                    el.add(ing.format);
                }
                el.add(")");
            }
            el.add("***");
            return String.join(RECORD_SEPARATOR, el);
        }

        private void add(String field, Setting ing) {
            List<Setting> settings = byField.get(field);
            if (settings == null) {
                settings = new ArrayList<>();
                byField.put(field, settings);
            } else {
                // remove completely obscured settings.
                // transformation of partially obscured would be confusing to user and complex
                Selector addedSelector = ing.selector;
                settings.removeIf(t -> t.selector.includedIn(addedSelector));
            }
            settings.add(ing);
        }

        void set(String field, String format, Selector selector) {
            add(field, new Setting(format, selector));
        }

        /**
         * Lookup format Replace fields with context specific formats.
         *
         * @return format string
         */
        String format(String field, Selector selector) {
            List<Setting> settings = byField.get(field);
            if (settings == null) {
                return ""; //TODO error?
            }
            String format = null;
            // Iterate backward, as most recent setting that covers the case is used
            for (int i = settings.size() - 1; i >= 0; --i) {
                Setting ing = settings.get(i);
                if (ing.selector.covers(selector)) {
                    format = ing.format;
                    break;
                }
            }
            if (format == null || format.isEmpty()) {
                return "";
            }
            Matcher m = FIELD_PATTERN.matcher(format);
            StringBuilder sb = new StringBuilder(format.length());
            while (m.find()) {
                String fieldName = m.group(1);
                String sub = format(fieldName, selector);
                m.appendReplacement(sb, Matcher.quoteReplacement(sub));
            }
            m.appendTail(sb);
            return sb.toString();
        }

        String truncateVarValue(String value) {
            return truncateValue(value, VAR_VALUE_ADD_SELECTOR);
        }

        String truncateValue(String value, Selector selector) {
            if (value==null) {
                return "";
            } else {
                // Retrieve the truncation length
                String truncField = format(TRUNCATION_FIELD, selector);
                if (truncField.isEmpty()) {
                    // No truncation set, use whole value
                    return value;
                } else {
                    // Convert truncation length to int
                    // this is safe since it has been tested before it is set
                    int trunc = Integer.parseUnsignedInt(truncField);
                    int len = value.length();
                    if (len > trunc) {
                        if (trunc <= 13) {
                            // Very short truncations have no room for "..."
                            return value.substring(0, trunc);
                        } else {
                            // Normal truncation, make total length equal truncation length
                            int endLen = trunc / 3;
                            int startLen = trunc - 5 - endLen;
                            return value.substring(0, startLen) + " ... " + value.substring(len -endLen);
                        }
                    } else {
                        // Within truncation length, use whole value
                        return value;
                    }
                }
            }
        }

        // Compute the display output given full context and values
        String format(FormatCase fc, FormatAction fa, FormatWhen fw,
                      FormatResolve fr, FormatUnresolved fu, FormatErrors fe,
                      String name, String type, String value, String unresolved, List<String> errorLines) {
            return format("display", fc, fa, fw, fr, fu, fe,
                name, type, value, unresolved, errorLines);
        }

        // Compute the display output given full context and values
        String format(String field, FormatCase fc, FormatAction fa, FormatWhen fw,
                      FormatResolve fr, FormatUnresolved fu, FormatErrors fe,
                      String name, String type, String value, String unresolved, List<String> errorLines) {
            // Convert the context into a bit representation used as selectors for store field formats
            Selector selector  = new Selector(fc, fa, fw, fr, fu, fe);
            String fname = name==null? "" : name;
            String ftype = type==null? "" : type;
            // Compute the representation of value
            String fvalue = truncateValue(value, selector);
            String funresolved = unresolved==null? "" : unresolved;
            String errors = errorLines.stream()
                    .map(el -> String.format(
                            format("errorline", selector),
                            fname, ftype, fvalue, funresolved, "*cannot-use-errors-here*", el))
                    .collect(joining());
            return String.format(
                    format(field, selector),
                    fname, ftype, fvalue, funresolved, errors, "*cannot-use-err-here*");
        }

        void setPrompts(String prompt, String continuationPrompt) {
            this.prompt = prompt;
            this.continuationPrompt = continuationPrompt;
        }

        String getPrompt(String nextId) {
            return String.format(prompt, nextId);
        }

        String getContinuationPrompt(String nextId) {
            return String.format(continuationPrompt, nextId);
        }
    }

    // Class used to set custom eval output formats
    // For both /set format  -- Parse arguments, setting custom format, or printing error
    private class Setter {

        private final ArgTokenizer at;
        private final MessageHandler messageHandler;
        boolean valid = true;

        Setter(MessageHandler messageHandler, ArgTokenizer at) {
            this.messageHandler = messageHandler;
            this.at = at;
            at.allowedOptions("-retain");
        }

        void fluff(String format, Object... args) {
            messageHandler.fluff(format, args);
        }

        void hard(String format, Object... args) {
            messageHandler.hard(format, args);
        }

        void fluffmsg(String messageKey, Object... args) {
            messageHandler.fluffmsg(messageKey, args);
        }

        void hardmsg(String messageKey, Object... args) {
            messageHandler.hardmsg(messageKey, args);
        }

        boolean showFluff() {
            return messageHandler.showFluff();
        }

        void errorat(String messageKey, Object... args) {
            if (!valid) {
                // no spew of errors
                return;
            }
            valid = false;
            Object[] a2 = Arrays.copyOf(args, args.length + 2);
            a2[args.length] = at.whole();
            messageHandler.errormsg(messageKey, a2);
        }

        // Show format settings -- in a predictable order, for testing...
        void showFormatSettings(Mode sm, String f) {
            if (sm == null) {
                modeMap.entrySet().stream()
                        .sorted((es1, es2) -> es1.getKey().compareTo(es2.getKey()))
                        .forEach(m -> showFormatSettings(m.getValue(), f));
            } else {
                sm.byField.entrySet().stream()
                        .filter(ec -> (f == null)
                            ? !ec.getKey().equals(TRUNCATION_FIELD)
                            : ec.getKey().equals(f))
                        .sorted((ec1, ec2) -> ec1.getKey().compareTo(ec2.getKey()))
                        .forEach(ec -> {
                            ec.getValue().forEach(s -> {
                                hard("/set format %s %s %s %s",
                                        sm.name, ec.getKey(), toStringLiteral(s.format),
                                        s.selector.toString());

                            });
                        });
            }
        }

        void showTruncationSettings(Mode sm) {
            if (sm == null) {
                modeMap.values().forEach(this::showTruncationSettings);
            } else {
                List<Mode.Setting> trunc = sm.byField.get(TRUNCATION_FIELD);
                if (trunc != null) {
                    trunc.forEach(s -> {
                        hard("/set truncation %s %s %s",
                                sm.name, s.format,
                                s.selector.toString());
                    });
                }
            }
        }

        void showPromptSettings(Mode sm) {
            if (sm == null) {
                modeMap.values().forEach(this::showPromptSettings);
            } else {
                hard("/set prompt %s %s %s",
                        sm.name,
                        toStringLiteral(sm.prompt),
                        toStringLiteral(sm.continuationPrompt));
            }
        }

        void showModeSettings(String umode, String msg) {
            if (umode == null) {
                modeMap.values().forEach(this::showModeSettings);
            } else {
                Mode m;
                String retained = retainedMap.get(umode);
                if (retained == null) {
                    m = searchForMode(umode, msg);
                    if (m == null) {
                        return;
                    }
                    umode = m.name;
                    retained = retainedMap.get(umode);
                } else {
                    m = modeMap.get(umode);
                }
                if (retained != null) {
                    Mode rm = buildMode(encodedModeIterator(retained));
                    showModeSettings(rm);
                    hard("/set mode -retain %s", umode);
                    if (m != null && !m.equals(rm)) {
                        hard("");
                        showModeSettings(m);
                    }
                } else {
                    showModeSettings(m);
                }
            }
        }

        void showModeSettings(Mode sm) {
            hard("/set mode %s %s",
                    sm.name, sm.commandFluff ? "-command" : "-quiet");
            showPromptSettings(sm);
            showFormatSettings(sm, null);
            showTruncationSettings(sm);
        }

        void showFeedbackSetting() {
            if (retainedCurrentMode != null) {
                hard("/set feedback -retain %s", retainedCurrentMode.name);
            }
            if (mode != retainedCurrentMode) {
                hard("/set feedback %s", mode.name);
            }
        }

        // For /set prompt <mode> "<prompt>" "<continuation-prompt>"
        boolean