/* 
 *   Copyright (c) 2001, 2002 Michael J. Roberts.  All Rights Reserved.
 *   
 *   Please see the accompanying license file, LICENSE.TXT, for information
 *   on using and copying this software.  
 */
/*
Name
  vmparam.h - T3 Virtual Machine Initialization Parameters
Function
  Defines a set of parameters that we use to initialize the VM during
  program startup.  These parameters are #define'd here if not already
  #define'd elsewhere - this allows the system makefile to override
  the parameters as desired for larger or smaller local limits.  For
  example, with a conventional Unix makefile, one could override the
  stack size parameter like so:

    make -DVM_STACK_SIZE=4096 -funix/makefile
    
Notes
  
Modified
  10/07/01 MJRoberts  - Creation
*/

#ifndef VMPARAM_H
#define VMPARAM_H

/*
 *   Error stack size.  This is the number of bytes allocated per thread for
 *   our custom exception mechanism (see vmerr.h).  err_throw() uses this
 *   space to store our custom exception structures and their parameters 
 *   
 *   This limit doesn't affect exception handling for VM byte code, which
 *   uses the VM stack and VM objects.  This parameter only affects our
 *   internal exception handling mechanism at the C++ level (err_try,
 *   err_catch, etc).  It's unlikely that this will need to be modified for
 *   porting.
 */
#ifndef VM_ERR_STACK_SIZE
# define VM_ERR_STACK_BYTES  1024
#endif


/*
 *   Name and os_exe_seek resource type of VM error message file.  This file
 *   is optional; if present, it overrides the error message strings linked
 *   into the VM directly, to allow translated versions of the messages.
 *   These messages describe run-time exceptions.  
 */
#ifndef VM_ERR_MSG_FNAME
# define VM_ERR_MSG_FNAME "T3_VM.msg"
#endif
#ifndef VM_ERR_MSG_RESTYPE
# define VM_ERR_MSG_RESTYPE "XMSG"
#endif


/*
 *   VM stack size, in elements.  This is the number of slots allocated for
 *   the VM stack, which is used to store activation frames and intermediate
 *   calculation results during byte-code execution.  On a typical 32-bit
 *   machine, each slot takes 8 bytes.  
 */
#ifndef VM_STACK_SIZE
# define VM_STACK_SIZE  4096
#endif

/*
 *   VM stack reserve for stack overflow exception handling.  This is the
 *   number of slots we allocate to process a stack overflow exception.  This
 *   gives us a little space to work with in creating the exception object,
 *   so that the program can recover from the exception.  
 */
#ifndef VM_STACK_RESERVE
# define VM_STACK_RESERVE  64
#endif

/* 
 *   Debugger VM stack reserve size, in elements.  This is the number of
 *   slots we allocate for emergency use by the debugger in recovering from a
 *   stack overflow exception.  
 *   
 *   In debug builds only, this is used in lieu of the normal
 *   VM_STACK_RESERVE.  This is usually a bit larger than the normal reserve,
 *   to create enough headroom that the user can evaluate expressions and
 *   otherwise carry on with interactive debugging activities while stopped
 *   at the point of the overflow.  
 */
#ifndef VM_STACK_DBG_RESERVE
# define VM_STACK_DBG_RESERVE  256
#endif


/*
 *   Undo record limit and savepoint limit.  This controls the space
 *   allocated for the undo mechanism, which allows a snapshot of the
 *   machine state (called a "snapshot") to be taken such that the exact
 *   machine state as it was at the snapshot can be restored at a later
 *   time.  Undo information is kept as deltas from the savepoint, so the
 *   amount of space needed is proportional to the number of changes made
 *   since the snapshot, and is not related to the total program size.
 *   
 *   The record limit specifies the maximum number of individual changes can
 *   be stored in the undo log.  Each time a property, array element, or
 *   other individual scalar field of some object is changed, an undo record
 *   is consumed.  Empirically, the standard library seems to use around 200
 *   records per player turn, but this can vary substantially depending on
 *   what actions are performed, and a game with many daemons carrying on
 *   background operations could conceivably use a lot more space per turn.
 *   
 *   The savepoint limit specifies the maximum number of savepoints that can
 *   be created simultaneously.  There is a hard upper limit of 255
 *   savepoints.
 *   
 *   There is no visible error condition generated by exhausting the undo
 *   space - the undo mechanism simply discards the oldest savepoint or
 *   savepoints as needed to make room for new records.
 *   
 *   The default record limit is meant to allow about ten turns of undo to
 *   be saved.  We choose a much higher savepoint limit because savepoints
 *   consume almost no space in the undo log, so there is no cost of making
 *   this limit high enough that the undo record limit is almost certain to
 *   dominate - in other words, since savepoints cost nothing, there is no
 *   point in artificially limiting them.
 *   
 *   An undo record takes up about 16 bytes (on a machine with 32-bit
 *   pointers and 32-bit alignment; this will obviously vary for hardware
 *   with different sizes or alignment requirements).
 */
#ifndef VM_UNDO_MAX_RECORDS
# define VM_UNDO_MAX_RECORDS  65536
#endif
#ifndef VM_UNDO_MAX_SAVEPTS
# define VM_UNDO_MAX_SAVEPTS  255
#endif

#endif /* VMPARAM_H */
