# -*- coding: utf-8 -*-
# This file is public domain according to its author, Brian Hsu

from docutils.parsers.rst import Directive, directives
from docutils import nodes

try:
    import requests
except ImportError:
    requests = None  # NOQA

from nikola.plugin_categories import RestExtension
from nikola.utils import req_missing


class Plugin(RestExtension):

    name = "rest_gist"

    def set_site(self, site):
        self.site = site
        directives.register_directive('gist', GitHubGist)
        return super(Plugin, self).set_site(site)


class GitHubGist(Directive):
    """ Embed GitHub Gist.

        Usage:

          .. gist:: GIST_ID

        or

          .. gist:: GIST_URL


    """

    required_arguments = 1
    optional_arguments = 1
    option_spec = {'file': directives.unchanged}
    final_argument_whitespace = True
    has_content = False

    def get_raw_gist_with_filename(self, gistID, filename):
        url = '/'.join(("https://gist.github.com/raw", gistID, filename))
        return requests.get(url).text

    def get_raw_gist(self, gistID):
        url = "https://gist.github.com/raw/{0}".format(gistID)
        return requests.get(url).text

    def run(self):
        if 'https://' in self.arguments[0]:
            gistID = self.arguments[0].split('/')[-1].strip()
        else:
            gistID = self.arguments[0].strip()
        embedHTML = ""
        rawGist = ""

        if 'file' in self.options:
            filename = self.options['file']
            if requests is not None:
                rawGist = (self.get_raw_gist_with_filename(gistID, filename))
            embedHTML = ('<script src="https://gist.github.com/{0}.js'
                         '?file={1}"></script>').format(gistID, filename)
        else:
            if requests is not None:
                rawGist = (self.get_raw_gist(gistID))
            embedHTML = ('<script src="https://gist.github.com/{0}.js">'
                         '</script>').format(gistID)

        if requests is None:
            reqnode = nodes.raw(
                '', req_missing('requests', 'have inline gist source',
                                optional=True), format='html')
        else:
            reqnode = nodes.literal_block('', rawGist)

        return [nodes.raw('', embedHTML, format='html'),
                nodes.raw('', '<noscript>', format='html'),
                reqnode,
                nodes.raw('', '</noscript>', format='html')]
