<!--

@license Apache-2.0

Copyright (c) 2018 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# MINSTD Shuffle

> A linear congruential pseudorandom number generator ([LCG][lcg]) whose output is shuffled.

<section class="usage">

## Usage

```javascript
var minstd = require( '@stdlib/random/base/minstd-shuffle' );
```

#### minstd()

Returns a pseudorandom integer on the interval `[1, 2147483646]`.

```javascript
var v = minstd();
// returns <number>
```

#### minstd.normalized()

Returns a pseudorandom number on the interval `[0,1)`.

```javascript
var v = minstd.normalized();
// returns <number>
```

#### minstd.factory( \[options] )

Returns a linear congruential pseudorandom number generator ([LCG][lcg]) whose output is shuffled.

```javascript
var rand = minstd.factory();
```

The function accepts the following `options`:

-   **seed**: pseudorandom number generator seed.
-   **state**: an [`Int32Array`][@stdlib/array/int32] containing pseudorandom number generator state. If provided, the function ignores the `seed` option.
-   **copy**: `boolean` indicating whether to copy a provided pseudorandom number generator state. Setting this option to `false` allows sharing state between two or more pseudorandom number generators. Setting this option to `true` ensures that a returned generator has exclusive control over its internal state. Default: `true`.

By default, a random integer is used to seed the returned generator. To seed the generator, provide either an `integer` on the interval `[1, 2147483646]`

```javascript
var rand = minstd.factory({
    'seed': 1234
});

var v = rand();
// returns 1421600654
```

or, for arbitrary length seeds, an array-like `object` containing signed 32-bit integers

```javascript
var Int32Array = require( '@stdlib/array/int32' );

var rand = minstd.factory({
    'seed': new Int32Array( [ 1234 ] )
});

var r = rand();
// returns 20739838
```

To return a generator having a specific initial state, set the generator `state` option.

```javascript
var rand;
var bool;
var r;
var i;

// Generate pseudorandom numbers, thus progressing the generator state:
for ( i = 0; i < 1000; i++ ) {
    r = minstd();
}

// Create a new PRNG initialized to the current state of `minstd`:
rand = minstd.factory({
    'state': minstd.state
});

// Test that the generated pseudorandom numbers are the same:
bool = ( rand() === minstd() );
// returns true
```

#### minstd.NAME

The generator name.

```javascript
var str = minstd.NAME;
// returns 'minstd-shuffle'
```

#### minstd.MIN

Minimum possible value.

```javascript
var min = minstd.MIN;
// returns 1
```

#### minstd.MAX

Maximum possible value.

```javascript
var max = minstd.MAX;
// returns 2147483646
```

#### minstd.seed

The value used to seed `minstd()`.

```javascript
var rand;
var v;
var i;

// Generate pseudorandom values...
for ( i = 0; i < 100; i++ ) {
    v = minstd();
}

// Generate the same pseudorandom values...
rand = minstd.factory({
    'seed': minstd.seed
});
for ( i = 0; i < 100; i++ ) {
    v = rand();
}
```

#### minstd.seedLength

Length of generator seed.

```javascript
var len = minstd.seedLength;
// returns <number>
```

#### minstd.state

Writable property for getting and setting the generator state.

```javascript
var r = minstd();
// returns <number>

r = minstd();
// returns <number>

// ...

// Get a copy of the current state:
var state = minstd.state;
// returns <Int32Array>

r = minstd();
// returns <number>

r = minstd();
// returns <number>

// Reset the state:
minstd.state = state;

// Replay the last two pseudorandom numbers:
r = minstd();
// returns <number>

r = minstd();
// returns <number>

// ...
```

#### minstd.stateLength

Length of generator state.

```javascript
var len = minstd.stateLength;
// returns <number>
```

#### minstd.byteLength

Size (in bytes) of generator state.

```javascript
var sz = minstd.byteLength;
// returns <number>
```

#### minstd.toJSON()

Serializes the pseudorandom number generator as a JSON object.

```javascript
var o = minstd.toJSON();
// returns { 'type': 'PRNG', 'name': '...', 'state': {...}, 'params': [] }
```

</section>

<!-- /.usage -->

<section class="notes">

## Notes

-   Before output from a simple linear congruential generator ([LCG][lcg]) is returned, the output is shuffled using the Bays-Durham algorithm. This additional step considerably strengthens the "randomness quality" of a simple [LCG][lcg]'s output.
-   The generator has a period of approximately `2.1e9` (see [Numerical Recipes in C, 2nd Edition](#references), p. 279).
-   An [LCG][lcg] is fast and uses little memory. On the other hand, because the generator is a simple [linear congruential generator][lcg], the generator has recognized shortcomings. By today's PRNG standards, the generator's period is relatively short. In general, this generator is unsuitable for Monte Carlo simulations and cryptographic applications.
-   If PRNG state is "shared" (meaning a state array was provided during PRNG creation and **not** copied) and one sets the generator state to a state array having a different length, the PRNG does **not** update the existing shared state and, instead, points to the newly provided state array. In order to synchronize PRNG output according to the new shared state array, the state array for **each** relevant PRNG must be **explicitly** set.
-   If PRNG state is "shared" and one sets the generator state to a state array of the same length, the PRNG state is updated (along with the state of all other PRNGs sharing the PRNG's state array).

</section>

<!-- /.notes -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var minstd = require( '@stdlib/random/base/minstd-shuffle' );

var seed;
var rand;
var i;

// Generate pseudorandom numbers...
for ( i = 0; i < 100; i++ ) {
    console.log( minstd() );
}

// Create a new pseudorandom number generator...
seed = 1234;
rand = minstd.factory({
    'seed': seed
});
for ( i = 0; i < 100; i++ ) {
    console.log( rand() );
}

// Create another pseudorandom number generator using a previous seed...
rand = minstd.factory({
    'seed': minstd.seed
});
for ( i = 0; i < 100; i++ ) {
    console.log( rand() );
}
```

</section>

<!-- /.examples -->

* * *

<section class="references">

## References

-   Park, S. K., and K. W. Miller. 1988. "Random Number Generators: Good Ones Are Hard to Find." _Communications of the ACM_ 31 (10). New York, NY, USA: ACM: 1192–1201. doi:[10.1145/63039.63042][@park:1988].
-   Bays, Carter, and S. D. Durham. 1976. "Improving a Poor Random Number Generator." _ACM Transactions on Mathematical Software_ 2 (1). New York, NY, USA: ACM: 59–64. doi:[10.1145/355666.355670][@bays:1976].
-   Herzog, T.N., and G. Lord. 2002. _Applications of Monte Carlo Methods to Finance and Insurance_. ACTEX Publications. [https://books.google.com/books?id=vC7I\\\_gdX-A0C][@herzog:2002].
-   Press, William H., Brian P. Flannery, Saul A. Teukolsky, and William T. Vetterling. 1992. _Numerical Recipes in C: The Art of Scientific Computing, Second Edition_. Cambridge University Press.

</section>

<!-- /.references -->

<section class="links">

[lcg]: https://en.wikipedia.org/wiki/Linear_congruential_generator

[@park:1988]: http://dx.doi.org/10.1145/63039.63042

[@bays:1976]: http://dx.doi.org/10.1145/355666.355670

[@herzog:2002]: https://books.google.com/books?id=vC7I_gdX-A0C

[@stdlib/array/int32]: https://github.com/stdlib-js/array-int32

</section>

<!-- /.links -->
