#!/usr/bin/env python3

import json
import os
import xml.dom.minidom 

def parse_json_file(file_path):
    """Parses a JSON file and returns the data as Python data structures."""
    try:
        with open(file_path, 'r') as file:
            data = json.load(file)
            return data
    except FileNotFoundError:
        print(f"Error: File not found at {file_path}")
        return None
    except json.JSONDecodeError:
        print(f"Error: Invalid JSON format in {file_path}")
        return None

def create_xml_file_minidom(filename, list4217):
    """Creates an XML file using xml.dom.minidom."""

    doc = xml.dom.minidom.Document()
    
    root = doc.createElement("xsd:schema")
    root.setAttribute("xmlns:xsd", "http://www.w3.org/2001/XMLSchema")
    root.setAttribute("xmlns:clm5ISO42173A", "urn:un:unece:uncefact:codelist:standard:5:ISO42173A:2010-04-07")
    root.setAttribute("xmlns:ccts", "urn:un:unece:uncefact:documentation:standard:CoreComponentsTechnicalSpecification:2")
    root.setAttribute("targetNamespace", "urn:un:unece:uncefact:codelist:standard:5:ISO42173A:2010-04-07")
    root.setAttribute("elementFormDefault", "qualified")
    root.setAttribute("attributeFormDefault", "unqualified")
    root.setAttribute("version", "7.0")
    doc.appendChild(root)

    element1 = doc.createElement("xsd:element")
    element1.setAttribute("name", "ISO3AlphaCurrencyCode")
    element1.setAttribute("type", "clm5ISO42173A:ISO3AlphaCurrencyCodeContentType")
    root.appendChild(element1)

    content1 = doc.createElement("xsd:simpleType")
    content1.setAttribute("name", "ISO3AlphaCurrencyCodeContentType")
    root.appendChild(content1)

    content2 = doc.createElement("xsd:restriction")
    content2.setAttribute("base", "xsd:token")
    content1.appendChild(content2)
 
    for currency in list4217:
        current1 = doc.createElement("xsd:enumeration")
        current1.setAttribute("value", currency.get("alpha_3"))
        current2 = doc.createElement("xsd:annotation")
        current1.appendChild(current2)
        current3 = doc.createElement("xsd:documentation")
        current2.appendChild(current3)
        current4 = doc.createElement("ccts:Name")
        current3.appendChild(current4)
        current4.appendChild(doc.createTextNode(currency.get("name")))
        content2.appendChild(current1)
    
    xml_str = doc.toprettyxml(indent="        ")

    with open(filename, "w", encoding="utf-8") as file:
        file.write(xml_str)

    print(f"XML file created successfully: {filename}")

    
def main():
    file_path_4217 = "/usr/share/iso-codes/json/iso_4217.json"
    file_path_xml = "oadr_ISO_ISO3AlphaCurrencyCode_20100407.xsd"

    data4217 = parse_json_file(file_path_4217)

    list4217 = data4217.get('4217')

    if list4217:
        for currency in list4217:
            print("ID: %s, Currency: %s, Numerical: %s"%(currency.get("alpha_3"), currency.get("name"), currency.get("numeric")))
        create_xml_file_minidom(file_path_xml, list4217)
    
            
if __name__ == "__main__":
    main()
