/*
 * oFono - Open Source Telephony
 * Copyright (C) 2008-2011  Intel Corporation
 * Copyright (C) 2009-2010  Nokia Corporation and/or its subsidiary(-ies)
 *
 * SPDX-License-Identifier: GPL-2.0-only
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <ell/ell.h>

#include "util.h"

/*
	Name:			GSM 03.38 to Unicode
	Unicode version:	3.0
	Table version:		1.1
	Table format:		Format A
	Date:			2000 May 30
	Authors:		Ken Whistler
				Kent Karlsson
				Markus Kuhn

	Copyright (c) 2000 Unicode, Inc.  All Rights reserved.

	This file is provided as-is by Unicode, Inc. (The Unicode Consortium).
	No claims are made as to fitness for any particular purpose.  No
	warranties of any kind are expressed or implied.  The recipient
	agrees to determine applicability of information provided.  If this
	file has been provided on optical media by Unicode, Inc., the sole
	remedy for any claim will be exchange of defective media within 90
	days of receipt.

	Unicode, Inc. hereby grants the right to freely use the information
	supplied in this file in the creation of products supporting the
	Unicode Standard, and to make copies of this file in any form for
	internal or external distribution as long as this notice remains
	attached.
*/

#define GUND			0xFFFF

#define UTF8_LENGTH(c) \
	((c) < 0x80 ? 1 : ((c) < 0x800 ? 2 : 3))

struct codepoint {
	unsigned short from;
	unsigned short to;
};

struct conversion_table {
	/* To unicode locking shift table */
	const struct codepoint *locking_u;
	unsigned int locking_len_u;

	/* To unicode single shift table */
	const struct codepoint *single_u;
	unsigned int single_len_u;

	/* To GSM locking shift table, fixed size */
	const unsigned short *locking_g;

	/* To GSM single shift table */
	const struct codepoint *single_g;
	unsigned int single_len_g;
};

/* GSM to Unicode extension table, for GSM sequences starting with 0x1B */
static const struct codepoint def_ext_gsm[] = {
	{ 0x0A, 0x000C },		/* See NOTE 3 in 23.038 */
	{ 0x14, 0x005E },
	{ 0x1B, 0x0020 },		/* See NOTE 1 in 23.038 */
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x65, 0x20AC }
};

static const struct codepoint def_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.1. in 3GPP TS23.038, V.8.2.0 */
static const struct codepoint tur_ext_gsm[] = {
	{ 0x0A, 0x000C },		/* See NOTE 3 */
	{ 0x14, 0x005E },
	{ 0x1B, 0x0020 },		/* See NOTE 1 */
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x47, 0x011E },
	{ 0x49, 0x0130 },
	{ 0x53, 0x015E },
	{ 0x63, 0x00E7 },
	{ 0x65, 0x20AC },
	{ 0x67, 0x011F },
	{ 0x69, 0x0131 },
	{ 0x73, 0x015F }
};

static const struct codepoint tur_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00E7, 0x1B63 },
	{ 0x011E, 0x1B47 },
	{ 0x011F, 0x1B67 },
	{ 0x0130, 0x1B49 },
	{ 0x0131, 0x1B69 },
	{ 0x015E, 0x1B53 },
	{ 0x015F, 0x1B73 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.2. in 3GPP TS23.038 V.8.2.0*/
static const struct codepoint spa_ext_gsm[] = {
	{ 0x09, 0x00E7 },
	{ 0x0A, 0x000C },		/* See NOTE 3 */
	{ 0x14, 0x005E },
	{ 0x1B, 0x0020 },		/* See NOTE 1 */
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x00C1 },
	{ 0x49, 0x00CD },
	{ 0x4F, 0x00D3 },
	{ 0x55, 0x00DA },
	{ 0x61, 0x00E1 },
	{ 0x65, 0x20AC },
	{ 0x69, 0x00ED },
	{ 0x6F, 0x00F3 },
	{ 0x75, 0x00FA }
};

static const struct codepoint spa_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00C1, 0x1B41 },
	{ 0x00CD, 0x1B49 },
	{ 0x00D3, 0x1B4F },
	{ 0x00DA, 0x1B55 },
	{ 0x00E1, 0x1B61 },
	{ 0x00E7, 0x1B09 },
	{ 0x00ED, 0x1B69 },
	{ 0x00F3, 0x1B6F },
	{ 0x00FA, 0x1B75 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.3. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint por_ext_gsm[] = {
	{ 0x05, 0x00EA },
	{ 0x09, 0x00E7 },
	{ 0x0A, 0x000C },		/* See NOTE 3 */
	{ 0x0B, 0x00D4 },
	{ 0x0C, 0x00F4 },
	{ 0x0E, 0x00C1 },
	{ 0x0F, 0x00E1 },
	{ 0x12, 0x03A6 },
	{ 0x13, 0x0393 },
	{ 0x14, 0x005E },
	{ 0x15, 0x03A9 },
	{ 0x16, 0x03A0 },
	{ 0x17, 0x03A8 },
	{ 0x18, 0x03A3 },
	{ 0x19, 0x0398 },
	{ 0x1B, 0x0020 },		/* See NOTE 1 */
	{ 0x1F, 0x00CA },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x00C0 },
	{ 0x49, 0x00CD },
	{ 0x4F, 0x00D3 },
	{ 0x55, 0x00DA },
	{ 0x5B, 0x00C3 },
	{ 0x5C, 0x00D5 },
	{ 0x61, 0x00C2 },
	{ 0x65, 0x20AC },
	{ 0x69, 0x00ED },
	{ 0x6F, 0x00F3 },
	{ 0x75, 0x00FA },
	{ 0x7B, 0x00E3 },
	{ 0x7C, 0x00F5 },
	{ 0x7F, 0x00E2 }
};

static const struct codepoint por_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00C0, 0x1B41 },
	{ 0x00C1, 0x1B0E },
	{ 0x00C2, 0x1B61 },
	{ 0x00C3, 0x1B5B },
	{ 0x00CA, 0x1B1F },
	{ 0x00CD, 0x1B49 },
	{ 0x00D3, 0x1B4F },
	{ 0x00D4, 0x1B0B },
	{ 0x00D5, 0x1B5C },
	{ 0x00DA, 0x1B55 },
	{ 0x00E1, 0x1B0F },
	{ 0x00E2, 0x1B7F },
	{ 0x00E3, 0x1B7B },
	{ 0x00E7, 0x1B09 },
	{ 0x00EA, 0x1B05 },
	{ 0x00ED, 0x1B69 },
	{ 0x00F3, 0x1B6F },
	{ 0x00F4, 0x1B0C },
	{ 0x00F5, 0x1B7C },
	{ 0x00FA, 0x1B75 },
	{ 0x0393, 0x1B13 },
	{ 0x0398, 0x1B19 },
	{ 0x03A0, 0x1B16 },
	{ 0x03A3, 0x1B18 },
	{ 0x03A6, 0x1B12 },
	{ 0x03A8, 0x1B17 },
	{ 0x03A9, 0x1B15 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.4. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint ben_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },            /* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0X003C },
	{ 0x11, 0X003D },
	{ 0x12, 0X003E },
	{ 0x13, 0X00A1 },
	{ 0x14, 0X005E },
	{ 0x15, 0X00A1 },
	{ 0x16, 0X005F },
	{ 0x17, 0X0023 },
	{ 0x18, 0X002A },
	{ 0x19, 0X09E6 },
	{ 0x1A, 0X09E7 },
	{ 0x1B, 0X0020 },            /* See NOTE 1 */
	{ 0x1C, 0X09E8 },
	{ 0x1D, 0X09E9 },
	{ 0x1E, 0X09EA },
	{ 0x1F, 0X09EB },
	{ 0x20, 0x09EC },
	{ 0x21, 0x09ED },
	{ 0x22, 0x09EE },
	{ 0x23, 0x09EF },
	{ 0x24, 0x09DF },
	{ 0x25, 0x09E0 },
	{ 0x26, 0x09E1 },
	{ 0x27, 0x09E2 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x09E3 },
	{ 0x2B, 0x09F2 },
	{ 0x2C, 0x09F3 },
	{ 0x2D, 0x09F4 },
	{ 0x2E, 0x09F5 },
	{ 0x2F, 0x005C },
	{ 0x30, 0x09F6 },
	{ 0x31, 0x09F7 },
	{ 0x32, 0x09F8 },
	{ 0x33, 0x09F9 },
	{ 0x34, 0x09FA },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint ben_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0X0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0X002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0X003C, 0x1B10 },
	{ 0X003D, 0x1B11 },
	{ 0X003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0X005E, 0x1B14 },
	{ 0X005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0X00A1, 0x1B13 },
	{ 0X00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x09DF, 0x1B24 },
	{ 0x09E0, 0x1B25 },
	{ 0x09E1, 0x1B26 },
	{ 0x09E2, 0x1B27 },
	{ 0x09E3, 0x1B2A },
	{ 0X09E6, 0x1B19 },
	{ 0X09E7, 0x1B1A },
	{ 0X09E8, 0x1B1C },
	{ 0X09E9, 0x1B1D },
	{ 0X09EA, 0x1B1E },
	{ 0X09EB, 0x1B1F },
	{ 0x09EC, 0x1B20 },
	{ 0x09ED, 0x1B21 },
	{ 0x09EE, 0x1B22 },
	{ 0x09EF, 0x1B23 },
	{ 0x09F2, 0x1B2B },
	{ 0x09F3, 0x1B2C },
	{ 0x09F4, 0x1B2D },
	{ 0x09F5, 0x1B2E },
	{ 0x09F6, 0x1B30 },
	{ 0x09F7, 0x1B31 },
	{ 0x09F8, 0x1B32 },
	{ 0x09F9, 0x1B33 },
	{ 0x09FA, 0x1B34 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.5. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint guj_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0AE6 },
	{ 0x1D, 0x0AE7 },
	{ 0x1E, 0x0AE8 },
	{ 0x1F, 0x0AE9 },
	{ 0x20, 0x0AEA },
	{ 0x21, 0x0AEB },
	{ 0x22, 0x0AEC },
	{ 0x23, 0x0AED },
	{ 0x24, 0x0AEE },
	{ 0x25, 0x0AEF },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint guj_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0AE6, 0x1B1C },
	{ 0x0AE7, 0x1B1D },
	{ 0x0AE8, 0x1B1E },
	{ 0x0AE9, 0x1B1F },
	{ 0x0AEA, 0x1B20 },
	{ 0x0AEB, 0x1B21 },
	{ 0x0AEC, 0x1B22 },
	{ 0x0AED, 0x1B23 },
	{ 0x0AEE, 0x1B24 },
	{ 0x0AEF, 0x1B25 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.6. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint hin_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0966 },
	{ 0x1D, 0x0967 },
	{ 0x1E, 0x0968 },
	{ 0x1F, 0x0969 },
	{ 0x20, 0x096A },
	{ 0x21, 0x096B },
	{ 0x22, 0x096C },
	{ 0x23, 0x096D },
	{ 0x24, 0x096E },
	{ 0x25, 0x096F },
	{ 0x26, 0x0951 },
	{ 0x27, 0x0952 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0953 },
	{ 0x2B, 0x0954 },
	{ 0x2C, 0x0958 },
	{ 0x2D, 0x0959 },
	{ 0x2E, 0x095A },
	{ 0x2F, 0x005C },
	{ 0x30, 0x095B },
	{ 0x31, 0x095C },
	{ 0x32, 0x095D },
	{ 0x33, 0x095E },
	{ 0x34, 0x095F },
	{ 0x35, 0x0960 },
	{ 0x36, 0x0961 },
	{ 0x37, 0x0962 },
	{ 0x38, 0x0963 },
	{ 0x39, 0x0970 },
	{ 0x3A, 0x0971 },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint hin_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0951, 0x1B26 },
	{ 0x0952, 0x1B27 },
	{ 0x0953, 0x1B2A },
	{ 0x0954, 0x1B2B },
	{ 0x0958, 0x1B2C },
	{ 0x0959, 0x1B2D },
	{ 0x095A, 0x1B2E },
	{ 0x095B, 0x1B30 },
	{ 0x095C, 0x1B31 },
	{ 0x095D, 0x1B32 },
	{ 0x095E, 0x1B33 },
	{ 0x095F, 0x1B34 },
	{ 0x0960, 0x1B35 },
	{ 0x0961, 0x1B36 },
	{ 0x0962, 0x1B37 },
	{ 0x0963, 0x1B38 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0966, 0x1B1C },
	{ 0x0967, 0x1B1D },
	{ 0x0968, 0x1B1E },
	{ 0x0969, 0x1B1F },
	{ 0x096A, 0x1B20 },
	{ 0x096B, 0x1B21 },
	{ 0x096C, 0x1B22 },
	{ 0x096D, 0x1B23 },
	{ 0x096E, 0x1B24 },
	{ 0x096F, 0x1B25 },
	{ 0x0970, 0x1B39 },
	{ 0x0971, 0x1B3A },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.7. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint kan_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0CE6 },
	{ 0x1D, 0x0CE7 },
	{ 0x1E, 0x0CE8 },
	{ 0x1F, 0x0CE9 },
	{ 0x20, 0x0CEA },
	{ 0x21, 0x0CEB },
	{ 0x22, 0x0CEC },
	{ 0x23, 0x0CED },
	{ 0x24, 0x0CEE },
	{ 0x25, 0x0CEF },
	{ 0x26, 0x0CDE },
	{ 0x27, 0x0CF1 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0CF2 },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005D },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint kan_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3C },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0CDE, 0x1B26 },
	{ 0x0CE6, 0x1B1C },
	{ 0x0CE7, 0x1B1D },
	{ 0x0CE8, 0x1B1E },
	{ 0x0CE9, 0x1B1F },
	{ 0x0CEA, 0x1B20 },
	{ 0x0CEB, 0x1B21 },
	{ 0x0CEC, 0x1B22 },
	{ 0x0CED, 0x1B23 },
	{ 0x0CEE, 0x1B24 },
	{ 0x0CEF, 0x1B25 },
	{ 0x0CF1, 0x1B27 },
	{ 0x0CF2, 0x1B2A },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.8. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint mal_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0D66 },
	{ 0x1D, 0x0D67 },
	{ 0x1E, 0x0D68 },
	{ 0x1F, 0x0D69 },
	{ 0x20, 0x0D6A },
	{ 0x21, 0x0D6B },
	{ 0x22, 0x0D6C },
	{ 0x23, 0x0D6D },
	{ 0x24, 0x0D6E },
	{ 0x25, 0x0D6F },
	{ 0x26, 0x0D70 },
	{ 0x27, 0x0D71 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0D72 },
	{ 0x2B, 0x0D73 },
	{ 0x2C, 0x0D74 },
	{ 0x2D, 0x0D75 },
	{ 0x2E, 0x0D7A },
	{ 0x2F, 0x005C },
	{ 0x30, 0x0D7B },
	{ 0x31, 0x0D7C },
	{ 0x32, 0x0D7D },
	{ 0x33, 0x0D7E },
	{ 0x34, 0x0D7F },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint mal_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0D66, 0x1B1C },
	{ 0x0D67, 0x1B1D },
	{ 0x0D68, 0x1B1E },
	{ 0x0D69, 0x1B1F },
	{ 0x0D6A, 0x1B20 },
	{ 0x0D6B, 0x1B21 },
	{ 0x0D6C, 0x1B22 },
	{ 0x0D6D, 0x1B23 },
	{ 0x0D6E, 0x1B24 },
	{ 0x0D6F, 0x1B25 },
	{ 0x0D70, 0x1B26 },
	{ 0x0D71, 0x1B27 },
	{ 0x0D72, 0x1B2A },
	{ 0x0D73, 0x1B2B },
	{ 0x0D74, 0x1B2C },
	{ 0x0D75, 0x1B2D },
	{ 0x0D7A, 0x1B2E },
	{ 0x0D7B, 0x1B30 },
	{ 0x0D7C, 0x1B31 },
	{ 0x0D7D, 0x1B32 },
	{ 0x0D7E, 0x1B33 },
	{ 0x0D7F, 0x1B34 },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.9. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint ori_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0B66 },
	{ 0x1D, 0x0B67 },
	{ 0x1E, 0x0B68 },
	{ 0x1F, 0x0B69 },
	{ 0x20, 0x0B6A },
	{ 0x21, 0x0B6B },
	{ 0x22, 0x0B6C },
	{ 0x23, 0x0B6D },
	{ 0x24, 0x0B6E },
	{ 0x25, 0x0B6F },
	{ 0x26, 0x0B5C },
	{ 0x27, 0x0B5D },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0B5F },
	{ 0x2B, 0x0B70 },
	{ 0x2C, 0x0B71 },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint ori_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0B5C, 0x1B26 },
	{ 0x0B5D, 0x1B27 },
	{ 0x0B5F, 0x1B2A },
	{ 0x0B66, 0x1B1C },
	{ 0x0B67, 0x1B1D },
	{ 0x0B68, 0x1B1E },
	{ 0x0B69, 0x1B1F },
	{ 0x0B6A, 0x1B20 },
	{ 0x0B6B, 0x1B21 },
	{ 0x0B6C, 0x1B22 },
	{ 0x0B6D, 0x1B23 },
	{ 0x0B6E, 0x1B24 },
	{ 0x0B6F, 0x1B25 },
	{ 0x0B70, 0x1B2B },
	{ 0x0B71, 0x1B2C },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.10. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint pun_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0A66 },
	{ 0x1D, 0x0A67 },
	{ 0x1E, 0x0A68 },
	{ 0x1F, 0x0A69 },
	{ 0x20, 0x0A6A },
	{ 0x21, 0x0A6B },
	{ 0x22, 0x0A6C },
	{ 0x23, 0x0A6D },
	{ 0x24, 0x0A6E },
	{ 0x25, 0x0A6F },
	{ 0x26, 0x0A59 },
	{ 0x27, 0x0A5A },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0A5B },
	{ 0x2B, 0x0A5C },
	{ 0x2C, 0x0A5E },
	{ 0x2D, 0x0A75 },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint pun_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0A59, 0x1B26 },
	{ 0x0A5A, 0x1B27 },
	{ 0x0A5B, 0x1B2A },
	{ 0x0A5C, 0x1B2B },
	{ 0x0A5E, 0x1B2C },
	{ 0x0A66, 0x1B1C },
	{ 0x0A67, 0x1B1D },
	{ 0x0A68, 0x1B1E },
	{ 0x0A69, 0x1B1F },
	{ 0x0A6A, 0x1B20 },
	{ 0x0A6B, 0x1B21 },
	{ 0x0A6C, 0x1B22 },
	{ 0x0A6D, 0x1B23 },
	{ 0x0A6E, 0x1B24 },
	{ 0x0A6F, 0x1B25 },
	{ 0x0A75, 0x1B2D },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.11. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint tam_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0964 },
	{ 0x1A, 0x0965 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0BE6 },
	{ 0x1D, 0x0BE7 },
	{ 0x1E, 0x0BE8 },
	{ 0x1F, 0x0BE9 },
	{ 0x20, 0x0BEA },
	{ 0x21, 0x0BEB },
	{ 0x22, 0x0BEC },
	{ 0x23, 0x0BED },
	{ 0x24, 0x0BEE },
	{ 0x25, 0x0BEF },
	{ 0x26, 0x0BF3 },
	{ 0x27, 0x0BF4 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0BF5 },
	{ 0x2B, 0x0BF6 },
	{ 0x2C, 0x0BF7 },
	{ 0x2D, 0x0BF8 },
	{ 0x2E, 0x0BFA },
	{ 0x2F, 0x005C },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint tam_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0964, 0x1B19 },
	{ 0x0965, 0x1B1A },
	{ 0x0BE6, 0x1B1C },
	{ 0x0BE7, 0x1B1D },
	{ 0x0BE8, 0x1B1E },
	{ 0x0BE9, 0x1B1F },
	{ 0x0BEA, 0x1B20 },
	{ 0x0BEB, 0x1B21 },
	{ 0x0BEC, 0x1B22 },
	{ 0x0BED, 0x1B23 },
	{ 0x0BEE, 0x1B24 },
	{ 0x0BEF, 0x1B25 },
	{ 0x0BF3, 0x1B26 },
	{ 0x0BF4, 0x1B27 },
	{ 0x0BF5, 0x1B2A },
	{ 0x0BF6, 0x1B2B },
	{ 0x0BF7, 0x1B2C },
	{ 0x0BF8, 0x1B2D },
	{ 0x0BFA, 0x1B2E },
	{ 0x20AC, 0x1B65 }
};

/* Appendix A.2.12. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint tel_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x0C66 },
	{ 0x1D, 0x0C67 },
	{ 0x1E, 0x0C68 },
	{ 0x1F, 0x0C69 },
	{ 0x20, 0x0C6A },
	{ 0x21, 0x0C6B },
	{ 0x22, 0x0C6C },
	{ 0x23, 0x0C6D },
	{ 0x24, 0x0C6E },
	{ 0x25, 0x0C6F },
	{ 0x26, 0x0C58 },
	{ 0x27, 0x0C59 },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x0C78 },
	{ 0x2B, 0x0C79 },
	{ 0x2C, 0x0C7A },
	{ 0x2D, 0x0C7B },
	{ 0x2E, 0x0C7C },
	{ 0x2F, 0x005C },
	{ 0x30, 0x0C7D },
	{ 0x31, 0x0C7E },
	{ 0x32, 0x0C7F },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A }
};

static const struct codepoint tel_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0C58, 0x1B26 },
	{ 0x0C59, 0x1B27 },
	{ 0x0C66, 0x1B1C },
	{ 0x0C67, 0x1B1D },
	{ 0x0C68, 0x1B1E },
	{ 0x0C69, 0x1B1F },
	{ 0x0C6A, 0x1B20 },
	{ 0x0C6B, 0x1B21 },
	{ 0x0C6C, 0x1B22 },
	{ 0x0C6D, 0x1B23 },
	{ 0x0C6E, 0x1B24 },
	{ 0x0C6F, 0x1B25 },
	{ 0x0C78, 0x1B2A },
	{ 0x0C79, 0x1B2B },
	{ 0x0C7A, 0x1B2C },
	{ 0x0C7B, 0x1B2D },
	{ 0x0C7C, 0x1B2E },
	{ 0x0C7D, 0x1B30 },
	{ 0x0C7E, 0x1B31 },
	{ 0x0C7F, 0x1B32 }
};

/* Appendix A.2.13. in 3GPP TS23.038 V.8.2.0 */
static const struct codepoint urd_ext_gsm[] = {
	{ 0x00, 0x0040 },
	{ 0x01, 0x00A3 },
	{ 0x02, 0x0024 },
	{ 0x03, 0x00A5 },
	{ 0x04, 0x00BF },
	{ 0x05, 0x0022 },
	{ 0x06, 0x00A4 },
	{ 0x07, 0x0025 },
	{ 0x08, 0x0026 },
	{ 0x09, 0x0027 },
	{ 0x0A, 0x000C },	/* See NOTE 3 */
	{ 0x0B, 0x002A },
	{ 0x0C, 0x002B },
	{ 0x0E, 0x002D },
	{ 0x0F, 0x002F },
	{ 0x10, 0x003C },
	{ 0x11, 0x003D },
	{ 0x12, 0x003E },
	{ 0x13, 0x00A1 },
	{ 0x14, 0x005E },
	{ 0x15, 0x00A1 },
	{ 0x16, 0x005F },
	{ 0x17, 0x0023 },
	{ 0x18, 0x002A },
	{ 0x19, 0x0600 },
	{ 0x1A, 0x0601 },
	{ 0x1B, 0x0020 },	/* See NOTE 1 */
	{ 0x1C, 0x06F0 },
	{ 0x1D, 0x06F1 },
	{ 0x1E, 0x06F2 },
	{ 0x1F, 0x06F3 },
	{ 0x20, 0x06F4 },
	{ 0x21, 0x06F5 },
	{ 0x22, 0x06F6 },
	{ 0x23, 0x06F7 },
	{ 0x24, 0x06F8 },
	{ 0x25, 0x06F9 },
	{ 0x26, 0x060C },
	{ 0x27, 0x060D },
	{ 0x28, 0x007B },
	{ 0x29, 0x007D },
	{ 0x2A, 0x060E },
	{ 0x2B, 0x060F },
	{ 0x2C, 0x0610 },
	{ 0x2D, 0x0611 },
	{ 0x2E, 0x0612 },
	{ 0x2F, 0x005C },
	{ 0x30, 0x0613 },
	{ 0x31, 0x0614 },
	{ 0x32, 0x061B },
	{ 0x33, 0x061F },
	{ 0x34, 0x0640 },
	{ 0x35, 0x0652 },
	{ 0x36, 0x0658 },
	{ 0x37, 0x066B },
	{ 0x38, 0x066C },
	{ 0x39, 0x0672 },
	{ 0x3A, 0x0673 },
	{ 0x3B, 0x06CD },
	{ 0x3C, 0x005B },
	{ 0x3D, 0x007E },
	{ 0x3E, 0x005D },
	{ 0x3F, 0x06D4 },
	{ 0x40, 0x007C },
	{ 0x41, 0x0041 },
	{ 0x42, 0x0042 },
	{ 0x43, 0x0043 },
	{ 0x44, 0x0044 },
	{ 0x45, 0x0045 },
	{ 0x46, 0x0046 },
	{ 0x47, 0x0047 },
	{ 0x48, 0x0048 },
	{ 0x49, 0x0049 },
	{ 0x4A, 0x004A },
	{ 0x4B, 0x004B },
	{ 0x4C, 0x004C },
	{ 0x4D, 0x004D },
	{ 0x4E, 0x004E },
	{ 0x4F, 0x004F },
	{ 0x50, 0x0050 },
	{ 0x51, 0x0051 },
	{ 0x52, 0x0052 },
	{ 0x53, 0x0053 },
	{ 0x54, 0x0054 },
	{ 0x55, 0x0055 },
	{ 0x56, 0x0056 },
	{ 0x57, 0x0057 },
	{ 0x58, 0x0058 },
	{ 0x59, 0x0059 },
	{ 0x5A, 0x005A },
	{ 0x65, 0x20AC }
};

static const struct codepoint urd_ext_unicode[] = {
	{ 0x000C, 0x1B0A },
	{ 0x0022, 0x1B05 },
	{ 0x0023, 0x1B17 },
	{ 0x0024, 0x1B02 },
	{ 0x0025, 0x1B07 },
	{ 0x0026, 0x1B08 },
	{ 0x0027, 0x1B09 },
	{ 0x002A, 0x1B0B },
	{ 0x002A, 0x1B18 },
	{ 0x002B, 0x1B0C },
	{ 0x002D, 0x1B0E },
	{ 0x002F, 0x1B0F },
	{ 0x003C, 0x1B10 },
	{ 0x003D, 0x1B11 },
	{ 0x003E, 0x1B12 },
	{ 0x0040, 0x1B00 },
	{ 0x0041, 0x1B41 },
	{ 0x0042, 0x1B42 },
	{ 0x0043, 0x1B43 },
	{ 0x0044, 0x1B44 },
	{ 0x0045, 0x1B45 },
	{ 0x0046, 0x1B46 },
	{ 0x0047, 0x1B47 },
	{ 0x0048, 0x1B48 },
	{ 0x0049, 0x1B49 },
	{ 0x004A, 0x1B4A },
	{ 0x004B, 0x1B4B },
	{ 0x004C, 0x1B4C },
	{ 0x004D, 0x1B4D },
	{ 0x004E, 0x1B4E },
	{ 0x004F, 0x1B4F },
	{ 0x0050, 0x1B50 },
	{ 0x0051, 0x1B51 },
	{ 0x0052, 0x1B52 },
	{ 0x0053, 0x1B53 },
	{ 0x0054, 0x1B54 },
	{ 0x0055, 0x1B55 },
	{ 0x0056, 0x1B56 },
	{ 0x0057, 0x1B57 },
	{ 0x0058, 0x1B58 },
	{ 0x0059, 0x1B59 },
	{ 0x005A, 0x1B5A },
	{ 0x005B, 0x1B3C },
	{ 0x005C, 0x1B2F },
	{ 0x005D, 0x1B3E },
	{ 0x005E, 0x1B14 },
	{ 0x005F, 0x1B16 },
	{ 0x007B, 0x1B28 },
	{ 0x007C, 0x1B40 },
	{ 0x007D, 0x1B29 },
	{ 0x007E, 0x1B3D },
	{ 0x00A1, 0x1B13 },
	{ 0x00A1, 0x1B15 },
	{ 0x00A3, 0x1B01 },
	{ 0x00A4, 0x1B06 },
	{ 0x00A5, 0x1B03 },
	{ 0x00BF, 0x1B04 },
	{ 0x0600, 0x1B19 },
	{ 0x0601, 0x1B1A },
	{ 0x060C, 0x1B26 },
	{ 0x060D, 0x1B27 },
	{ 0x060E, 0x1B2A },
	{ 0x060F, 0x1B2B },
	{ 0x0610, 0x1B2C },
	{ 0x0611, 0x1B2D },
	{ 0x0612, 0x1B2E },
	{ 0x0613, 0x1B30 },
	{ 0x0614, 0x1B31 },
	{ 0x061B, 0x1B32 },
	{ 0x061F, 0x1B33 },
	{ 0x0640, 0x1B34 },
	{ 0x0652, 0x1B35 },
	{ 0x0658, 0x1B36 },
	{ 0x066B, 0x1B37 },
	{ 0x066C, 0x1B38 },
	{ 0x0672, 0x1B39 },
	{ 0x0673, 0x1B3A },
	{ 0x06CD, 0x1B3B },
	{ 0x06D4, 0x1B3F },
	{ 0x06F0, 0x1B1C },
	{ 0x06F1, 0x1B1D },
	{ 0x06F2, 0x1B1E },
	{ 0x06F3, 0x1B1F },
	{ 0x06F4, 0x1B20 },
	{ 0x06F5, 0x1B21 },
	{ 0x06F6, 0x1B22 },
	{ 0x06F7, 0x1B23 },
	{ 0x06F8, 0x1B24 },
	{ 0x06F9, 0x1B25 },
	{ 0x20AC, 0x1B65 }
};

/* Used for conversion of GSM to Unicode */
static const unsigned short def_gsm[] = {
	0x0040, 0x00A3, 0x0024, 0x00A5, 0x00E8, 0x00E9, 0x00F9, 0x00EC,
	0x00F2, 0x00C7, 0x000A, 0x00D8, 0x00F8, 0x000D, 0x00C5, 0x00E5,
	0x0394, 0x005F, 0x03A6, 0x0393, 0x039B, 0x03A9, 0x03A0, 0x03A8,
	0x03A3, 0x0398, 0x039E, 0x00A0, 0x00C6, 0x00E6, 0x00DF, 0x00C9,
	0x0020, 0x0021, 0x0022, 0x0023, 0x00A4, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
	0x00A1, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005A, 0x00C4, 0x00D6, 0x00D1, 0x00DC, 0x00A7,
	0x00BF, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x00E4, 0x00F6, 0x00F1, 0x00FC, 0x00E0
};

/* Used for conversion of Unicode to GSM */
static const struct codepoint def_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0022, 0x22 }, { 0x0023, 0x23 }, { 0x0024, 0x02 }, { 0x0025, 0x25 },
	{ 0x0026, 0x26 }, { 0x0027, 0x27 }, { 0x0028, 0x28 }, { 0x0029, 0x29 },
	{ 0x002A, 0x2A }, { 0x002B, 0x2B }, { 0x002C, 0x2C }, { 0x002D, 0x2D },
	{ 0x002E, 0x2E }, { 0x002F, 0x2F }, { 0x0030, 0x30 }, { 0x0031, 0x31 },
	{ 0x0032, 0x32 }, { 0x0033, 0x33 }, { 0x0034, 0x34 }, { 0x0035, 0x35 },
	{ 0x0036, 0x36 }, { 0x0037, 0x37 }, { 0x0038, 0x38 }, { 0x0039, 0x39 },
	{ 0x003A, 0x3A }, { 0x003B, 0x3B }, { 0x003C, 0x3C }, { 0x003D, 0x3D },
	{ 0x003E, 0x3E }, { 0x003F, 0x3F }, { 0x0040, 0x00 }, { 0x0041, 0x41 },
	{ 0x0042, 0x42 }, { 0x0043, 0x43 }, { 0x0044, 0x44 }, { 0x0045, 0x45 },
	{ 0x0046, 0x46 }, { 0x0047, 0x47 }, { 0x0048, 0x48 }, { 0x0049, 0x49 },
	{ 0x004A, 0x4A }, { 0x004B, 0x4B }, { 0x004C, 0x4C }, { 0x004D, 0x4D },
	{ 0x004E, 0x4E }, { 0x004F, 0x4F }, { 0x0050, 0x50 }, { 0x0051, 0x51 },
	{ 0x0052, 0x52 }, { 0x0053, 0x53 }, { 0x0054, 0x54 }, { 0x0055, 0x55 },
	{ 0x0056, 0x56 }, { 0x0057, 0x57 }, { 0x0058, 0x58 }, { 0x0059, 0x59 },
	{ 0x005A, 0x5A }, { 0x005F, 0x11 }, { 0x0061, 0x61 }, { 0x0062, 0x62 },
	{ 0x0063, 0x63 }, { 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 },
	{ 0x0067, 0x67 }, { 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A },
	{ 0x006B, 0x6B }, { 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E },
	{ 0x006F, 0x6F }, { 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 },
	{ 0x0073, 0x73 }, { 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 },
	{ 0x0077, 0x77 }, { 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A },
	{ 0x00A0, 0x20 }, { 0x00A1, 0x40 }, { 0x00A3, 0x01 }, { 0x00A4, 0x24 },
	{ 0x00A5, 0x03 }, { 0x00A7, 0x5F }, { 0x00BF, 0x60 }, { 0x00C4, 0x5B },
	{ 0x00C5, 0x0E }, { 0x00C6, 0x1C }, { 0x00C7, 0x09 }, { 0x00C9, 0x1F },
	{ 0x00D1, 0x5D }, { 0x00D6, 0x5C }, { 0x00D8, 0x0B }, { 0x00DC, 0x5E },
	{ 0x00DF, 0x1E }, { 0x00E0, 0x7F }, { 0x00E4, 0x7B }, { 0x00E5, 0x0F },
	{ 0x00E6, 0x1D }, { 0x00E8, 0x04 }, { 0x00E9, 0x05 }, { 0x00EC, 0x07 },
	{ 0x00F1, 0x7D }, { 0x00F2, 0x08 }, { 0x00F6, 0x7C }, { 0x00F8, 0x0C },
	{ 0x00F9, 0x06 }, { 0x00FC, 0x7E }, { 0x0393, 0x13 }, { 0x0394, 0x10 },
	{ 0x0398, 0x19 }, { 0x039B, 0x14 }, { 0x039E, 0x1A }, { 0x03A0, 0x16 },
	{ 0x03A3, 0x18 }, { 0x03A6, 0x12 }, { 0x03A8, 0x17 }, { 0x03A9, 0x15 }
};

/* Appendix A.3.1 in 3GPP TS23.038 */
static const unsigned short tur_gsm[] = {
	0x0040, 0x00A3, 0x0024, 0x00A5, 0x20AC, 0x00E9, 0x00F9, 0x0131,
	0x00F2, 0x00C7, 0x000A, 0x011E, 0x011F, 0x000D, 0x00C5, 0x00E5,
	0x0394, 0x005F, 0x03A6, 0x0393, 0x039B, 0x03A9, 0x03A0, 0x03A8,
	0x03A3, 0x0398, 0x039E, 0x00A0, 0x015E, 0x015F, 0x00DF, 0x00C9,
	0x0020, 0x0021, 0x0022, 0x0023, 0x00A4, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
	0x0130, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005A, 0x00C4, 0x00D6, 0x00D1, 0x00DC, 0x00A7,
	0x00E7, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x00E4, 0x00F6, 0x00F1, 0x00FC, 0x00E0
};

static const struct codepoint tur_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0022, 0x22 }, { 0x0023, 0x23 }, { 0x0024, 0x02 }, { 0x0025, 0x25 },
	{ 0x0026, 0x26 }, { 0x0027, 0x27 }, { 0x0028, 0x28 }, { 0x0029, 0x29 },
	{ 0x002A, 0x2A }, { 0x002B, 0x2B }, { 0x002C, 0x2C }, { 0x002D, 0x2D },
	{ 0x002E, 0x2E }, { 0x002F, 0x2F }, { 0x0030, 0x30 }, { 0x0031, 0x31 },
	{ 0x0032, 0x32 }, { 0x0033, 0x33 }, { 0x0034, 0x34 }, { 0x0035, 0x35 },
	{ 0x0036, 0x36 }, { 0x0037, 0x37 }, { 0x0038, 0x38 }, { 0x0039, 0x39 },
	{ 0x003A, 0x3A }, { 0x003B, 0x3B }, { 0x003C, 0x3C }, { 0x003D, 0x3D },
	{ 0x003E, 0x3E }, { 0x003F, 0x3F }, { 0x0040, 0x00 }, { 0x0041, 0x41 },
	{ 0x0042, 0x42 }, { 0x0043, 0x43 }, { 0x0044, 0x44 }, { 0x0045, 0x45 },
	{ 0x0046, 0x46 }, { 0x0047, 0x47 }, { 0x0048, 0x48 }, { 0x0049, 0x49 },
	{ 0x004A, 0x4A }, { 0x004B, 0x4B }, { 0x004C, 0x4C }, { 0x004D, 0x4D },
	{ 0x004E, 0x4E }, { 0x004F, 0x4F }, { 0x0050, 0x50 }, { 0x0051, 0x51 },
	{ 0x0052, 0x52 }, { 0x0053, 0x53 }, { 0x0054, 0x54 }, { 0x0055, 0x55 },
	{ 0x0056, 0x56 }, { 0x0057, 0x57 }, { 0x0058, 0x58 }, { 0x0059, 0x59 },
	{ 0x005A, 0x5A }, { 0x005F, 0x11 }, { 0x0061, 0x61 }, { 0x0062, 0x62 },
	{ 0x0063, 0x63 }, { 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 },
	{ 0x0067, 0x67 }, { 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A },
	{ 0x006B, 0x6B }, { 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E },
	{ 0x006F, 0x6F }, { 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 },
	{ 0x0073, 0x73 }, { 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 },
	{ 0x0077, 0x77 }, { 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A },
	{ 0x00A0, 0x20 }, { 0x00A3, 0x01 }, { 0x00A4, 0x24 }, { 0x00A5, 0x03 },
	{ 0x00A7, 0x5F }, { 0x00C4, 0x5B }, { 0x00C5, 0x0E }, { 0x00C7, 0x09 },
	{ 0x00C9, 0x1F }, { 0x00D1, 0x5D }, { 0x00D6, 0x5C }, { 0x00DC, 0x5E },
	{ 0x00DF, 0x1E }, { 0x00E0, 0x7F }, { 0x00E4, 0x7B }, { 0x00E5, 0x0F },
	{ 0x00E7, 0x60 }, { 0x00E9, 0x05 }, { 0x00F1, 0x7D }, { 0x00F2, 0x08 },
	{ 0x00F6, 0x7C }, { 0x00F9, 0x06 }, { 0x00FC, 0x7E }, { 0x011E, 0x0B },
	{ 0x011F, 0x0C }, { 0x0130, 0x40 }, { 0x0131, 0x07 }, { 0x015E, 0x1C },
	{ 0x015F, 0x1D }, { 0x0393, 0x13 }, { 0x0394, 0x10 }, { 0x0398, 0x19 },
	{ 0x039B, 0x14 }, { 0x039E, 0x1A }, { 0x03A0, 0x16 }, { 0x03A3, 0x18 },
	{ 0x03A6, 0x12 }, { 0x03A8, 0x17 }, { 0x03A9, 0x15 }, { 0x20AC, 0x04 }
};

/* Appendix A.3.2 in 3GPP TS23.038 */
static const unsigned short por_gsm[] = {
	0x0040, 0x00A3, 0x0024, 0x00A5, 0x00EA, 0x00E9, 0x00FA, 0x00ED,
	0x00F3, 0x00E7, 0x000A, 0x00D4, 0x00F4, 0x000D, 0x00C1, 0x00E1,
	0x0394, 0x005F, 0x00AA, 0x00C7, 0x00C0, 0x221E, 0x005E, 0x005C,
	0x20ac, 0x00D3, 0x007C, 0x00A0, 0x00C2, 0x00E2, 0x00CA, 0x00C9,
	0x0020, 0x0021, 0x0022, 0x0023, 0x00BA, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
	0x00CD, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004A, 0x004B, 0x004C, 0x004D, 0x004E, 0x004F,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005A, 0x00C3, 0x00D5, 0x00DA, 0x00DC, 0x00A7,
	0x007E, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x00E3, 0x00F5, 0x0060, 0x00FC, 0x00E0
};

static const struct codepoint por_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0022, 0x22 }, { 0x0023, 0x23 }, { 0x0024, 0x02 }, { 0x0025, 0x25 },
	{ 0x0026, 0x26 }, { 0x0027, 0x27 }, { 0x0028, 0x28 }, { 0x0029, 0x29 },
	{ 0x002A, 0x2A }, { 0x002B, 0x2B }, { 0x002C, 0x2C }, { 0x002D, 0x2D },
	{ 0x002E, 0x2E }, { 0x002F, 0x2F }, { 0x0030, 0x30 }, { 0x0031, 0x31 },
	{ 0x0032, 0x32 }, { 0x0033, 0x33 }, { 0x0034, 0x34 }, { 0x0035, 0x35 },
	{ 0x0036, 0x36 }, { 0x0037, 0x37 }, { 0x0038, 0x38 }, { 0x0039, 0x39 },
	{ 0x003A, 0x3A }, { 0x003B, 0x3B }, { 0x003C, 0x3C }, { 0x003D, 0x3D },
	{ 0x003E, 0x3E }, { 0x003F, 0x3F }, { 0x0040, 0x00 }, { 0x0041, 0x41 },
	{ 0x0042, 0x42 }, { 0x0043, 0x43 }, { 0x0044, 0x44 }, { 0x0045, 0x45 },
	{ 0x0046, 0x46 }, { 0x0047, 0x47 }, { 0x0048, 0x48 }, { 0x0049, 0x49 },
	{ 0x004A, 0x4A }, { 0x004B, 0x4B }, { 0x004C, 0x4C }, { 0x004D, 0x4D },
	{ 0x004E, 0x4E }, { 0x004F, 0x4F }, { 0x0050, 0x50 }, { 0x0051, 0x51 },
	{ 0x0052, 0x52 }, { 0x0053, 0x53 }, { 0x0054, 0x54 }, { 0x0055, 0x55 },
	{ 0x0056, 0x56 }, { 0x0057, 0x57 }, { 0x0058, 0x58 }, { 0x0059, 0x59 },
	{ 0x005A, 0x5A }, { 0x005C, 0x17 }, { 0x005E, 0x16 }, { 0x005F, 0x11 },
	{ 0x0060, 0x7D }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x007C, 0x1A },
	{ 0x007E, 0x60 }, { 0x00A0, 0x20 }, { 0x00A3, 0x01 }, { 0x00A5, 0x03 },
	{ 0x00A7, 0x5F }, { 0x00AA, 0x12 }, { 0x00BA, 0x24 }, { 0x00C0, 0x14 },
	{ 0x00C1, 0x0E }, { 0x00C2, 0x1C }, { 0x00C3, 0x5B }, { 0x00C7, 0x13 },
	{ 0x00C9, 0x1F }, { 0x00CA, 0x1E }, { 0x00CD, 0x40 }, { 0x00D3, 0x19 },
	{ 0x00D4, 0x0B }, { 0x00D5, 0x5C }, { 0x00DA, 0x5D }, { 0x00DC, 0x5E },
	{ 0x00E0, 0x7F }, { 0x00E1, 0x0F }, { 0x00E2, 0x1D }, { 0x00E3, 0x7B },
	{ 0x00E7, 0x09 }, { 0x00E9, 0x05 }, { 0x00EA, 0x04 }, { 0x00ED, 0x07 },
	{ 0x00F3, 0x08 }, { 0x00F4, 0x0C }, { 0x00F5, 0x7C }, { 0x00FA, 0x06 },
	{ 0x00FC, 0x7E }, { 0x0394, 0x10 }, { 0x20AC, 0x18 }, { 0x221E, 0x15 }
};

/* Appendix A.3.4 in 3GPP TS23.038 */
static const unsigned short ben_gsm[] = {
	0x0981, 0x0982, 0x0983, 0x0985, 0x0986, 0x0987, 0x0988, 0x0989,
	0x098A, 0x098B, 0x000A, 0x098C, 0x0020, 0x000D, 0x0020, 0x098F,
	0x0990, 0x0020, 0x0020, 0x0993, 0x0994, 0x0995, 0x0996, 0x0997,
	0x0998, 0x0999, 0x099A, 0x00A0, 0x099B, 0x099C, 0x099D, 0x099E,
	0x0020, 0x0021, 0x099F, 0x09A0, 0x09A1, 0x09A2, 0x09A3, 0x09A4,
	0x0029, 0x0028, 0x09A5, 0x09A6, 0x002C, 0x09A7, 0x002E, 0x09A8,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x09AA, 0x09AB, 0x003F,
	0x09AC, 0x09AD, 0x09AE, 0x09AF, 0x09B0, 0x0020, 0x09B2, 0x0020,
	0x0020, 0x0020, 0x09B6, 0x09B7, 0x09B8, 0x09B9, 0x09BC, 0x09BD,
	0x09BE, 0x09BF, 0x09C0, 0x09C1, 0x09C2, 0x09C3, 0x09C4, 0x0020,
	0x0020, 0x09C7, 0x09C8, 0x0020, 0x0020, 0x09CB, 0x09CC, 0x09CD,
	0x09CE, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x09D7, 0x09DC, 0x09DD, 0x09F0, 0x09F1
};

static const struct codepoint ben_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0981, 0x00 }, { 0x0982, 0x01 }, { 0x0983, 0x02 }, { 0x0985, 0x03 },
	{ 0x0986, 0x04 }, { 0x0987, 0x05 }, { 0x0988, 0x06 }, { 0x0989, 0x07 },
	{ 0x098A, 0x08 }, { 0x098B, 0x09 }, { 0x098C, 0x0B }, { 0x098F, 0x0F },
	{ 0x0990, 0x10 }, { 0x0993, 0x13 }, { 0x0994, 0x14 }, { 0x0995, 0x15 },
	{ 0x0996, 0x16 }, { 0x0997, 0x17 }, { 0x0998, 0x18 }, { 0x0999, 0x19 },
	{ 0x099A, 0x1A }, { 0x099B, 0x1C }, { 0x099C, 0x1D }, { 0x099D, 0x1E },
	{ 0x099E, 0x1F }, { 0x099F, 0x22 }, { 0x09A0, 0x23 }, { 0x09A1, 0x24 },
	{ 0x09A2, 0x25 }, { 0x09A3, 0x26 }, { 0x09A4, 0x27 }, { 0x09A5, 0x2A },
	{ 0x09A6, 0x2B }, { 0x09A7, 0x2D }, { 0x09A8, 0x2F }, { 0x09AA, 0x3D },
	{ 0x09AB, 0x3E }, { 0x09AC, 0x40 }, { 0x09AD, 0x41 }, { 0x09AE, 0x42 },
	{ 0x09AF, 0x43 }, { 0x09B0, 0x44 }, { 0x09B2, 0x46 }, { 0x09B6, 0x4A },
	{ 0x09B7, 0x4B }, { 0x09B8, 0x4C }, { 0x09B9, 0x4D }, { 0x09BC, 0x4E },
	{ 0x09BD, 0x4F }, { 0x09BE, 0x50 }, { 0x09BF, 0x51 }, { 0x09C0, 0x52 },
	{ 0x09C1, 0x53 }, { 0x09C2, 0x54 }, { 0x09C3, 0x55 }, { 0x09C4, 0x56 },
	{ 0x09C7, 0x59 }, { 0x09C8, 0x5A }, { 0x09CB, 0x5D }, { 0x09CC, 0x5E },
	{ 0x09CD, 0x5F }, { 0x09CE, 0x60 }, { 0x09D7, 0x7B }, { 0x09DC, 0x7C },
	{ 0x09DD, 0x7D }, { 0x09F0, 0x7E }, { 0x09F1, 0x7F }
};

/* Appendix A.3.5 in 3GPP TS23.038 */
static const unsigned short guj_gsm[] = {
	0x0A81, 0x0A82, 0x0A83, 0x0A85, 0x0A86, 0x0A87, 0x0A88, 0x0A89,
	0x0A8A, 0x0A8B, 0x000A, 0x0A8C, 0x0A8D, 0x000D, 0x0020, 0x0A8F,
	0x0A90, 0x0A91, 0x0020, 0x0A93, 0x0A94, 0x0A95, 0x0A96, 0x0A97,
	0x0A98, 0x0A99, 0x0A9A, 0x00A0, 0x0A9B, 0x0A9C, 0x0A9D, 0x0A9E,
	0x0020, 0x0021, 0x0A9F, 0x0AA0, 0x0AA1, 0x0AA2, 0x0AA3, 0x0AA4,
	0x0029, 0x0028, 0x0AA5, 0x0AA6, 0x002C, 0x0AA7, 0x002E, 0x0AA8,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0AAA, 0x0AAB, 0x003F,
	0x0AAC, 0x0AAD, 0x0AAE, 0x0AAF, 0x0AB0, 0x0020, 0x0AB2, 0x0AB3,
	0x0020, 0x0AB5, 0x0AB6, 0x0AB7, 0x0AB8, 0x0AB9, 0x0ABC, 0x0ABD,
	0x0ABE, 0x0ABF, 0x0AC0, 0x0AC1, 0x0AC2, 0x0AC3, 0x0AC4, 0x0AC5,
	0x0020, 0x0AC7, 0x0AC8, 0x0AC9, 0x0020, 0x0ACB, 0x0ACC, 0x0ACD,
	0x0AD0, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x0AE0, 0x0AE1, 0x0AE2, 0x0AE3, 0x0AF1
};

static const struct codepoint guj_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0A81, 0x00 }, { 0x0A82, 0x01 }, { 0x0A83, 0x02 }, { 0x0A85, 0x03 },
	{ 0x0A86, 0x04 }, { 0x0A87, 0x05 }, { 0x0A88, 0x06 }, { 0x0A89, 0x07 },
	{ 0x0A8A, 0x08 }, { 0x0A8B, 0x09 }, { 0x0A8C, 0x0B }, { 0x0A8D, 0x0C },
	{ 0x0A8F, 0x0F }, { 0x0A90, 0x10 }, { 0x0A91, 0x11 }, { 0x0A93, 0x13 },
	{ 0x0A94, 0x14 }, { 0x0A95, 0x15 }, { 0x0A96, 0x16 }, { 0x0A97, 0x17 },
	{ 0x0A98, 0x18 }, { 0x0A99, 0x19 }, { 0x0A9A, 0x1A }, { 0x0A9B, 0x1C },
	{ 0x0A9C, 0x1D }, { 0x0A9D, 0x1E }, { 0x0A9E, 0x1F }, { 0x0A9F, 0x22 },
	{ 0x0AA0, 0x23 }, { 0x0AA1, 0x24 }, { 0x0AA2, 0x25 }, { 0x0AA3, 0x26 },
	{ 0x0AA4, 0x27 }, { 0x0AA5, 0x2A }, { 0x0AA6, 0x2B }, { 0x0AA7, 0x2D },
	{ 0x0AA8, 0x2F }, { 0x0AAA, 0x3D }, { 0x0AAB, 0x3E }, { 0x0AAC, 0x40 },
	{ 0x0AAD, 0x41 }, { 0x0AAE, 0x42 }, { 0x0AAF, 0x43 }, { 0x0AB0, 0x44 },
	{ 0x0AB2, 0x46 }, { 0x0AB3, 0x47 }, { 0x0AB5, 0x49 }, { 0x0AB6, 0x4A },
	{ 0x0AB7, 0x4B }, { 0x0AB8, 0x4C }, { 0x0AB9, 0x4D }, { 0x0ABC, 0x4E },
	{ 0x0ABD, 0x4F }, { 0x0ABE, 0x50 }, { 0x0ABF, 0x51 }, { 0x0AC0, 0x52 },
	{ 0x0AC1, 0x53 }, { 0x0AC2, 0x54 }, { 0x0AC3, 0x55 }, { 0x0AC4, 0x56 },
	{ 0x0AC5, 0x57 }, { 0x0AC7, 0x59 }, { 0x0AC8, 0x5A }, { 0x0AC9, 0x5B },
	{ 0x0ACB, 0x5D }, { 0x0ACC, 0x5E }, { 0x0ACD, 0x5F }, { 0x0AD0, 0x60 },
	{ 0x0AE0, 0x7B }, { 0x0AE1, 0x7C }, { 0x0AE2, 0x7D }, { 0x0AE3, 0x7E },
	{ 0x0AF1, 0x7F }
};

/* Appendix A.3.6 in 3GPP TS23.038 */
static const unsigned short hin_gsm[] = {
	0x0901, 0x0902, 0x0903, 0x0905, 0x0906, 0x0907, 0x0908, 0x0909,
	0x090A, 0x090B, 0x000A, 0x090C, 0x090D, 0x000D, 0x090E, 0x090F,
	0x0910, 0x0911, 0x0912, 0x0913, 0x0914, 0x0915, 0x0916, 0x0917,
	0x0918, 0x0919, 0x091A, 0x00A0, 0x091B, 0x091C, 0x091D, 0x091E,
	0x0020, 0x0021, 0x091F, 0x0920, 0x0921, 0x0922, 0x0923, 0x0924,
	0x0029, 0x0028, 0x0925, 0x0926, 0x002C, 0x0927, 0x002E, 0x0928,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0929, 0x092A, 0x092B, 0x003F,
	0x092C, 0x092D, 0x092E, 0x092F, 0x0930, 0x0931, 0x0932, 0x0933,
	0x0934, 0x0935, 0x0936, 0x0937, 0x0938, 0x0939, 0x093C, 0x093D,
	0x093E, 0x093F, 0x0940, 0x0941, 0x0942, 0x0943, 0x0944, 0x0945,
	0x0946, 0x0947, 0x0948, 0x0949, 0x094A, 0x094B, 0x094C, 0x094D,
	0x0950, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x0972, 0x097B, 0x097C, 0x097E, 0x097F
};

static const struct codepoint hin_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0901, 0x00 }, { 0x0902, 0x01 }, { 0x0903, 0x02 }, { 0x0905, 0x03 },
	{ 0x0906, 0x04 }, { 0x0907, 0x05 }, { 0x0908, 0x06 }, { 0x0909, 0x07 },
	{ 0x090A, 0x08 }, { 0x090B, 0x09 }, { 0x090C, 0x0B }, { 0x090D, 0x0C },
	{ 0x090E, 0x0E }, { 0x090F, 0x0F }, { 0x0910, 0x10 }, { 0x0911, 0x11 },
	{ 0x0912, 0x12 }, { 0x0913, 0x13 }, { 0x0914, 0x14 }, { 0x0915, 0x15 },
	{ 0x0916, 0x16 }, { 0x0917, 0x17 }, { 0x0918, 0x18 }, { 0x0919, 0x19 },
	{ 0x091A, 0x1A }, { 0x091B, 0x1C }, { 0x091C, 0x1D }, { 0x091D, 0x1E },
	{ 0x091E, 0x1F }, { 0x091F, 0x22 }, { 0x0920, 0x23 }, { 0x0921, 0x24 },
	{ 0x0922, 0x25 }, { 0x0923, 0x26 }, { 0x0924, 0x27 }, { 0x0925, 0x2A },
	{ 0x0926, 0x2B }, { 0x0927, 0x2D }, { 0x0928, 0x2F }, { 0x0929, 0x3C },
	{ 0x092A, 0x3D }, { 0x092B, 0x3E }, { 0x092C, 0x40 }, { 0x092D, 0x41 },
	{ 0x092E, 0x42 }, { 0x092F, 0x43 }, { 0x0930, 0x44 }, { 0x0931, 0x45 },
	{ 0x0932, 0x46 }, { 0x0933, 0x47 }, { 0x0934, 0x48 }, { 0x0935, 0x49 },
	{ 0x0936, 0x4A }, { 0x0937, 0x4B }, { 0x0938, 0x4C }, { 0x0939, 0x4D },
	{ 0x093C, 0x4E }, { 0x093D, 0x4F }, { 0x093E, 0x50 }, { 0x093F, 0x51 },
	{ 0x0940, 0x52 }, { 0x0941, 0x53 }, { 0x0942, 0x54 }, { 0x0943, 0x55 },
	{ 0x0944, 0x56 }, { 0x0945, 0x57 }, { 0x0946, 0x58 }, { 0x0947, 0x59 },
	{ 0x0948, 0x5A }, { 0x0949, 0x5B }, { 0x094A, 0x5C }, { 0x094B, 0x5D },
	{ 0x094C, 0x5E }, { 0x094D, 0x5F }, { 0x0950, 0x60 }, { 0x0972, 0x7B },
	{ 0x097B, 0x7C }, { 0x097C, 0x7D }, { 0x097E, 0x7E }, { 0x097F, 0x7F }
};

/* Appendix A.3.7 in 3GPP TS23.038 */
static const unsigned short kan_gsm[] = {
	0x0020, 0x0C82, 0x0C83, 0x0C85, 0x0C86, 0x0C87, 0x0C88, 0x0C89,
	0x0C8A, 0x0C8B, 0x000A, 0x0C8C, 0x0020, 0x000D, 0x0C8E, 0x0C8F,
	0x0C90, 0x0020, 0x0C92, 0x0C93, 0x0C94, 0x0C95, 0x0C96, 0x0C97,
	0x0C98, 0x0C99, 0x0C9A, 0x00A0, 0x0C9B, 0x0C9C, 0x0C9D, 0x0C9E,
	0x0020, 0x0021, 0x0C9F, 0x0CA0, 0x0CAA, 0x0CA2, 0x0CA3, 0x0CA4,
	0x0029, 0x0028, 0x0CA5, 0x0CA6, 0x002C, 0x0CA7, 0x002E, 0x0CA8,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0CAA, 0x0CAB, 0x003F,
	0x0CAC, 0x0CAD, 0x0CAE, 0x0CAF, 0x0CB0, 0x0CB1, 0x0CB2, 0x0CB3,
	0x0020, 0x0CB5, 0x0CB6, 0x0CB7, 0x0CB8, 0x0CB9, 0x0CBC, 0x0CBD,
	0x0CBE, 0x0CBF, 0x0CC0, 0x0CC1, 0x0CC2, 0x0CC3, 0x0CC4, 0x0020,
	0x0CC6, 0x0CC7, 0x0CC8, 0x0020, 0x0CCA, 0x0CCB, 0x0CCC, 0x0CCD,
	0x0CD5, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076,	0x0077,
	0x0078, 0x0079, 0x007A, 0x0CD6, 0x0CE0, 0x0CE1, 0x0CE2,	0x0CE3
};

static const struct codepoint kan_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0C82, 0x01 }, { 0x0C83, 0x02 }, { 0x0C85, 0x03 }, { 0x0C86, 0x04 },
	{ 0x0C87, 0x05 }, { 0x0C88, 0x06 }, { 0x0C89, 0x07 }, { 0x0C8A, 0x08 },
	{ 0x0C8B, 0x09 }, { 0x0C8C, 0x0B }, { 0x0C8E, 0x0E }, { 0x0C8F, 0x0F },
	{ 0x0C90, 0x10 }, { 0x0C92, 0x12 }, { 0x0C93, 0x13 }, { 0x0C94, 0x14 },
	{ 0x0C95, 0x15 }, { 0x0C96, 0x16 }, { 0x0C97, 0x17 }, { 0x0C98, 0x18 },
	{ 0x0C99, 0x19 }, { 0x0C9A, 0x1A }, { 0x0C9B, 0x1C }, { 0x0C9C, 0x1D },
	{ 0x0C9D, 0x1E }, { 0x0C9E, 0x1F }, { 0x0C9F, 0x22 }, { 0x0CA0, 0x23 },
	{ 0x0CA2, 0x25 }, { 0x0CA3, 0x26 }, { 0x0CA4, 0x27 }, { 0x0CA5, 0x2A },
	{ 0x0CA6, 0x2B }, { 0x0CA7, 0x2D }, { 0x0CA8, 0x2F }, { 0x0CAA, 0x3D },
	{ 0x0CAB, 0x3E }, { 0x0CAC, 0x40 }, { 0x0CAD, 0x41 }, { 0x0CAE, 0x42 },
	{ 0x0CAF, 0x43 }, { 0x0CB0, 0x44 }, { 0x0CB1, 0x45 }, { 0x0CB2, 0x46 },
	{ 0x0CB3, 0x47 }, { 0x0CB5, 0x49 }, { 0x0CB6, 0x4A }, { 0x0CB7, 0x4B },
	{ 0x0CB8, 0x4C }, { 0x0CB9, 0x4D }, { 0x0CBC, 0x4E }, { 0x0CBD, 0x4F },
	{ 0x0CBE, 0x50 }, { 0x0CBF, 0x51 }, { 0x0CC0, 0x52 }, { 0x0CC1, 0x53 },
	{ 0x0CC2, 0x54 }, { 0x0CC3, 0x55 }, { 0x0CC4, 0x56 }, { 0x0CC6, 0x58 },
	{ 0x0CC7, 0x59 }, { 0x0CC8, 0x5A }, { 0x0CCA, 0x5C }, { 0x0CCB, 0x5D },
	{ 0x0CCC, 0x5E }, { 0x0CCD, 0x5F }, { 0x0CD5, 0x60 }, { 0x0CD6, 0x7B },
	{ 0x0CE0, 0x7C }, { 0x0CE1, 0x7D }, { 0x0CE2, 0x7E }, { 0x0CE3, 0x7F }
};

/* Appendix A.3.8 in 3GPP TS23.038 */
static const unsigned short mal_gsm[] = {
	0x0020, 0x0D02, 0x0D03, 0x0D05, 0x0D06, 0x0D07, 0x0D08, 0x0D09,
	0x0D0A, 0x0D0B, 0x000A, 0x0D0C, 0x0020, 0x000D, 0x0D0E, 0x0D0F,
	0x0D10, 0x0020, 0x0D12, 0x0D13, 0x0D14, 0x0D15, 0x0D16, 0x0D17,
	0x0D18, 0x0D19, 0x0D1A, 0x00A0, 0x0D1B, 0x0D1C, 0x0D1D, 0x0D1E,
	0x0020, 0x0021, 0x0D1F, 0x0D20, 0x0D21, 0x0D22, 0x0D23, 0x0D24,
	0x0029, 0x0028, 0x0D25, 0x0D26, 0x002C, 0x0D27, 0x002C, 0x0D28,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0D2A, 0x0D2B, 0x003F,
	0x0D2C, 0x0D2D, 0x0D2E, 0x0D2F, 0x0D30, 0x0D31, 0x0D32, 0x0D33,
	0x0D34, 0x0D35, 0x0D36, 0x0D37, 0x0D38, 0x0D39, 0x0020, 0x0D3D,
	0x0D3E, 0x0D3F, 0x0D40, 0x0D41, 0x0D42, 0x0D43, 0x0D44, 0x0020,
	0x0D46, 0x0D47, 0x0D48, 0x0020, 0x0D4A, 0x0D4B, 0x0D4C, 0x0D4D,
	0x0D57, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076,	0x0077,
	0x0078, 0x0079, 0x007A, 0x0D60, 0x0D61, 0x0D62, 0x0D63,	0x0D79
};

static const struct codepoint mal_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2E }, { 0x0030, 0x30 },
	{ 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 }, { 0x0034, 0x34 },
	{ 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 }, { 0x0038, 0x38 },
	{ 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B }, { 0x003F, 0x3F },
	{ 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 }, { 0x0064, 0x64 },
	{ 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 }, { 0x0068, 0x68 },
	{ 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B }, { 0x006C, 0x6C },
	{ 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F }, { 0x0070, 0x70 },
	{ 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 }, { 0x0074, 0x74 },
	{ 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 }, { 0x0078, 0x78 },
	{ 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 }, { 0x0D02, 0x01 },
	{ 0x0D03, 0x02 }, { 0x0D05, 0x03 }, { 0x0D06, 0x04 }, { 0x0D07, 0x05 },
	{ 0x0D08, 0x06 }, { 0x0D09, 0x07 }, { 0x0D0A, 0x08 }, { 0x0D0B, 0x09 },
	{ 0x0D0C, 0x0B }, { 0x0D0E, 0x0E }, { 0x0D0F, 0x0F }, { 0x0D10, 0x10 },
	{ 0x0D12, 0x12 }, { 0x0D13, 0x13 }, { 0x0D14, 0x14 }, { 0x0D15, 0x15 },
	{ 0x0D16, 0x16 }, { 0x0D17, 0x17 }, { 0x0D18, 0x18 }, { 0x0D19, 0x19 },
	{ 0x0D1A, 0x1A }, { 0x0D1B, 0x1C }, { 0x0D1C, 0x1D }, { 0x0D1D, 0x1E },
	{ 0x0D1E, 0x1F }, { 0x0D1F, 0x22 }, { 0x0D20, 0x23 }, { 0x0D21, 0x24 },
	{ 0x0D22, 0x25 }, { 0x0D23, 0x26 }, { 0x0D24, 0x27 }, { 0x0D25, 0x2A },
	{ 0x0D26, 0x2B }, { 0x0D27, 0x2D }, { 0x0D28, 0x2F }, { 0x0D2A, 0x3D },
	{ 0x0D2B, 0x3E }, { 0x0D2C, 0x40 }, { 0x0D2D, 0x41 }, { 0x0D2E, 0x42 },
	{ 0x0D2F, 0x43 }, { 0x0D30, 0x44 }, { 0x0D31, 0x45 }, { 0x0D32, 0x46 },
	{ 0x0D33, 0x47 }, { 0x0D34, 0x48 }, { 0x0D35, 0x49 }, { 0x0D36, 0x4A },
	{ 0x0D37, 0x4B }, { 0x0D38, 0x4C }, { 0x0D39, 0x4D }, { 0x0D3D, 0x4F },
	{ 0x0D3E, 0x50 }, { 0x0D3F, 0x51 }, { 0x0D40, 0x52 }, { 0x0D41, 0x53 },
	{ 0x0D42, 0x54 }, { 0x0D43, 0x55 }, { 0x0D44, 0x56 }, { 0x0D46, 0x58 },
	{ 0x0D47, 0x59 }, { 0x0D48, 0x5A }, { 0x0D4A, 0x5C }, { 0x0D4B, 0x5D },
	{ 0x0D4C, 0x5E }, { 0x0D4D, 0x5F }, { 0x0D57, 0x60 }, { 0x0D60, 0x7B },
	{ 0x0D61, 0x7C }, { 0x0D62, 0x7D }, { 0x0D63, 0x7E }, { 0x0D79, 0x7F }
};

/* Appendix A.3.9 in 3GPP TS23.038 */
static const unsigned short ori_gsm[] = {
	0x0B01, 0x0B02, 0x0B03, 0x0B05, 0x0B06, 0x0B07, 0x0B08, 0x0B09,
	0x0B0A, 0x0B0B, 0x000A, 0x0B0C, 0x0020, 0x000D, 0x0020, 0x0B0F,
	0x0B10, 0x0020, 0x0020, 0x0B13, 0x0B14, 0x0B15, 0x0B16, 0x0B17,
	0x0B18, 0x0B19, 0x0B1A, 0x00A0,	0x0B1B, 0x0B1C, 0x0B1D, 0x0B1E,
	0x0020, 0x0021, 0x0B1F, 0x0B20, 0x0B21, 0x0B22, 0x0B23, 0x0B24,
	0x0029, 0x0028, 0x0B25, 0x0B26, 0x002C, 0x0B27, 0x002E, 0x0B28,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0B2A, 0x0B2B, 0x003F,
	0x0B2C, 0x0B2D, 0x0B2E, 0x0B2F, 0x0B30, 0x0020, 0x0B32, 0x0B33,
	0x0020, 0x0B35, 0x0B36, 0x0B37, 0x0B38, 0x0B39, 0x0B3C, 0x0B3D,
	0x0B3E, 0x0B3F, 0x0B40, 0x0B41, 0x0B42, 0x0B43, 0x0B44, 0x0020,
	0x0020, 0x0B47, 0x0B48, 0x0020, 0x0020, 0x0B4B, 0x0B4C, 0x0B4D,
	0x0B56, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A,	0x0B57, 0x0B60, 0x0B61, 0x0B62, 0x0B63
};

static const struct codepoint ori_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0B01, 0x00 }, { 0x0B02, 0x01 }, { 0x0B03, 0x02 }, { 0x0B05, 0x03 },
	{ 0x0B06, 0x04 }, { 0x0B07, 0x05 }, { 0x0B08, 0x06 }, { 0x0B09, 0x07 },
	{ 0x0B0A, 0x08 }, { 0x0B0B, 0x09 }, { 0x0B0C, 0x0B }, { 0x0B0F, 0x0F },
	{ 0x0B10, 0x10 }, { 0x0B13, 0x13 }, { 0x0B14, 0x14 }, { 0x0B15, 0x15 },
	{ 0x0B16, 0x16 }, { 0x0B17, 0x17 }, { 0x0B18, 0x18 }, { 0x0B19, 0x19 },
	{ 0x0B1A, 0x1A }, { 0x0B1B, 0x1C }, { 0x0B1C, 0x1D }, { 0x0B1D, 0x1E },
	{ 0x0B1E, 0x1F }, { 0x0B1F, 0x22 }, { 0x0B20, 0x23 }, { 0x0B21, 0x24 },
	{ 0x0B22, 0x25 }, { 0x0B23, 0x26 }, { 0x0B24, 0x27 }, { 0x0B25, 0x2A },
	{ 0x0B26, 0x2B }, { 0x0B27, 0x2D }, { 0x0B28, 0x2F }, { 0x0B2A, 0x3D },
	{ 0x0B2B, 0x3E }, { 0x0B2C, 0x40 }, { 0x0B2D, 0x41 }, { 0x0B2E, 0x42 },
	{ 0x0B2F, 0x43 }, { 0x0B30, 0x44 }, { 0x0B32, 0x46 }, { 0x0B33, 0x47 },
	{ 0x0B35, 0x49 }, { 0x0B36, 0x4A }, { 0x0B37, 0x4B }, { 0x0B38, 0x4C },
	{ 0x0B39, 0x4D }, { 0x0B3C, 0x4E }, { 0x0B3D, 0x4F }, { 0x0B3E, 0x50 },
	{ 0x0B3F, 0x51 }, { 0x0B40, 0x52 }, { 0x0B41, 0x53 }, { 0x0B42, 0x54 },
	{ 0x0B43, 0x55 }, { 0x0B44, 0x56 }, { 0x0B47, 0x59 }, { 0x0B48, 0x5A },
	{ 0x0B4B, 0x5D }, { 0x0B4C, 0x5E }, { 0x0B4D, 0x5F }, { 0x0B56, 0x60 },
	{ 0x0B57, 0x7B }, { 0x0B60, 0x7C }, { 0x0B61, 0x7D }, { 0x0B62, 0x7E },
	{ 0x0B63, 0x7F }
};

/* Appendix A.3.10 in 3GPP TS23.038 */
static const unsigned short pun_gsm[] = {
	0x0A01, 0x0A02, 0x0A03, 0x0A05, 0x0A06, 0x0A07, 0x0A08, 0x0A09,
	0x0A0A, 0x0020, 0x000A, 0x0020, 0x0020, 0x000D, 0x0020, 0x0A0F,
	0x0A10, 0x0020, 0x0020, 0x0A13, 0x0A14, 0x0A15, 0x0A16, 0x0A17,
	0x0A18, 0x0A19, 0x0A1A, 0x00A0, 0x0A1B, 0x0A1C, 0x0A1D, 0x0A1E,
	0x0020, 0x0021, 0x0A1F, 0x0A20, 0x0A21, 0x0A22, 0x0A23, 0x0A24,
	0x0029, 0x0028, 0x0A25, 0x0A26, 0x002C, 0x0A27, 0x002E, 0x0A28,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0A2A, 0x0A2B, 0x003F,
	0x0A2C, 0x0A2D, 0x0A2E, 0x0A2F, 0x0A30, 0x0020, 0x0A32, 0x0A33,
	0x0020, 0x0A35, 0x0A36, 0x0020, 0x0A38, 0x0A39, 0x0A3C, 0x0020,
	0x0A3E, 0x0A3F, 0x0A40, 0x0A41, 0x0A42, 0x0020, 0x0020, 0x0020,
	0x0020, 0x0A47, 0x0A48, 0x0020, 0x0020, 0x0A4B, 0x0A4C, 0x0A4D,
	0x0A51, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x0A70, 0x0A71, 0x0A72, 0x0A73, 0x0A74
};

static const struct codepoint pun_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0A01, 0x00 }, { 0x0A02, 0x01 }, { 0x0A03, 0x02 }, { 0x0A05, 0x03 },
	{ 0x0A06, 0x04 }, { 0x0A07, 0x05 }, { 0x0A08, 0x06 }, { 0x0A09, 0x07 },
	{ 0x0A0A, 0x08 }, { 0x0A0F, 0x0F }, { 0x0A10, 0x10 }, { 0x0A13, 0x13 },
	{ 0x0A14, 0x14 }, { 0x0A15, 0x15 }, { 0x0A16, 0x16 }, { 0x0A17, 0x17 },
	{ 0x0A18, 0x18 }, { 0x0A19, 0x19 }, { 0x0A1A, 0x1A }, { 0x0A1B, 0x1C },
	{ 0x0A1C, 0x1D }, { 0x0A1D, 0x1E }, { 0x0A1E, 0x1F }, { 0x0A1F, 0x22 },
	{ 0x0A20, 0x23 }, { 0x0A21, 0x24 }, { 0x0A22, 0x25 }, { 0x0A23, 0x26 },
	{ 0x0A24, 0x27 }, { 0x0A25, 0x2A }, { 0x0A26, 0x2B }, { 0x0A27, 0x2D },
	{ 0x0A28, 0x2F }, { 0x0A2A, 0x3D }, { 0x0A2B, 0x3E }, { 0x0A2C, 0x40 },
	{ 0x0A2D, 0x41 }, { 0x0A2E, 0x42 }, { 0x0A2F, 0x43 }, { 0x0A30, 0x44 },
	{ 0x0A32, 0x46 }, { 0x0A33, 0x47 }, { 0x0A35, 0x49 }, { 0x0A36, 0x4A },
	{ 0x0A38, 0x4C }, { 0x0A39, 0x4D }, { 0x0A3C, 0x4E }, { 0x0A3E, 0x50 },
	{ 0x0A3F, 0x51 }, { 0x0A40, 0x52 }, { 0x0A41, 0x53 }, { 0x0A42, 0x54 },
	{ 0x0A47, 0x59 }, { 0x0A48, 0x5A }, { 0x0A4B, 0x5D }, { 0x0A4C, 0x5E },
	{ 0x0A4D, 0x5F }, { 0x0A51, 0x60 }, { 0x0A70, 0x7B }, { 0x0A71, 0x7C },
	{ 0x0A72, 0x7D }, { 0x0A73, 0x7E }, { 0x0A74, 0x7F }
};

/* Appendix A.3.11 in 3GPP TS23.038 */
static const unsigned short tam_gsm[] = {
	0x0020, 0x0B82, 0x0B83, 0x0B85, 0x0B86, 0x0B87, 0x0B88, 0x0B89,
	0x0B8A, 0x0020, 0x000A, 0x0020, 0x0020, 0x000D, 0x0B8E, 0x0B8F,
	0x0B90, 0x0020, 0x0B92, 0x0B93, 0x0B94, 0x0B95, 0x0020, 0x0020,
	0x0020, 0x0B99, 0x0B9A, 0x00A0, 0x0020, 0x0B9C, 0x0020, 0x0B9E,
	0x0020, 0x0021, 0x0B9F, 0x0020, 0x0020, 0x0020, 0x0BA3, 0x0BA4,
	0x0029, 0x0028, 0x0020, 0x0020, 0x002C, 0x0020, 0x002E, 0x0BA8,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0BA9, 0x0BAA, 0x0020, 0x003F,
	0x0020, 0x0020, 0x0BAE, 0x0BAF, 0x0BB0, 0x0BB1, 0x0BB2, 0x0BB3,
	0x0BB4, 0x0BB5, 0x0BB6, 0x0BB7, 0x0BB8, 0x0BB9, 0x0020, 0x0020,
	0x0BBE, 0x0BBF, 0x0BC0, 0x0BC1, 0x0BC2, 0x0020, 0x0020, 0x0020,
	0x0BC6, 0x0BC7, 0x0BC8, 0x0020, 0x0BCA, 0x0BCB, 0x0BCC, 0x0BCD,
	0x0BD0, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070,	0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078,	0x0079, 0x007A, 0x0BD7, 0x0BF0, 0x0BF1, 0x0BF2, 0x0BF9
};

static const struct codepoint tam_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0B82, 0x01 }, { 0x0B83, 0x02 }, { 0x0B85, 0x03 }, { 0x0B86, 0x04 },
	{ 0x0B87, 0x05 }, { 0x0B88, 0x06 }, { 0x0B89, 0x07 }, { 0x0B8A, 0x08 },
	{ 0x0B8E, 0x0E }, { 0x0B8F, 0x0F }, { 0x0B90, 0x10 }, { 0x0B92, 0x12 },
	{ 0x0B93, 0x13 }, { 0x0B94, 0x14 }, { 0x0B95, 0x15 }, { 0x0B99, 0x19 },
	{ 0x0B9A, 0x1A }, { 0x0B9C, 0x1D }, { 0x0B9E, 0x1F }, { 0x0B9F, 0x22 },
	{ 0x0BA3, 0x26 }, { 0x0BA4, 0x27 }, { 0x0BA8, 0x2F }, { 0x0BA9, 0x3C },
	{ 0x0BAA, 0x3D }, { 0x0BAE, 0x42 }, { 0x0BAF, 0x43 }, { 0x0BB0, 0x44 },
	{ 0x0BB1, 0x45 }, { 0x0BB2, 0x46 }, { 0x0BB3, 0x47 }, { 0x0BB4, 0x48 },
	{ 0x0BB5, 0x49 }, { 0x0BB6, 0x4A }, { 0x0BB7, 0x4B }, { 0x0BB8, 0x4C },
	{ 0x0BB9, 0x4D }, { 0x0BBE, 0x50 }, { 0x0BBF, 0x51 }, { 0x0BC0, 0x52 },
	{ 0x0BC1, 0x53 }, { 0x0BC2, 0x54 }, { 0x0BC6, 0x58 }, { 0x0BC7, 0x59 },
	{ 0x0BC8, 0x5A }, { 0x0BCA, 0x5C }, { 0x0BCB, 0x5D }, { 0x0BCC, 0x5E },
	{ 0x0BCD, 0x5F }, { 0x0BD0, 0x60 }, { 0x0BD7, 0x7B }, { 0x0BF0, 0x7C },
	{ 0x0BF1, 0x7D }, { 0x0BF2, 0x7E }, { 0x0BF9, 0x7F }
};

/* Appendix A.3.12 in 3GPP TS23.038 */
static const unsigned short tel_gsm[] = {
	0x0C01, 0x0C02, 0x0C03, 0x0C05, 0x0C06, 0x0C07, 0x0C08, 0x0C09,
	0x0C0A, 0x0C0B, 0x000A, 0x0C0C, 0x0020, 0x000D, 0x0C0E, 0x0C0F,
	0x0C10, 0x0020, 0x0C12, 0x0C13, 0x0C14, 0x0C15, 0x0C16, 0x0C17,
	0x0C18, 0x0C19, 0x0C1A, 0x00A0, 0x0C1B, 0x0C1C, 0x0C1D, 0x0C1E,
	0x0020, 0x0021, 0x0C1F, 0x0C20, 0x0C21, 0x0C22, 0x0C23, 0x0C24,
	0x0029, 0x0028,	0x0C25, 0x0C26, 0x002C, 0x0C27, 0x002E, 0x0C28,
	0x0030, 0x0031,	0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x0020, 0x0C2A, 0x0C2B, 0x003F,
	0x0C2C, 0x0C2D, 0x0C2E, 0x0C2F, 0x0C30, 0x0C31, 0x0C32, 0x0C33,
	0x0020, 0x0C35, 0x0C36, 0x0C37, 0x0C38, 0x0C39, 0x0020, 0x0C3D,
	0x0C3E, 0x0C3F, 0x0C40, 0x0C41, 0x0C42, 0x0C43, 0x0C44, 0x0020,
	0x0C46, 0x0C47, 0x0C48, 0x0020, 0x0C4A, 0x0C4B, 0x0C4C, 0x0C4D,
	0x0C55, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x0C56, 0x0C60, 0x0C61, 0x0C62, 0x0C63
};

static const struct codepoint tel_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0C01, 0x00 }, { 0x0C02, 0x01 }, { 0x0C03, 0x02 }, { 0x0C05, 0x03 },
	{ 0x0C06, 0x04 }, { 0x0C07, 0x05 }, { 0x0C08, 0x06 }, { 0x0C09, 0x07 },
	{ 0x0C0A, 0x08 }, { 0x0C0B, 0x09 }, { 0x0C0C, 0x0B }, { 0x0C0E, 0x0E },
	{ 0x0C0F, 0x0F }, { 0x0C10, 0x10 }, { 0x0C12, 0x12 }, { 0x0C13, 0x13 },
	{ 0x0C14, 0x14 }, { 0x0C15, 0x15 }, { 0x0C16, 0x16 }, { 0x0C17, 0x17 },
	{ 0x0C18, 0x18 }, { 0x0C19, 0x19 }, { 0x0C1A, 0x1A }, { 0x0C1B, 0x1C },
	{ 0x0C1C, 0x1D }, { 0x0C1D, 0x1E }, { 0x0C1E, 0x1F }, { 0x0C1F, 0x22 },
	{ 0x0C20, 0x23 }, { 0x0C21, 0x24 }, { 0x0C22, 0x25 }, { 0x0C23, 0x26 },
	{ 0x0C24, 0x27 }, { 0x0C25, 0x2A }, { 0x0C26, 0x2B }, { 0x0C27, 0x2D },
	{ 0x0C28, 0x2F }, { 0x0C2A, 0x3D }, { 0x0C2B, 0x3E }, { 0x0C2C, 0x40 },
	{ 0x0C2D, 0x41 }, { 0x0C2E, 0x42 }, { 0x0C2F, 0x43 }, { 0x0C30, 0x44 },
	{ 0x0C31, 0x45 }, { 0x0C32, 0x46 }, { 0x0C33, 0x47 }, { 0x0C35, 0x49 },
	{ 0x0C36, 0x4A }, { 0x0C37, 0x4B }, { 0x0C38, 0x4C }, { 0x0C39, 0x4D },
	{ 0x0C3D, 0x4F }, { 0x0C3E, 0x50 }, { 0x0C3F, 0x51 }, { 0x0C40, 0x52 },
	{ 0x0C41, 0x53 }, { 0x0C42, 0x54 }, { 0x0C43, 0x55 }, { 0x0C44, 0x56 },
	{ 0x0C46, 0x58 }, { 0x0C47, 0x59 }, { 0x0C48, 0x5A }, { 0x0C4A, 0x5C },
	{ 0x0C4B, 0x5D }, { 0x0C4C, 0x5E }, { 0x0C4D, 0x5F }, { 0x0C55, 0x60 },
	{ 0x0C56, 0x7B }, { 0x0C60, 0x7C }, { 0x0C61, 0x7D }, { 0x0C62, 0x7E },
	{ 0x0C63, 0x7F }
};

/* Appendix A.3.13 in 3GPP TS23.038 */
static const unsigned short urd_gsm[] = {
	0x0627, 0x0622, 0x0628, 0x067B, 0x0680, 0x067E, 0x06A6, 0x062A,
	0x06C2, 0x067F, 0x000A, 0x0679, 0x067D, 0x000D, 0x067A, 0x067C,
	0x062B, 0x062C, 0x0681, 0x0684, 0x0683, 0x0685, 0x0686, 0x0687,
	0x062D, 0x062E, 0x062F, 0x00A0, 0x068C, 0x0688, 0x0689, 0x068A,
	0x0020, 0x0021, 0x068F, 0x068D, 0x0630, 0x0631, 0x0691, 0x0693,
	0x0029, 0x0028, 0x0699, 0x0632, 0x002C, 0x0696, 0x002E, 0x0698,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003A, 0x003B, 0x069A, 0x0633, 0x0634, 0x003F,
	0x0635, 0x0636, 0x0637, 0x0638, 0x0639, 0x0641, 0x0642, 0x06A9,
	0x06AA, 0x06AB, 0x06AF, 0x06B3, 0x06B1, 0x0644, 0x0645, 0x0646,
	0x06BA, 0x06BB, 0x06BC, 0x0648, 0x06C4, 0x06D5, 0x06C1, 0x06BE,
	0x0621, 0x06CC, 0x06D0, 0x06D2, 0x064D, 0x0650, 0x064F, 0x0657,
	0x0654, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007A, 0x0655, 0x0651, 0x0653, 0x0656, 0x0670
};

const struct codepoint urd_unicode[] = {
	{ 0x000A, 0x0A }, { 0x000D, 0x0D }, { 0x0020, 0x20 }, { 0x0021, 0x21 },
	{ 0x0028, 0x29 }, { 0x0029, 0x28 }, { 0x002C, 0x2C }, { 0x002E, 0x2E },
	{ 0x0030, 0x30 }, { 0x0031, 0x31 }, { 0x0032, 0x32 }, { 0x0033, 0x33 },
	{ 0x0034, 0x34 }, { 0x0035, 0x35 }, { 0x0036, 0x36 }, { 0x0037, 0x37 },
	{ 0x0038, 0x38 }, { 0x0039, 0x39 }, { 0x003A, 0x3A }, { 0x003B, 0x3B },
	{ 0x003F, 0x3F }, { 0x0061, 0x61 }, { 0x0062, 0x62 }, { 0x0063, 0x63 },
	{ 0x0064, 0x64 }, { 0x0065, 0x65 }, { 0x0066, 0x66 }, { 0x0067, 0x67 },
	{ 0x0068, 0x68 }, { 0x0069, 0x69 }, { 0x006A, 0x6A }, { 0x006B, 0x6B },
	{ 0x006C, 0x6C }, { 0x006D, 0x6D }, { 0x006E, 0x6E }, { 0x006F, 0x6F },
	{ 0x0070, 0x70 }, { 0x0071, 0x71 }, { 0x0072, 0x72 }, { 0x0073, 0x73 },
	{ 0x0074, 0x74 }, { 0x0075, 0x75 }, { 0x0076, 0x76 }, { 0x0077, 0x77 },
	{ 0x0078, 0x78 }, { 0x0079, 0x79 }, { 0x007A, 0x7A }, { 0x00A0, 0x20 },
	{ 0x0621, 0x58 }, { 0x0622, 0x01 }, { 0x0627, 0x00 }, { 0x0628, 0x02 },
	{ 0x062A, 0x07 }, { 0x062B, 0x10 }, { 0x062C, 0x11 }, { 0x062D, 0x18 },
	{ 0x062E, 0x19 }, { 0x062F, 0x1A }, { 0x0630, 0x24 }, { 0x0631, 0x25 },
	{ 0x0632, 0x2B }, { 0x0633, 0x3D }, { 0x0634, 0x3E }, { 0x0635, 0x40 },
	{ 0x0636, 0x41 }, { 0x0637, 0x42 }, { 0x0638, 0x43 }, { 0x0639, 0x44 },
	{ 0x0641, 0x45 }, { 0x0642, 0x46 }, { 0x0644, 0x4D }, { 0x0645, 0x4E },
	{ 0x0646, 0x4F }, { 0x0648, 0x53 }, { 0x064D, 0x5C }, { 0x064F, 0x5E },
	{ 0x0650, 0x5D }, { 0x0651, 0x7C }, { 0x0653, 0x7D }, { 0x0654, 0x60 },
	{ 0x0655, 0x7B }, { 0x0656, 0x7E }, { 0x0657, 0x5F }, { 0x0670, 0x7F },
	{ 0x0679, 0x0B }, { 0x067A, 0x0E }, { 0x067B, 0x03 }, { 0x067C, 0x0F },
	{ 0x067D, 0x0C }, { 0x067E, 0x05 }, { 0x067F, 0x09 }, { 0x0680, 0x04 },
	{ 0x0681, 0x12 }, { 0x0683, 0x14 }, { 0x0684, 0x13 }, { 0x0685, 0x15 },
	{ 0x0686, 0x16 }, { 0x0687, 0x17 }, { 0x0688, 0x1D }, { 0x0689, 0x1E },
	{ 0x068A, 0x1F }, { 0x068C, 0x1C }, { 0x068D, 0x23 }, { 0x068F, 0x22 },
	{ 0x0691, 0x26 }, { 0x0693, 0x27 }, { 0x0696, 0x2D }, { 0x0698, 0x2F },
	{ 0x0699, 0x2A }, { 0x069A, 0x3C }, { 0x06A6, 0x06 }, { 0x06A9, 0x47 },
	{ 0x06AA, 0x48 }, { 0x06AB, 0x49 }, { 0x06AF, 0x4A }, { 0x06B1, 0x4C },
	{ 0x06B3, 0x4B }, { 0x06BA, 0x50 }, { 0x06BB, 0x51 }, { 0x06BC, 0x52 },
	{ 0x06BE, 0x57 }, { 0x06C1, 0x56 }, { 0x06C2, 0x08 }, { 0x06C4, 0x54 },
	{ 0x06CC, 0x59 }, { 0x06D0, 0x5A }, { 0x06D2, 0x5B }, { 0x06D5, 0x55 }
};

static int compare_codepoints(const void *a, const void *b)
{
	const struct codepoint *ca = (const struct codepoint *) a;
	const struct codepoint *cb = (const struct codepoint *) b;

	return (ca->from > cb->from) - (ca->from < cb->from);
}

static unsigned short codepoint_lookup(struct codepoint *key,
					const struct codepoint *table,
					unsigned int len)
{
	struct codepoint *result = NULL;

	result = bsearch(key, table, len, sizeof(struct codepoint),
				compare_codepoints);

	return result ? result->to : GUND;
}

static unsigned short gsm_locking_shift_lookup(struct conversion_table *t,
						unsigned char k)
{
	return t->locking_g[k];
}

static unsigned short gsm_single_shift_lookup(struct conversion_table *t,
						unsigned char k)
{
	struct codepoint key = { k, 0 };
	return codepoint_lookup(&key, t->single_g, t->single_len_g);
}

static unsigned short unicode_locking_shift_lookup(struct conversion_table *t,
							unsigned short k)
{
	struct codepoint key = { k, 0 };
	return codepoint_lookup(&key, t->locking_u, t->locking_len_u);
}

static unsigned short unicode_single_shift_lookup(struct conversion_table *t,
							unsigned short k)
{
	struct codepoint key = { k, 0 };
	return codepoint_lookup(&key, t->single_u, t->single_len_u);
}

static bool populate_locking_shift(struct conversion_table *t,
					enum gsm_dialect lang)
{
	switch (lang) {
	case GSM_DIALECT_DEFAULT:
	case GSM_DIALECT_SPANISH:
		t->locking_g = def_gsm;
		t->locking_u = def_unicode;
		t->locking_len_u = L_ARRAY_SIZE(def_unicode);
		return true;

	case GSM_DIALECT_TURKISH:
		t->locking_g = tur_gsm;
		t->locking_u = tur_unicode;
		t->locking_len_u = L_ARRAY_SIZE(tur_unicode);
		return true;

	case GSM_DIALECT_PORTUGUESE:
		t->locking_g = por_gsm;
		t->locking_u = por_unicode;
		t->locking_len_u = L_ARRAY_SIZE(por_unicode);
		return true;

	case GSM_DIALECT_BENGALI:
		t->locking_g = ben_gsm;
		t->locking_u = ben_unicode;
		t->locking_len_u = L_ARRAY_SIZE(ben_unicode);
		return true;

	case GSM_DIALECT_GUJARATI:
                t->locking_g = guj_gsm;
                t->locking_u = guj_unicode;
                t->locking_len_u = L_ARRAY_SIZE(guj_unicode);
                return true;

	case GSM_DIALECT_HINDI:
		t->locking_g = hin_gsm;
		t->locking_u = hin_unicode;
		t->locking_len_u = L_ARRAY_SIZE(hin_unicode);
		return true;

	case GSM_DIALECT_KANNADA:
		t->locking_g = kan_gsm;
		t->locking_u = kan_unicode;
		t->locking_len_u = L_ARRAY_SIZE(kan_unicode);
		return true;

	case GSM_DIALECT_MALAYALAM:
		t->locking_g = mal_gsm;
		t->locking_u = mal_unicode;
		t->locking_len_u = L_ARRAY_SIZE(mal_unicode);
		return true;

	case GSM_DIALECT_ORIYA:
		t->locking_g = ori_gsm;
		t->locking_u = ori_unicode;
		t->locking_len_u = L_ARRAY_SIZE(ori_unicode);
		return true;

	case GSM_DIALECT_PUNJABI:
		t->locking_g = pun_gsm;
		t->locking_u = pun_unicode;
		t->locking_len_u = L_ARRAY_SIZE(pun_unicode);
		return true;

	case GSM_DIALECT_TAMIL:
		t->locking_g = tam_gsm;
		t->locking_u = tam_unicode;
		t->locking_len_u = L_ARRAY_SIZE(tam_unicode);
		return true;

	case GSM_DIALECT_TELUGU:
		t->locking_g = tel_gsm;
		t->locking_u = tel_unicode;
		t->locking_len_u = L_ARRAY_SIZE(tel_unicode);
		return true;

	case GSM_DIALECT_URDU:
		t->locking_g = urd_gsm;
		t->locking_u = urd_unicode;
		t->locking_len_u = L_ARRAY_SIZE(urd_unicode);
		return true;
        }

	return false;
}

static bool populate_single_shift(struct conversion_table *t,
					enum gsm_dialect lang)
{
	switch (lang) {
	case GSM_DIALECT_DEFAULT:
		t->single_g = def_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(def_ext_gsm);
		t->single_u = def_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(def_ext_unicode);
		return true;

	case GSM_DIALECT_TURKISH:
		t->single_g = tur_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(tur_ext_gsm);
		t->single_u = tur_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(tur_ext_unicode);
		return true;

	case GSM_DIALECT_SPANISH:
		t->single_g = spa_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(spa_ext_gsm);
		t->single_u = spa_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(spa_ext_unicode);
		return true;

	case GSM_DIALECT_PORTUGUESE:
		t->single_g = por_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(por_ext_gsm);
		t->single_u = por_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(por_ext_unicode);
		return true;

	case GSM_DIALECT_BENGALI:
		t->single_g = ben_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(ben_ext_gsm);
		t->single_u = ben_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(ben_ext_unicode);
		return true;

	case GSM_DIALECT_GUJARATI:
                t->single_g = guj_ext_gsm;
                t->single_len_g = L_ARRAY_SIZE(guj_ext_gsm);
                t->single_u = guj_ext_unicode;
                t->single_len_u = L_ARRAY_SIZE(guj_ext_unicode);
                return true;

	case GSM_DIALECT_HINDI:
		t->single_g = hin_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(hin_ext_gsm);
		t->single_u = hin_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(hin_ext_unicode);
		return true;

	case GSM_DIALECT_KANNADA:
		t->single_g = kan_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(kan_ext_gsm);
		t->single_u = kan_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(kan_ext_unicode);
		return true;

	case GSM_DIALECT_MALAYALAM:
		t->single_g = mal_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(mal_ext_gsm);
		t->single_u = mal_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(mal_ext_unicode);
		return true;

	case GSM_DIALECT_ORIYA:
		t->single_g = ori_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(ori_ext_gsm);
		t->single_u = ori_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(ori_ext_unicode);
		return true;

	case GSM_DIALECT_PUNJABI:
		t->single_g = pun_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(pun_ext_gsm);
		t->single_u = pun_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(pun_ext_unicode);
		return true;

	case GSM_DIALECT_TAMIL:
		t->single_g = tam_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(tam_ext_gsm);
		t->single_u = tam_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(tam_ext_unicode);
		return true;

	case GSM_DIALECT_TELUGU:
		t->single_g = tel_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(tel_ext_gsm);
		t->single_u = tel_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(tel_ext_unicode);
		return true;

	case GSM_DIALECT_URDU:
		t->single_g = urd_ext_gsm;
		t->single_len_g = L_ARRAY_SIZE(urd_ext_gsm);
		t->single_u = urd_ext_unicode;
		t->single_len_u = L_ARRAY_SIZE(urd_ext_unicode);
		return true;
        }

	return false;
}

static bool conversion_table_init(struct conversion_table *t,
					enum gsm_dialect locking,
					enum gsm_dialect single)
{
	memset(t, 0, sizeof(struct conversion_table));

	return populate_locking_shift(t, locking) &&
			populate_single_shift(t, single);
}

/*!
 * Converts text coded using GSM codec into UTF8 encoded text, using
 * the given language identifiers for single shift and locking shift
 * tables.  If len is less than 0, and terminator character is given,
 * the length is computed automatically.
 *
 * Returns newly-allocated UTF8 encoded string or NULL if the conversion
 * could not be performed.  Returns the number of bytes read from the
 * GSM encoded string in items_read (if not NULL), not including the
 * terminator character. Returns the number of bytes written into the UTF8
 * encoded string in items_written (if not NULL) not including the terminal
 * '\0' character.  The caller is responsible for freeing the returned value.
 */
char *convert_gsm_to_utf8_with_lang(const unsigned char *text, long len,
					long *items_read, long *items_written,
					unsigned char terminator,
					enum gsm_dialect locking_lang,
					enum gsm_dialect single_lang)
{
	char *res = NULL;
	char *out;
	long i = 0;
	long res_length;

	struct conversion_table t;

	if (!conversion_table_init(&t, locking_lang, single_lang))
		return NULL;

	if (len < 0 && !terminator)
		goto error;

	if (len < 0) {
		i = 0;

		while (text[i] != terminator)
			i++;

		len = i;
	}

	for (i = 0, res_length = 0; i < len; i++) {
		unsigned short c;

		if (text[i] > 0x7f)
			goto error;

		if (text[i] == 0x1b) {
			++i;
			if (i >= len)
				goto error;

			c = gsm_single_shift_lookup(&t, text[i]);

			/*
			 * According to the comment in the table from
			 * 3GPP 23.038, Section 6.2.1.1:
			 * "In the event that an MS receives a code where
			 * a symbol is not represented in the above table
			 * then the MS shall display either the character
			 * shown in the main GSM 7 bit default  alphabet
			 * table in subclause 6.2.1., or the character from
			 * the National Language Locking Shift Table in the
			 * case where the locking shift mechanism as defined
			 * in subclause 6.2.1.2.3 is used."
			 */
			if (c == GUND)
				c = gsm_locking_shift_lookup(&t, text[i]);
		} else
			c = gsm_locking_shift_lookup(&t, text[i]);

		res_length += UTF8_LENGTH(c);
	}

	res = l_malloc(res_length + 1);
	out = res;

	i = 0;
	while (out < res + res_length) {
		unsigned short c;

		if (text[i] == 0x1b) {
			c = gsm_single_shift_lookup(&t, text[++i]);

			if (c == GUND)
				c = gsm_locking_shift_lookup(&t, text[i]);
		} else
			c = gsm_locking_shift_lookup(&t, text[i]);

		out += l_utf8_from_wchar(c, out);

		++i;
	}

	*out = '\0';

	if (items_written)
		*items_written = out - res;

error:
	if (items_read)
		*items_read = i;

	return res;
}

char *convert_gsm_to_utf8(const unsigned char *text, long len,
				long *items_read, long *items_written,
				unsigned char terminator)
{
	return convert_gsm_to_utf8_with_lang(text, len, items_read,
						items_written,
						terminator,
						GSM_DIALECT_DEFAULT,
						GSM_DIALECT_DEFAULT);
}

/*!
 * Converts UTF-8 encoded text to GSM alphabet.  The result is unpacked,
 * with the 7th bit always 0.  If terminator is not 0, a terminator character
 * is appended to the result.  This should be in the range 0x80-0xf0
 *
 * Returns the encoded data or NULL if the data could not be encoded.  The
 * data must be freed by the caller.  If items_read is not NULL, it contains
 * the actual number of bytes read.  If items_written is not NULL, contains
 * the number of bytes written.
 */
unsigned char *convert_utf8_to_gsm_with_lang(const char *text, long len,
					long *items_read, long *items_written,
					unsigned char terminator,
					enum gsm_dialect locking_lang,
					enum gsm_dialect single_lang)
{
	struct conversion_table t;
	long nchars = 0;
	const char *in;
	unsigned char *out;
	unsigned char *res = NULL;
	long res_len;
	long i;

	if (!conversion_table_init(&t, locking_lang, single_lang))
		return NULL;

	in = text;
	res_len = 0;

	while ((len < 0 || text + len - in > 0) && *in) {
		long max = len < 0 ? 4 : text + len - in;
		wchar_t c;
		unsigned short converted;
		int nread = l_utf8_get_codepoint(in, max, &c);

		if (nread < 0)
			goto err_out;

		if (c > 0xffff)
			goto err_out;

		converted = unicode_locking_shift_lookup(&t, c);
		if (converted == GUND)
			converted = unicode_single_shift_lookup(&t, c);

		if (converted == GUND)
			goto err_out;

		if (converted & 0x1b00)
			res_len += 2;
		else
			res_len += 1;

		in += nread;
		nchars += 1;
	}

	res = l_malloc(res_len + (terminator ? 1 : 0));
	in = text;
	out = res;

	for (i = 0; i < nchars; i++) {
		wchar_t c;
		unsigned short converted;
		int nread = l_utf8_get_codepoint(in, 4, &c);

		converted = unicode_locking_shift_lookup(&t, c);
		if (converted == GUND)
			converted = unicode_single_shift_lookup(&t, c);

		if (converted & 0x1b00) {
			*out = 0x1b;
			++out;
		}

		*out = converted;
		++out;
		in += nread;
	}

	if (terminator)
		*out = terminator;

	if (items_written)
		*items_written = out - res;

err_out:
	if (items_read)
		*items_read = in - text;

	return res;
}

unsigned char *convert_utf8_to_gsm(const char *text, long len,
					long *items_read, long *items_written,
					unsigned char terminator)
{
	return convert_utf8_to_gsm_with_lang(text, len, items_read,
						items_written,
						terminator,
						GSM_DIALECT_DEFAULT,
						GSM_DIALECT_DEFAULT);
}

/*!
 * Converts UTF-8 encoded text to GSM alphabet. It finds an encoding
 * that uses the minimum set of GSM dialects based on the hint given.
 *
 * It first attempts to use the default dialect's single shift and
 * locking shift tables. It then tries with only the single shift
 * table of the hinted dialect, and finally with both the single shift
 * and locking shift tables of the hinted dialect.
 *
 * Returns the encoded data or NULL if no suitable encoding could be
 * found. The data must be freed by the caller. If items_read is not
 * NULL, it contains the actual number of bytes read. If items_written
 * is not NULL, it contains the number of bytes written. If
 * used_locking and used_single are not NULL, they will contain the
 * dialects used for the locking shift and single shift tables.
 */
unsigned char *convert_utf8_to_gsm_best_lang(const char *utf8, long len,
					long *items_read, long *items_written,
					unsigned char terminator,
					enum gsm_dialect hint,
					enum gsm_dialect *used_locking,
					enum gsm_dialect *used_single)
{
	enum gsm_dialect locking = GSM_DIALECT_DEFAULT;
	enum gsm_dialect single = GSM_DIALECT_DEFAULT;
	unsigned char *encoded;

	encoded = convert_utf8_to_gsm_with_lang(utf8, len, items_read,
						items_written, terminator,
						locking, single);
	if (encoded != NULL)
		goto out;

	if (hint == GSM_DIALECT_DEFAULT)
		return NULL;

	single = hint;
	encoded = convert_utf8_to_gsm_with_lang(utf8, len, items_read,
						items_written, terminator,
						locking, single);
	if (encoded != NULL)
		goto out;

	/* Spanish dialect uses the default locking shift table */
	if (hint == GSM_DIALECT_SPANISH)
		return NULL;

	locking = hint;
	encoded = convert_utf8_to_gsm_with_lang(utf8, len, items_read,
						items_written, terminator,
						locking, single);

	if (encoded == NULL)
		return NULL;

out:
	if (used_locking != NULL)
		*used_locking = locking;

	if (used_single != NULL)
		*used_single = single;

	return encoded;
}

/*!
 * Decodes the hex encoded data and converts to a byte array.  If terminator
 * is not 0, the terminator character is appended to the end of the result.
 * This might be useful for converting GSM encoded data if the CSCS is set
 * to HEX.
 *
 * Please note that this since GSM does allow embedded null characeters, use
 * of the terminator or the items_writen is encouraged to find the real size
 * of the result.
 */
unsigned char *decode_hex_own_buf(const char *in, long len, long *items_written,
					unsigned char terminator,
					unsigned char *buf)
{
	long i, j;
	char c;
	unsigned char b;

	if (!buf)
		return NULL;

	if (len < 0)
		len = strlen(in);

	len &= ~0x1;

	for (i = 0, j = 0; i < len; i++, j++) {
		c = toupper(in[i]);

		if (c >= '0' && c <= '9')
			b = c - '0';
		else if (c >= 'A' && c <= 'F')
			b = 10 + c - 'A';
		else
			return NULL;

		i += 1;

		c = toupper(in[i]);

		if (c >= '0' && c <= '9')
			b = b * 16 + c - '0';
		else if (c >= 'A' && c <= 'F')
			b = b * 16 + 10 + c - 'A';
		else
			return NULL;

		buf[j] = b;
	}

	if (terminator)
		buf[j] = terminator;

	if (items_written)
		*items_written = j;

	return buf;
}

/*!
 * Encodes the data using hexadecimal characters.  len can be negative,
 * in that case the terminator is used to find the last character.  This is
 * useful for handling GSM-encoded strings which allow ASCII NULL character
 * in the stream.
 */
char *encode_hex_own_buf(const unsigned char *in, long len,
				unsigned char terminator, char *buf)
{
	long i, j;
	char c;

	if (len < 0) {
		i = 0;

		while (in[i] != terminator)
			i++;

		len = i;
	}

	for (i = 0, j = 0; i < len; i++, j++) {
		c = (in[i] >> 4) & 0xf;

		if (c <= 9)
			buf[j] = '0' + c;
		else
			buf[j] = 'A' + c - 10;

		j += 1;

		c = (in[i]) & 0xf;

		if (c <= 9)
			buf[j] = '0' + c;
		else
			buf[j] = 'A' + c - 10;
	}

	buf[j] = '\0';

	return buf;
}

unsigned char *unpack_7bit_own_buf(const unsigned char *in, long len,
					int byte_offset, bool ussd,
					long max_to_unpack, long *items_written,
					unsigned char terminator,
					unsigned char *buf)
{
	unsigned char rest = 0;
	unsigned char *out = buf;
	int bits = 7 - (byte_offset % 7);
	long i;

	if (len <= 0)
		return NULL;

	/* In the case of CB, unpack as much as possible */
	if (ussd)
		max_to_unpack = len * 8 / 7;

	for (i = 0; (i < len) && ((out-buf) < max_to_unpack); i++) {
		/* Grab what we have in the current octet */
		*out = (in[i] & ((1 << bits) - 1)) << (7 - bits);

		/* Append what we have from the previous octet, if any */
		*out |= rest;

		/* Figure out the remainder */
		rest = (in[i] >> bits) & ((1 << (8-bits)) - 1);

		/*
		 * We have the entire character, here we don't increate
		 * out if this is we started at an offset.  Instead
		 * we effectively populate variable rest
		 */
		if (i != 0 || bits == 7)
			out++;

		if ((out-buf) == max_to_unpack)
			break;

		/*
		 * We expected only 1 bit from this octet, means there's 7
		 * left, take care of them here
		 */
		if (bits == 1) {
			*out = rest;
			out++;
			bits = 7;
			rest = 0;
		} else {
			bits = bits - 1;
		}
	}

	/*
	 * According to 23.038 6.1.2.3.1, last paragraph:
	 * "If the total number of characters to be sent equals (8n-1)
	 * where n=1,2,3 etc. then there are 7 spare bits at the end
	 * of the message. To avoid the situation where the receiving
	 * entity confuses 7 binary zero pad bits as the @ character,
	 * the carriage return or <CR> character shall be used for
	 * padding in this situation, just as for Cell Broadcast."
	 *
	 * "The receiving entity shall remove the final <CR> character where
	 * the message ends on an octet boundary with <CR> as the last
	 * character.
	 */
	if (ussd && (((out - buf) % 8) == 0) && (*(out - 1) == '\r'))
		out = out - 1;

	if (terminator)
		*out = terminator;

	if (items_written)
		*items_written = out - buf;

	return buf;
}

unsigned char *unpack_7bit(const unsigned char *in, long len, int byte_offset,
				bool ussd, long max_to_unpack,
				long *items_written, unsigned char terminator)
{
	unsigned char *buf = l_malloc(len * 8 / 7 + (terminator ? 1 : 0));

	return unpack_7bit_own_buf(in, len, byte_offset, ussd, max_to_unpack,
				items_written, terminator, buf);
}

unsigned char *pack_7bit_own_buf(const unsigned char *in, long len,
					int byte_offset, bool ussd,
					long *items_written,
					unsigned char terminator,
					unsigned char *buf)
{
	int bits = 7 - (byte_offset % 7);
	unsigned char *out = buf;
	long i;
	long total_bits;

	if (len == 0)
		return NULL;

	if (len < 0) {
		i = 0;

		while (in[i] != terminator)
			i++;

		len = i;
	}

	total_bits = len * 7;

	if (bits != 7) {
		total_bits += bits;
		bits = bits - 1;
		*out = 0;
	}

	for (i = 0; i < len; i++) {
		if (bits != 7) {
			*out |= (in[i] & ((1 << (7 - bits)) - 1)) <<
					(bits + 1);
			out++;
		}

		/* This is a no op when bits == 0, lets keep valgrind happy */
		if (bits != 0)
			*out = in[i] >> (7 - bits);

		if (bits == 0)
			bits = 7;
		else
			bits = bits - 1;
	}

	/*
	 * If <CR> is intended to be the last character and the message
	 * (including the wanted <CR>) ends on an octet boundary, then
	 * another <CR> must be added together with a padding bit 0. The
	 * receiving entity will perform the carriage return function twice,
	 * but this will not result in misoperation as the definition of
	 * <CR> in clause 6.1.1 is identical to the definition of <CR><CR>.
	 */
	if (ussd && ((total_bits % 8) == 1))
		*out |= '\r' << 1;

	if (bits != 7)
		out++;

	if (ussd && ((total_bits % 8) == 0) && (in[len - 1] == '\r')) {
		*out = '\r';
		out++;
	}

	if (items_written)
		*items_written = out - buf;

	return buf;
}

unsigned char *pack_7bit(const unsigned char *in, long len, int byte_offset,
				bool ussd, long *items_written,
				unsigned char terminator)
{
	int bits = 7 - (byte_offset % 7);
	long i;
	long total_bits;
	unsigned char *buf;

	if (len == 0 || items_written == NULL)
		return NULL;

	if (len < 0) {
		i = 0;

		while (in[i] != terminator)
			i++;

		len = i;
	}

	total_bits = len * 7;

	if (bits != 7)
		total_bits += bits;

	/* Round up number of bytes, must append <cr> if true */
	if (ussd && ((total_bits % 8) == 0) && (in[len - 1] == '\r'))
		buf = l_malloc((total_bits + 14) / 8);
	else
		buf = l_malloc((total_bits + 7) / 8);

	return pack_7bit_own_buf(in, len, byte_offset, ussd, items_written,
					terminator, buf);
}

char *sim_string_to_utf8(const unsigned char *buffer, int length)
{
	struct conversion_table t;
	int i;
	int j;
	int num_chars;
	unsigned short ucs2_offset;
	int res_len;
	int offset;
	char *utf8 = NULL;
	char *out;

	if (!conversion_table_init(&t, GSM_DIALECT_DEFAULT,
					GSM_DIALECT_DEFAULT))
		return NULL;

	if (length < 1)
		return NULL;

	if (buffer[0] < 0x80) {
		/*
		 * We have to find the real length, since on SIM file system
		 * alpha fields are 0xff padded
		 */
		for (i = 0; i < length; i++)
			if (buffer[i] == 0xff)
				break;

		return convert_gsm_to_utf8(buffer, i, NULL, NULL, 0);
	}

	switch (buffer[0]) {
	case 0x80:
		if (((length - 1) % 2) == 1) {
			if (buffer[length - 1] != 0xff)
				return NULL;

			length = length - 1;
		}

		for (i = 1; i < length; i += 2)
			if (buffer[i] == 0xff && buffer[i + 1] == 0xff)
				break;

		return l_utf8_from_ucs2be(buffer + 1, i - 1);
	case 0x81:
		if (length < 3 || (buffer[1] > (length - 3)))
			return NULL;

		num_chars = buffer[1];
		ucs2_offset = buffer[2] << 7;
		offset = 3;
		break;

	case 0x82:
		if (length < 4 || buffer[1] > length - 4)
			return NULL;

		num_chars = buffer[1];
		ucs2_offset = (buffer[2] << 8) | buffer[3];
		offset = 4;
		break;

	case 0xff: /* Special case of empty string */
		num_chars = 0;
		ucs2_offset = 0;
		offset = 0;
		break;

	default:
		return NULL;
	}

	res_len = 0;
	i = offset;
	j = 0;

	while ((i < length) && (j < num_chars)) {
		unsigned short c;

		if (buffer[i] & 0x80) {
			c = (buffer[i++] & 0x7f) + ucs2_offset;

			if (c >= 0xd800 && c < 0xe000)
				return NULL;

			res_len += UTF8_LENGTH(c);
			j += 1;
			continue;
		}

		if (buffer[i] == 0x1b) {
			++i;
			if (i >= length)
				return NULL;

			c = gsm_single_shift_lookup(&t, buffer[i++]);

			if (c == 0)
				return NULL;

			j += 2;
		} else {
			c = gsm_locking_shift_lookup(&t, buffer[i++]);
			j += 1;
		}

		res_len += UTF8_LENGTH(c);
	}

	if (j != num_chars)
		return NULL;

	/* Check that the string is padded out to the length by 0xff */
	for (; i < length; i++)
		if (buffer[i] != 0xff)
			return NULL;

	utf8 = l_malloc(res_len + 1);
	i = offset;
	out = utf8;

	while (out < utf8 + res_len) {
		unsigned short c;

		if (buffer[i] & 0x80)
			c = (buffer[i++] & 0x7f) + ucs2_offset;
		else if (buffer[i] == 0x1b) {
			++i;
			c = gsm_single_shift_lookup(&t, buffer[i++]);
		} else
			c = gsm_locking_shift_lookup(&t, buffer[i++]);

		out += l_utf8_from_wchar(c, out);
	}

	*out = '\0';

	return utf8;
}

unsigned char *utf8_to_sim_string(const char *utf, int max_length,
					int *out_length)
{
	unsigned char *result;
	void *ucs2;
	long gsm_bytes;
	size_t converted;

	result = convert_utf8_to_gsm(utf, -1, NULL, &gsm_bytes, 0);
	if (result) {
		if (gsm_bytes > max_length) {
			gsm_bytes = max_length;
			while (gsm_bytes && result[gsm_bytes - 1] == 0x1b)
				gsm_bytes -= 1;
		}

		*out_length = gsm_bytes;
		return result;
	}

	/* NOTE: UCS2 formats with an offset are never used */

	ucs2 = l_utf8_to_ucs2be(utf, &converted);
	if (!ucs2)
		return NULL;

	/* converted includes null terminator, drop */
	converted -= 2;

	if (max_length != -1 && (int) converted + 1 > max_length)
		converted = (max_length - 1) & ~1;

	result = l_malloc(converted + 1);
	result[0] = 0x80;
	memcpy(&result[1], ucs2, converted);
	*out_length = converted + 1;
	l_free(ucs2);

	return result;
}

/*!
 * Converts UCS2 encoded text to GSM alphabet. The result is unpacked,
 * with the 7th bit always 0. If terminator is not 0, a terminator character
 * is appended to the result.
 *
 * Returns the encoded data or NULL if the data could not be encoded. The
 * data must be freed by the caller. If items_read is not NULL, it contains
 * the actual number of bytes read. If items_written is not NULL, contains
 * the number of bytes written.
 */
unsigned char *convert_ucs2_to_gsm_with_lang(const unsigned char *text,
					long len, long *items_read,
					long *items_written,
					unsigned char terminator,
					enum gsm_dialect locking_lang,
					enum gsm_dialect single_lang)
{
	struct conversion_table t;
	const unsigned char *in;
	unsigned char *out;
	unsigned char *res = NULL;
	long res_len;
	long i;

	if (!conversion_table_init(&t, locking_lang, single_lang))
		return NULL;

	if (len < 1 || len % 2)
		return NULL;

	in = text;
	res_len = 0;

	for (i = 0; i < len; i += 2) {
		uint16_t c = l_get_be16(in + i);
		uint16_t converted = GUND;

		converted = unicode_locking_shift_lookup(&t, c);

		if (converted == GUND)
			converted = unicode_single_shift_lookup(&t, c);

		if (converted == GUND)
			goto err_out;

		if (converted & 0x1b00)
			res_len += 2;
		else
			res_len += 1;
	}

	res = l_malloc(res_len + (terminator ? 1 : 0));

	in = text;
	out = res;

	for (i = 0; i < len; i += 2) {
		uint16_t c = l_get_be16(in + i);
		uint16_t converted = GUND;

		converted = unicode_locking_shift_lookup(&t, c);

		if (converted == GUND)
			converted = unicode_single_shift_lookup(&t, c);

		if (converted & 0x1b00) {
			*out = 0x1b;
			++out;
		}

		*out = converted;
		++out;
	}

	if (terminator)
		*out = terminator;

	if (items_written)
		*items_written = out - res;

err_out:
	if (items_read)
		*items_read = i;

	return res;
}

unsigned char *convert_ucs2_to_gsm(const unsigned char *text, long len,
				long *items_read, long *items_written,
				unsigned char terminator)
{
	return convert_ucs2_to_gsm_with_lang(text, len, items_read,
						items_written,
						terminator,
						GSM_DIALECT_DEFAULT,
						GSM_DIALECT_DEFAULT);
}

bool iso639_2_from_language(enum cbs_language lang, char *iso639)
{
	switch (lang) {
	case CBS_LANGUAGE_GERMAN:
		iso639[0] = 'd';
		iso639[1] = 'e';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_ENGLISH:
		iso639[0] = 'e';
		iso639[1] = 'n';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_ITALIAN:
		iso639[0] = 'i';
		iso639[1] = 't';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_FRENCH:
		iso639[0] = 'f';
		iso639[1] = 'r';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_SPANISH:
		iso639[0] = 'e';
		iso639[1] = 's';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_DUTCH:
		iso639[0] = 'n';
		iso639[1] = 'l';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_SWEDISH:
		iso639[0] = 's';
		iso639[1] = 'v';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_DANISH:
		iso639[0] = 'd';
		iso639[1] = 'a';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_PORTUGESE:
		iso639[0] = 'p';
		iso639[1] = 't';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_FINNISH:
		iso639[0] = 'f';
		iso639[1] = 'i';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_NORWEGIAN:
		iso639[0] = 'n';
		iso639[1] = 'o';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_GREEK:
		iso639[0] = 'e';
		iso639[1] = 'l';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_TURKISH:
		iso639[0] = 't';
		iso639[1] = 'r';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_HUNGARIAN:
		iso639[0] = 'h';
		iso639[1] = 'u';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_POLISH:
		iso639[0] = 'p';
		iso639[1] = 'l';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_CZECH:
		iso639[0] = 'c';
		iso639[1] = 's';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_HEBREW:
		iso639[0] = 'h';
		iso639[1] = 'e';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_ARABIC:
		iso639[0] = 'a';
		iso639[1] = 'r';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_RUSSIAN:
		iso639[0] = 'r';
		iso639[1] = 'u';
		iso639[2] = '\0';
		return true;
	case CBS_LANGUAGE_ICELANDIC:
		iso639[0] = 'i';
		iso639[1] = 's';
		iso639[2] = '\0';
		return true;
	default:
		iso639[0] = '\0';
		break;
	}

	return false;
}
