#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from nova.notifications.objects import base
from nova.objects import base as nova_base
from nova.objects import fields


@nova_base.NovaObjectRegistry.register_notification
class AggregatePayload(base.NotificationPayloadBase):
    SCHEMA = {
        'id': ('aggregate', 'id'),
        'uuid': ('aggregate', 'uuid'),
        'name': ('aggregate', 'name'),
        'hosts': ('aggregate', 'hosts'),
        'metadata': ('aggregate', 'metadata'),
    }
    # Version 1.0: Initial version
    #         1.1: Making the id field nullable
    VERSION = '1.1'
    fields = {
        # NOTE(gibi): id is nullable as aggregate.create.start is sent before
        # the id is generated by the db
        'id': fields.IntegerField(nullable=True),
        'uuid': fields.UUIDField(nullable=False),
        'name': fields.StringField(),
        'hosts': fields.ListOfStringsField(nullable=True),
        'metadata': fields.DictOfStringsField(nullable=True),
    }

    def __init__(self, aggregate):
        super(AggregatePayload, self).__init__()
        self.populate_schema(aggregate=aggregate)


@base.notification_sample('aggregate-create-start.json')
@base.notification_sample('aggregate-create-end.json')
@base.notification_sample('aggregate-delete-start.json')
@base.notification_sample('aggregate-delete-end.json')
@base.notification_sample('aggregate-add_host-start.json')
@base.notification_sample('aggregate-add_host-end.json')
@base.notification_sample('aggregate-remove_host-start.json')
@base.notification_sample('aggregate-remove_host-end.json')
@base.notification_sample('aggregate-update_metadata-start.json')
@base.notification_sample('aggregate-update_metadata-end.json')
@base.notification_sample('aggregate-update_prop-start.json')
@base.notification_sample('aggregate-update_prop-end.json')
@base.notification_sample('aggregate-cache_images-start.json')
@base.notification_sample('aggregate-cache_images-end.json')
@nova_base.NovaObjectRegistry.register_notification
class AggregateNotification(base.NotificationBase):
    # Version 1.0: Initial version
    VERSION = '1.0'

    fields = {
        'payload': fields.ObjectField('AggregatePayload')
    }


@nova_base.NovaObjectRegistry.register_notification
class AggregateCachePayload(base.NotificationPayloadBase):
    SCHEMA = {
        'id': ('aggregate', 'id'),
        'uuid': ('aggregate', 'uuid'),
        'name': ('aggregate', 'name'),
    }

    # Version 1.0: Initial version
    VERSION = '1.0'

    fields = {
        'id': fields.IntegerField(),
        'uuid': fields.UUIDField(),
        'name': fields.StringField(),

        # The host that we just worked
        'host': fields.StringField(),

        # The images that were downloaded or are already there
        'images_cached': fields.ListOfStringsField(),

        # The images that are unable to be cached for some reason
        'images_failed': fields.ListOfStringsField(),

        # The N/M progress information for this operation
        'index': fields.IntegerField(),
        'total': fields.IntegerField(),
    }

    def __init__(self, aggregate, host, index, total):
        super(AggregateCachePayload, self).__init__()
        self.populate_schema(aggregate=aggregate)
        self.host = host
        self.index = index
        self.total = total


@base.notification_sample('aggregate-cache_images-progress.json')
@nova_base.NovaObjectRegistry.register_notification
class AggregateCacheNotification(base.NotificationBase):
    # Version 1.0: Initial version
    VERSION = '1.0'

    fields = {
        'payload': fields.ObjectField('AggregateCachePayload'),
    }
