package mahonia

// Data tables for 4-byte characters in GB18030 encoding.
// Based on http://source.icu-project.org/repos/icu/data/trunk/charset/data/ucm/gb-18030-2005.ucm

// gb18030Linear converts a 32-bit big-endian representation of a 4-byte
// character into a linearly-increasing integer, starting from the base
// sequence of 0x81308130
func gb18030Linear(g uint32) uint32 {
	lin := g>>24 - 0x81
	lin = lin*10 + (g>>16)&0xff - 0x30
	lin = lin*126 + (g>>8)&0xff - 0x81
	lin = lin*10 + g&0xff - 0x30
	return lin
}

// Equivalent ranges between GB18030 and Unicode.
var gb18030Ranges = []struct {
	firstRune, lastRune rune
	firstGB, lastGB     uint32
}{
	{0x10000, 0x10FFFF, gb18030Linear(0x90308130), gb18030Linear(0xE3329A35)},
	{0x9FA6, 0xD7FF, gb18030Linear(0x82358F33), gb18030Linear(0x8336C738)},
	{0x0452, 0x200F, gb18030Linear(0x8130D330), gb18030Linear(0x8136A531)},
	{0xE865, 0xF92B, gb18030Linear(0x8336D030), gb18030Linear(0x84308534)},
	{0x2643, 0x2E80, gb18030Linear(0x8137A839), gb18030Linear(0x8138FD38)},
	{0xFA2A, 0xFE2F, gb18030Linear(0x84309C38), gb18030Linear(0x84318537)},
	{0x3CE1, 0x4055, gb18030Linear(0x8231D438), gb18030Linear(0x8232AF32)},
	{0x361B, 0x3917, gb18030Linear(0x8230A633), gb18030Linear(0x8230F237)},
	{0x49B8, 0x4C76, gb18030Linear(0x8234A131), gb18030Linear(0x8234E733)},
	{0x4160, 0x4336, gb18030Linear(0x8232C937), gb18030Linear(0x8232F837)},
	{0x478E, 0x4946, gb18030Linear(0x8233E838), gb18030Linear(0x82349638)},
	{0x44D7, 0x464B, gb18030Linear(0x8233A339), gb18030Linear(0x8233C931)},
	{0xFFE6, 0xFFFF, gb18030Linear(0x8431A234), gb18030Linear(0x8431A439)},
}

// The higest value returned by gb18030Linear for characters in gb18030Data
const maxGB18030Linear = 39393

// Unicode equivalents for characters not handled algorithmically.
var gb18030Data = []struct {
	unicode uint16
	gb18030 uint32
}{
	{0x0080, 0x81308130},
	{0x0081, 0x81308131},
	{0x0082, 0x81308132},
	{0x0083, 0x81308133},
	{0x0084, 0x81308134},
	{0x0085, 0x81308135},
	{0x0086, 0x81308136},
	{0x0087, 0x81308137},
	{0x0088, 0x81308138},
	{0x0089, 0x81308139},
	{0x008A, 0x81308230},
	{0x008B, 0x81308231},
	{0x008C, 0x81308232},
	{0x008D, 0x81308233},
	{0x008E, 0x81308234},
	{0x008F, 0x81308235},
	{0x0090, 0x81308236},
	{0x0091, 0x81308237},
	{0x0092, 0x81308238},
	{0x0093, 0x81308239},
	{0x0094, 0x81308330},
	{0x0095, 0x81308331},
	{0x0096, 0x81308332},
	{0x0097, 0x81308333},
	{0x0098, 0x81308334},
	{0x0099, 0x81308335},
	{0x009A, 0x81308336},
	{0x009B, 0x81308337},
	{0x009C, 0x81308338},
	{0x009D, 0x81308339},
	{0x009E, 0x81308430},
	{0x009F, 0x81308431},
	{0x00A0, 0x81308432},
	{0x00A1, 0x81308433},
	{0x00A2, 0x81308434},
	{0x00A3, 0x81308435},
	{0x00A5, 0x81308436},
	{0x00A6, 0x81308437},
	{0x00A9, 0x81308438},
	{0x00AA, 0x81308439},
	{0x00AB, 0x81308530},
	{0x00AC, 0x81308531},
	{0x00AD, 0x81308532},
	{0x00AE, 0x81308533},
	{0x00AF, 0x81308534},
	{0x00B2, 0x81308535},
	{0x00B3, 0x81308536},
	{0x00B4, 0x81308537},
	{0x00B5, 0x81308538},
	{0x00B6, 0x81308539},
	{0x00B8, 0x81308630},
	{0x00B9, 0x81308631},
	{0x00BA, 0x81308632},
	{0x00BB, 0x81308633},
	{0x00BC, 0x81308634},
	{0x00BD, 0x81308635},
	{0x00BE, 0x81308636},
	{0x00BF, 0x81308637},
	{0x00C0, 0x81308638},
	{0x00C1, 0x81308639},
	{0x00C2, 0x81308730},
	{0x00C3, 0x81308731},
	{0x00C4, 0x81308732},
	{0x00C5, 0x81308733},
	{0x00C6, 0x81308734},
	{0x00C7, 0x81308735},
	{0x00C8, 0x81308736},
	{0x00C9, 0x81308737},
	{0x00CA, 0x81308738},
	{0x00CB, 0x81308739},
	{0x00CC, 0x81308830},
	{0x00CD, 0x81308831},
	{0x00CE, 0x81308832},
	{0x00CF, 0x81308833},
	{0x00D0, 0x81308834},
	{0x00D1, 0x81308835},
	{0x00D2, 0x81308836},
	{0x00D3, 0x81308837},
	{0x00D4, 0x81308838},
	{0x00D5, 0x81308839},
	{0x00D6, 0x81308930},
	{0x00D8, 0x81308931},
	{0x00D9, 0x81308932},
	{0x00DA, 0x81308933},
	{0x00DB, 0x81308934},
	{0x00DC, 0x81308935},
	{0x00DD, 0x81308936},
	{0x00DE, 0x81308937},
	{0x00DF, 0x81308938},
	{0x00E2, 0x81308939},
	{0x00E3, 0x81308A30},
	{0x00E4, 0x81308A31},
	{0x00E5, 0x81308A32},
	{0x00E6, 0x81308A33},
	{0x00E7, 0x81308A34},
	{0x00EB, 0x81308A35},
	{0x00EE, 0x81308A36},
	{0x00EF, 0x81308A37},
	{0x00F0, 0x81308A38},
	{0x00F1, 0x81308A39},
	{0x00F4, 0x81308B30},
	{0x00F5, 0x81308B31},
	{0x00F6, 0x81308B32},
	{0x00F8, 0x81308B33},
	{0x00FB, 0x81308B34},
	{0x00FD, 0x81308B35},
	{0x00FE, 0x81308B36},
	{0x00FF, 0x81308B37},
	{0x0100, 0x81308B38},
	{0x0102, 0x81308B39},
	{0x0103, 0x81308C30},
	{0x0104, 0x81308C31},
	{0x0105, 0x81308C32},
	{0x0106, 0x81308C33},
	{0x0107, 0x81308C34},
	{0x0108, 0x81308C35},
	{0x0109, 0x81308C36},
	{0x010A, 0x81308C37},
	{0x010B, 0x81308C38},
	{0x010C, 0x81308C39},
	{0x010D, 0x81308D30},
	{0x010E, 0x81308D31},
	{0x010F, 0x81308D32},
	{0x0110, 0x81308D33},
	{0x0111, 0x81308D34},
	{0x0112, 0x81308D35},
	{0x0114, 0x81308D36},
	{0x0115, 0x81308D37},
	{0x0116, 0x81308D38},
	{0x0117, 0x81308D39},
	{0x0118, 0x81308E30},
	{0x0119, 0x81308E31},
	{0x011A, 0x81308E32},
	{0x011C, 0x81308E33},
	{0x011D, 0x81308E34},
	{0x011E, 0x81308E35},
	{0x011F, 0x81308E36},
	{0x0120, 0x81308E37},
	{0x0121, 0x81308E38},
	{0x0122, 0x81308E39},
	{0x0123, 0x81308F30},
	{0x0124, 0x81308F31},
	{0x0125, 0x81308F32},
	{0x0126, 0x81308F33},
	{0x0127, 0x81308F34},
	{0x0128, 0x81308F35},
	{0x0129, 0x81308F36},
	{0x012A, 0x81308F37},
	{0x012C, 0x81308F38},
	{0x012D, 0x81308F39},
	{0x012E, 0x81309030},
	{0x012F, 0x81309031},
	{0x0130, 0x81309032},
	{0x0131, 0x81309033},
	{0x0132, 0x81309034},
	{0x0133, 0x81309035},
	{0x0134, 0x81309036},
	{0x0135, 0x81309037},
	{0x0136, 0x81309038},
	{0x0137, 0x81309039},
	{0x0138, 0x81309130},
	{0x0139, 0x81309131},
	{0x013A, 0x81309132},
	{0x013B, 0x81309133},
	{0x013C, 0x81309134},
	{0x013D, 0x81309135},
	{0x013E, 0x81309136},
	{0x013F, 0x81309137},
	{0x0140, 0x81309138},
	{0x0141, 0x81309139},
	{0x0142, 0x81309230},
	{0x0143, 0x81309231},
	{0x0145, 0x81309232},
	{0x0146, 0x81309233},
	{0x0147, 0x81309234},
	{0x0149, 0x81309235},
	{0x014A, 0x81309236},
	{0x014B, 0x81309237},
	{0x014C, 0x81309238},
	{0x014E, 0x81309239},
	{0x014F, 0x81309330},
	{0x0150, 0x81309331},
	{0x0151, 0x81309332},
	{0x0152, 0x81309333},
	{0x0153, 0x81309334},
	{0x0154, 0x81309335},
	{0x0155, 0x81309336},
	{0x0156, 0x81309337},
	{0x0157, 0x81309338},
	{0x0158, 0x81309339},
	{0x0159, 0x81309430},
	{0x015A, 0x81309431},
	{0x015B, 0x81309432},
	{0x015C, 0x81309433},
	{0x015D, 0x81309434},
	{0x015E, 0x81309435},
	{0x015F, 0x81309436},
	{0x0160, 0x81309437},
	{0x0161, 0x81309438},
	{0x0162, 0x81309439},
	{0x0163, 0x81309530},
	{0x0164, 0x81309531},
	{0x0165, 0x81309532},
	{0x0166, 0x81309533},
	{0x0167, 0x81309534},
	{0x0168, 0x81309535},
	{0x0169, 0x81309536},
	{0x016A, 0x81309537},
	{0x016C, 0x81309538},
	{0x016D, 0x81309539},
	{0x016E, 0x81309630},
	{0x016F, 0x81309631},
	{0x0170, 0x81309632},
	{0x0171, 0x81309633},
	{0x0172, 0x81309634},
	{0x0173, 0x81309635},
	{0x0174, 0x81309636},
	{0x0175, 0x81309637},
	{0x0176, 0x81309638},
	{0x0177, 0x81309639},
	{0x0178, 0x81309730},
	{0x0179, 0x81309731},
	{0x017A, 0x81309732},
	{0x017B, 0x81309733},
	{0x017C, 0x81309734},
	{0x017D, 0x81309735},
	{0x017E, 0x81309736},
	{0x017F, 0x81309737},
	{0x0180, 0x81309738},
	{0x0181, 0x81309739},
	{0x0182, 0x81309830},
	{0x0183, 0x81309831},
	{0x0184, 0x81309832},
	{0x0185, 0x81309833},
	{0x0186, 0x81309834},
	{0x0187, 0x81309835},
	{0x0188, 0x81309836},
	{0x0189, 0x81309837},
	{0x018A, 0x81309838},
	{0x018B, 0x81309839},
	{0x018C, 0x81309930},
	{0x018D, 0x81309931},
	{0x018E, 0x81309932},
	{0x018F, 0x81309933},
	{0x0190, 0x81309934},
	{0x0191, 0x81309935},
	{0x0192, 0x81309936},
	{0x0193, 0x81309937},
	{0x0194, 0x81309938},
	{0x0195, 0x81309939},
	{0x0196, 0x81309A30},
	{0x0197, 0x81309A31},
	{0x0198, 0x81309A32},
	{0x0199, 0x81309A33},
	{0x019A, 0x81309A34},
	{0x019B, 0x81309A35},
	{0x019C, 0x81309A36},
	{0x019D, 0x81309A37},
	{0x019E, 0x81309A38},
	{0x019F, 0x81309A39},
	{0x01A0, 0x81309B30},
	{0x01A1, 0x81309B31},
	{0x01A2, 0x81309B32},
	{0x01A3, 0x81309B33},
	{0x01A4, 0x81309B34},
	{0x01A5, 0x81309B35},
	{0x01A6, 0x81309B36},
	{0x01A7, 0x81309B37},
	{0x01A8, 0x81309B38},
	{0x01A9, 0x81309B39},
	{0x01AA, 0x81309C30},
	{0x01AB, 0x81309C31},
	{0x01AC, 0x81309C32},
	{0x01AD, 0x81309C33},
	{0x01AE, 0x81309C34},
	{0x01AF, 0x81309C35},
	{0x01B0, 0x81309C36},
	{0x01B1, 0x81309C37},
	{0x01B2, 0x81309C38},
	{0x01B3, 0x81309C39},
	{0x01B4, 0x81309D30},
	{0x01B5, 0x81309D31},
	{0x01B6, 0x81309D32},
	{0x01B7, 0x81309D33},
	{0x01B8, 0x81309D34},
	{0x01B9, 0x81309D35},
	{0x01BA, 0x81309D36},
	{0x01BB, 0x81309D37},
	{0x01BC, 0x81309D38},
	{0x01BD, 0x81309D39},
	{0x01BE, 0x81309E30},
	{0x01BF, 0x81309E31},
	{0x01C0, 0x81309E32},
	{0x01C1, 0x81309E33},
	{0x01C2, 0x81309E34},
	{0x01C3, 0x81309E35},
	{0x01C4, 0x81309E36},
	{0x01C5, 0x81309E37},
	{0x01C6, 0x81309E38},
	{0x01C7, 0x81309E39},
	{0x01C8, 0x81309F30},
	{0x01C9, 0x81309F31},
	{0x01CA, 0x81309F32},
	{0x01CB, 0x81309F33},
	{0x01CC, 0x81309F34},
	{0x01CD, 0x81309F35},
	{0x01CF, 0x81309F36},
	{0x01D1, 0x81309F37},
	{0x01D3, 0x81309F38},
	{0x01D5, 0x81309F39},
	{0x01D7, 0x8130A030},
	{0x01D9, 0x8130A031},
	{0x01DB, 0x8130A032},
	{0x01DD, 0x8130A033},
	{0x01DE, 0x8130A034},
	{0x01DF, 0x8130A035},
	{0x01E0, 0x8130A036},
	{0x01E1, 0x8130A037},
	{0x01E2, 0x8130A038},
	{0x01E3, 0x8130A039},
	{0x01E4, 0x8130A130},
	{0x01E5, 0x8130A131},
	{0x01E6, 0x8130A132},
	{0x01E7, 0x8130A133},
	{0x01E8, 0x8130A134},
	{0x01E9, 0x8130A135},
	{0x01EA, 0x8130A136},
	{0x01EB, 0x8130A137},
	{0x01EC, 0x8130A138},
	{0x01ED, 0x8130A139},
	{0x01EE, 0x8130A230},
	{0x01EF, 0x8130A231},
	{0x01F0, 0x8130A232},
	{0x01F1, 0x8130A233},
	{0x01F2, 0x8130A234},
	{0x01F3, 0x8130A235},
	{0x01F4, 0x8130A236},
	{0x01F5, 0x8130A237},
	{0x01F6, 0x8130A238},
	{0x01F7, 0x8130A239},
	{0x01F8, 0x8130A330},
	{0x01FA, 0x8130A331},
	{0x01FB, 0x8130A332},
	{0x01FC, 0x8130A333},
	{0x01FD, 0x8130A334},
	{0x01FE, 0x8130A335},
	{0x01FF, 0x8130A336},
	{0x0200, 0x8130A337},
	{0x0201, 0x8130A338},
	{0x0202, 0x8130A339},
	{0x0203, 0x8130A430},
	{0x0204, 0x8130A431},
	{0x0205, 0x8130A432},
	{0x0206, 0x8130A433},
	{0x0207, 0x8130A434},
	{0x0208, 0x8130A435},
	{0x0209, 0x8130A436},
	{0x020A, 0x8130A437},
	{0x020B, 0x8130A438},
	{0x020C, 0x8130A439},
	{0x020D, 0x8130A530},
	{0x020E, 0x8130A531},
	{0x020F, 0x8130A532},
	{0x0210, 0x8130A533},
	{0x0211, 0x8130A534},
	{0x0212, 0x8130A535},
	{0x0213, 0x8130A536},
	{0x0214, 0x8130A537},
	{0x0215, 0x8130A538},
	{0x0216, 0x8130A539},
	{0x0217, 0x8130A630},
	{0x0218, 0x8130A631},
	{0x0219, 0x8130A632},
	{0x021A, 0x8130A633},
	{0x021B, 0x8130A634},
	{0x021C, 0x8130A635},
	{0x021D, 0x8130A636},
	{0x021E, 0x8130A637},
	{0x021F, 0x8130A638},
	{0x0220, 0x8130A639},
	{0x0221, 0x8130A730},
	{0x0222, 0x8130A731},
	{0x0223, 0x8130A732},
	{0x0224, 0x8130A733},
	{0x0225, 0x8130A734},
	{0x0226, 0x8130A735},
	{0x0227, 0x8130A736},
	{0x0228, 0x8130A737},
	{0x0229, 0x8130A738},
	{0x022A, 0x8130A739},
	{0x022B, 0x8130A830},
	{0x022C, 0x8130A831},
	{0x022D, 0x8130A832},
	{0x022E, 0x8130A833},
	{0x022F, 0x8130A834},
	{0x0230, 0x8130A835},
	{0x0231, 0x8130A836},
	{0x0232, 0x8130A837},
	{0x0233, 0x8130A838},
	{0x0234, 0x8130A839},
	{0x0235, 0x8130A930},
	{0x0236, 0x8130A931},
	{0x0237, 0x8130A932},
	{0x0238, 0x8130A933},
	{0x0239, 0x8130A934},
	{0x023A, 0x8130A935},
	{0x023B, 0x8130A936},
	{0x023C, 0x8130A937},
	{0x023D, 0x8130A938},
	{0x023E, 0x8130A939},
	{0x023F, 0x8130AA30},
	{0x0240, 0x8130AA31},
	{0x0241, 0x8130AA32},
	{0x0242, 0x8130AA33},
	{0x0243, 0x8130AA34},
	{0x0244, 0x8130AA35},
	{0x0245, 0x8130AA36},
	{0x0246, 0x8130AA37},
	{0x0247, 0x8130AA38},
	{0x0248, 0x8130AA39},
	{0x0249, 0x8130AB30},
	{0x024A, 0x8130AB31},
	{0x024B, 0x8130AB32},
	{0x024C, 0x8130AB33},
	{0x024D, 0x8130AB34},
	{0x024E, 0x8130AB35},
	{0x024F, 0x8130AB36},
	{0x0250, 0x8130AB37},
	{0x0252, 0x8130AB38},
	{0x0253, 0x8130AB39},
	{0x0254, 0x8130AC30},
	{0x0255, 0x8130AC31},
	{0x0256, 0x8130AC32},
	{0x0257, 0x8130AC33},
	{0x0258, 0x8130AC34},
	{0x0259, 0x8130AC35},
	{0x025A, 0x8130AC36},
	{0x025B, 0x8130AC37},
	{0x025C, 0x8130AC38},
	{0x025D, 0x8130AC39},
	{0x025E, 0x8130AD30},
	{0x025F, 0x8130AD31},
	{0x0260, 0x8130AD32},
	{0x0262, 0x8130AD33},
	{0x0263, 0x8130AD34},
	{0x0264, 0x8130AD35},
	{0x0265, 0x8130AD36},
	{0x0266, 0x8130AD37},
	{0x0267, 0x8130AD38},
	{0x0268, 0x8130AD39},
	{0x0269, 0x8130AE30},
	{0x026A, 0x8130AE31},
	{0x026B, 0x8130AE32},
	{0x026C, 0x8130AE33},
	{0x026D, 0x8130AE34},
	{0x026E, 0x8130AE35},
	{0x026F, 0x8130AE36},
	{0x0270, 0x8130AE37},
	{0x0271, 0x8130AE38},
	{0x0272, 0x8130AE39},
	{0x0273, 0x8130AF30},
	{0x0274, 0x8130AF31},
	{0x0275, 0x8130AF32},
	{0x0276, 0x8130AF33},
	{0x0277, 0x8130AF34},
	{0x0278, 0x8130AF35},
	{0x0279, 0x8130AF36},
	{0x027A, 0x8130AF37},
	{0x027B, 0x8130AF38},
	{0x027C, 0x8130AF39},
	{0x027D, 0x8130B030},
	{0x027E, 0x8130B031},
	{0x027F, 0x8130B032},
	{0x0280, 0x8130B033},
	{0x0281, 0x8130B034},
	{0x0282, 0x8130B035},
	{0x0283, 0x8130B036},
	{0x0284, 0x8130B037},
	{0x0285, 0x8130B038},
	{0x0286, 0x8130B039},
	{0x0287, 0x8130B130},
	{0x0288, 0x8130B131},
	{0x0289, 0x8130B132},
	{0x028A, 0x8130B133},
	{0x028B, 0x8130B134},
	{0x028C, 0x8130B135},
	{0x028D, 0x8130B136},
	{0x028E, 0x8130B137},
	{0x028F, 0x8130B138},
	{0x0290, 0x8130B139},
	{0x0291, 0x8130B230},
	{0x0292, 0x8130B231},
	{0x0293, 0x8130B232},
	{0x0294, 0x8130B233},
	{0x0295, 0x8130B234},
	{0x0296, 0x8130B235},
	{0x0297, 0x8130B236},
	{0x0298, 0x8130B237},
	{0x0299, 0x8130B238},
	{0x029A, 0x8130B239},
	{0x029B, 0x8130B330},
	{0x029C, 0x8130B331},
	{0x029D, 0x8130B332},
	{0x029E, 0x8130B333},
	{0x029F, 0x8130B334},
	{0x02A0, 0x8130B335},
	{0x02A1, 0x8130B336},
	{0x02A2, 0x8130B337},
	{0x02A3, 0x8130B338},
	{0x02A4, 0x8130B339},
	{0x02A5, 0x8130B430},
	{0x02A6, 0x8130B431},
	{0x02A7, 0x8130B432},
	{0x02A8, 0x8130B433},
	{0x02A9, 0x8130B434},
	{0x02AA, 0x8130B435},
	{0x02AB, 0x8130B436},
	{0x02AC, 0x8130B437},
	{0x02AD, 0x8130B438},
	{0x02AE, 0x8130B439},
	{0x02AF, 0x8130B530},
	{0x02B0, 0x8130B531},
	{0x02B1, 0x8130B532},
	{0x02B2, 0x8130B533},
	{0x02B3, 0x8130B534},
	{0x02B4, 0x8130B535},
	{0x02B5, 0x8130B536},
	{0x02B6, 0x8130B537},
	{0x02B7, 0x8130B538},
	{0x02B8, 0x8130B539},
	{0x02B9, 0x8130B630},
	{0x02BA, 0x8130B631},
	{0x02BB, 0x8130B632},
	{0x02BC, 0x8130B633},
	{0x02BD, 0x8130B634},
	{0x02BE, 0x8130B635},
	{0x02BF, 0x8130B636},
	{0x02C0, 0x8130B637},
	{0x02C1, 0x8130B638},
	{0x02C2, 0x8130B639},
	{0x02C3, 0x8130B730},
	{0x02C4, 0x8130B731},
	{0x02C5, 0x8130B732},
	{0x02C6, 0x8130B733},
	{0x02C8, 0x8130B734},
	{0x02CC, 0x8130B735},
	{0x02CD, 0x8130B736},
	{0x02CE, 0x8130B737},
	{0x02CF, 0x8130B738},
	{0x02D0, 0x8130B739},
	{0x02D1, 0x8130B830},
	{0x02D2, 0x8130B831},
	{0x02D3, 0x8130B832},
	{0x02D4, 0x8130B833},
	{0x02D5, 0x8130B834},
	{0x02D6, 0x8130B835},
	{0x02D7, 0x8130B836},
	{0x02D8, 0x8130B837},
	{0x02DA, 0x8130B838},
	{0x02DB, 0x8130B839},
	{0x02DC, 0x8130B930},
	{0x02DD, 0x8130B931},
	{0x02DE, 0x8130B932},
	{0x02DF, 0x8130B933},
	{0x02E0, 0x8130B934},
	{0x02E1, 0x8130B935},
	{0x02E2, 0x8130B936},
	{0x02E3, 0x8130B937},
	{0x02E4, 0x8130B938},
	{0x02E5, 0x8130B939},
	{0x02E6, 0x8130BA30},
	{0x02E7, 0x8130BA31},
	{0x02E8, 0x8130BA32},
	{0x02E9, 0x8130BA33},
	{0x02EA, 0x8130BA34},
	{0x02EB, 0x8130BA35},
	{0x02EC, 0x8130BA36},
	{0x02ED, 0x8130BA37},
	{0x02EE, 0x8130BA38},
	{0x02EF, 0x8130BA39},
	{0x02F0, 0x8130BB30},
	{0x02F1, 0x8130BB31},
	{0x02F2, 0x8130BB32},
	{0x02F3, 0x8130BB33},
	{0x02F4, 0x8130BB34},
	{0x02F5, 0x8130BB35},
	{0x02F6, 0x8130BB36},
	{0x02F7, 0x8130BB37},
	{0x02F8, 0x8130BB38},
	{0x02F9, 0x8130BB39},
	{0x02FA, 0x8130BC30},
	{0x02FB, 0x8130BC31},
	{0x02FC, 0x8130BC32},
	{0x02FD, 0x8130BC33},
	{0x02FE, 0x8130BC34},
	{0x02FF, 0x8130BC35},
	{0x0300, 0x8130BC36},
	{0x0301, 0x8130BC37},
	{0x0302, 0x8130BC38},
	{0x0303, 0x8130BC39},
	{0x0304, 0x8130BD30},
	{0x0305, 0x8130BD31},
	{0x0306, 0x8130BD32},
	{0x0307, 0x8130BD33},
	{0x0308, 0x8130BD34},
	{0x0309, 0x8130BD35},
	{0x030A, 0x8130BD36},
	{0x030B, 0x8130BD37},
	{0x030C, 0x8130BD38},
	{0x030D, 0x8130BD39},
	{0x030E, 0x8130BE30},
	{0x030F, 0x8130BE31},
	{0x0310, 0x8130BE32},
	{0x0311, 0x8130BE33},
	{0x0312, 0x8130BE34},
	{0x0313, 0x8130BE35},
	{0x0314, 0x8130BE36},
	{0x0315, 0x8130BE37},
	{0x0316, 0x8130BE38},
	{0x0317, 0x8130BE39},
	{0x0318, 0x8130BF30},
	{0x0319, 0x8130BF31},
	{0x031A, 0x8130BF32},
	{0x031B, 0x8130BF33},
	{0x031C, 0x8130BF34},
	{0x031D, 0x8130BF35},
	{0x031E, 0x8130BF36},
	{0x031F, 0x8130BF37},
	{0x0320, 0x8130BF38},
	{0x0321, 0x8130BF39},
	{0x0322, 0x8130C030},
	{0x0323, 0x8130C031},
	{0x0324, 0x8130C032},
	{0x0325, 0x8130C033},
	{0x0326, 0x8130C034},
	{0x0327, 0x8130C035},
	{0x0328, 0x8130C036},
	{0x0329, 0x8130C037},
	{0x032A, 0x8130C038},
	{0x032B, 0x8130C039},
	{0x032C, 0x8130C130},
	{0x032D, 0x8130C131},
	{0x032E, 0x8130C132},
	{0x032F, 0x8130C133},
	{0x0330, 0x8130C134},
	{0x0331, 0x8130C135},
	{0x0332, 0x8130C136},
	{0x0333, 0x8130C137},
	{0x0334, 0x8130C138},
	{0x0335, 0x8130C139},
	{0x0336, 0x8130C230},
	{0x0337, 0x8130C231},
	{0x0338, 0x8130C232},
	{0x0339, 0x8130C233},
	{0x033A, 0x8130C234},
	{0x033B, 0x8130C235},
	{0x033C, 0x8130C236},
	{0x033D, 0x8130C237},
	{0x033E, 0x8130C238},
	{0x033F, 0x8130C239},
	{0x0340, 0x8130C330},
	{0x0341, 0x8130C331},
	{0x0342, 0x8130C332},
	{0x0343, 0x8130C333},
	{0x0344, 0x8130C334},
	{0x0345, 0x8130C335},
	{0x0346, 0x8130C336},
	{0x0347, 0x8130C337},
	{0x0348, 0x8130C338},
	{0x0349, 0x8130C339},
	{0x034A, 0x8130C430},
	{0x034B, 0x8130C431},
	{0x034C, 0x8130C432},
	{0x034D, 0x8130C433},
	{0x034E, 0x8130C434},
	{0x034F, 0x8130C435},
	{0x0350, 0x8130C436},
	{0x0351, 0x8130C437},
	{0x0352, 0x8130C438},
	{0x0353, 0x8130C439},
	{0x0354, 0x8130C530},
	{0x0355, 0x8130C531},
	{0x0356, 0x8130C532},
	{0x0357, 0x8130C533},
	{0x0358, 0x8130C534},
	{0x0359, 0x8130C535},
	{0x035A, 0x8130C536},
	{0x035B, 0x8130C537},
	{0x035C, 0x8130C538},
	{0x035D, 0x8130C539},
	{0x035E, 0x8130C630},
	{0x035F, 0x8130C631},
	{0x0360, 0x8130C632},
	{0x0361, 0x8130C633},
	{0x0362, 0x8130C634},
	{0x0363, 0x8130C635},
	{0x0364, 0x8130C636},
	{0x0365, 0x8130C637},
	{0x0366, 0x8130C638},
	{0x0367, 0x8130C639},
	{0x0368, 0x8130C730},
	{0x0369, 0x8130C731},
	{0x036A, 0x8130C732},
	{0x036B, 0x8130C733},
	{0x036C, 0x8130C734},
	{0x036D, 0x8130C735},
	{0x036E, 0x8130C736},
	{0x036F, 0x8130C737},
	{0x0370, 0x8130C738},
	{0x0371, 0x8130C739},
	{0x0372, 0x8130C830},
	{0x0373, 0x8130C831},
	{0x0374, 0x8130C832},
	{0x0375, 0x8130C833},
	{0x0376, 0x8130C834},
	{0x0377, 0x8130C835},
	{0x0378, 0x8130C836},
	{0x0379, 0x8130C837},
	{0x037A, 0x8130C838},
	{0x037B, 0x8130C839},
	{0x037C, 0x8130C930},
	{0x037D, 0x8130C931},
	{0x037E, 0x8130C932},
	{0x037F, 0x8130C933},
	{0x0380, 0x8130C934},
	{0x0381, 0x8130C935},
	{0x0382, 0x8130C936},
	{0x0383, 0x8130C937},
	{0x0384, 0x8130C938},
	{0x0385, 0x8130C939},
	{0x0386, 0x8130CA30},
	{0x0387, 0x8130CA31},
	{0x0388, 0x8130CA32},
	{0x0389, 0x8130CA33},
	{0x038A, 0x8130CA34},
	{0x038B, 0x8130CA35},
	{0x038C, 0x8130CA36},
	{0x038D, 0x8130CA37},
	{0x038E, 0x8130CA38},
	{0x038F, 0x8130CA39},
	{0x0390, 0x8130CB30},
	{0x03A2, 0x8130CB31},
	{0x03AA, 0x8130CB32},
	{0x03AB, 0x8130CB33},
	{0x03AC, 0x8130CB34},
	{0x03AD, 0x8130CB35},
	{0x03AE, 0x8130CB36},
	{0x03AF, 0x8130CB37},
	{0x03B0, 0x8130CB38},
	{0x03C2, 0x8130CB39},
	{0x03CA, 0x8130CC30},
	{0x03CB, 0x8130CC31},
	{0x03CC, 0x8130CC32},
	{0x03CD, 0x8130CC33},
	{0x03CE, 0x8130CC34},
	{0x03CF, 0x8130CC35},
	{0x03D0, 0x8130CC36},
	{0x03D1, 0x8130CC37},
	{0x03D2, 0x8130CC38},
	{0x03D3, 0x8130CC39},
	{0x03D4, 0x8130CD30},
	{0x03D5, 0x8130CD31},
	{0x03D6, 0x8130CD32},
	{0x03D7, 0x8130CD33},
	{0x03D8, 0x8130CD34},
	{0x03D9, 0x8130CD35},
	{0x03DA, 0x8130CD36},
	{0x03DB, 0x8130CD37},
	{0x03DC, 0x8130CD38},
	{0x03DD, 0x8130CD39},
	{0x03DE, 0x8130CE30},
	{0x03DF, 0x8130CE31},
	{0x03E0, 0x8130CE32},
	{0x03E1, 0x8130CE33},
	{0x03E2, 0x8130CE34},
	{0x03E3, 0x8130CE35},
	{0x03E4, 0x8130CE36},
	{0x03E5, 0x8130CE37},
	{0x03E6, 0x8130CE38},
	{0x03E7, 0x8130CE39},
	{0x03E8, 0x8130CF30},
	{0x03E9, 0x8130CF31},
	{0x03EA, 0x8130CF32},
	{0x03EB, 0x8130CF33},
	{0x03EC, 0x8130CF34},
	{0x03ED, 0x8130CF35},
	{0x03EE, 0x8130CF36},
	{0x03EF, 0x8130CF37},
	{0x03F0, 0x8130CF38},
	{0x03F1, 0x8130CF39},
	{0x03F2, 0x8130D030},
	{0x03F3, 0x8130D031},
	{0x03F4, 0x8130D032},
	{0x03F5, 0x8130D033},
	{0x03F6, 0x8130D034},
	{0x03F7, 0x8130D035},
	{0x03F8, 0x8130D036},
	{0x03F9, 0x8130D037},
	{0x03FA, 0x8130D038},
	{0x03FB, 0x8130D039},
	{0x03FC, 0x8130D130},
	{0x03FD, 0x8130D131},
	{0x03FE, 0x8130D132},
	{0x03FF, 0x8130D133},
	{0x0400, 0x8130D134},
	{0x0402, 0x8130D135},
	{0x0403, 0x8130D136},
	{0x0404, 0x8130D137},
	{0x0405, 0x8130D138},
	{0x0406, 0x8130D139},
	{0x0407, 0x8130D230},
	{0x0408, 0x8130D231},
	{0x0409, 0x8130D232},
	{0x040A, 0x8130D233},
	{0x040B, 0x8130D234},
	{0x040C, 0x8130D235},
	{0x040D, 0x8130D236},
	{0x040E, 0x8130D237},
	{0x040F, 0x8130D238},
	{0x0450, 0x8130D239},
	{0x2011, 0x8136A532},
	{0x2012, 0x8136A533},
	{0x2017, 0x8136A534},
	{0x201A, 0x8136A535},
	{0x201B, 0x8136A536},
	{0x201E, 0x8136A537},
	{0x201F, 0x8136A538},
	{0x2020, 0x8136A539},
	{0x2021, 0x8136A630},
	{0x2022, 0x8136A631},
	{0x2023, 0x8136A632},
	{0x2024, 0x8136A633},
	{0x2027, 0x8136A634},
	{0x2028, 0x8136A635},
	{0x2029, 0x8136A636},
	{0x202A, 0x8136A637},
	{0x202B, 0x8136A638},
	{0x202C, 0x8136A639},
	{0x202D, 0x8136A730},
	{0x202E, 0x8136A731},
	{0x202F, 0x8136A732},
	{0x2031, 0x8136A733},
	{0x2034, 0x8136A734},
	{0x2036, 0x8136A735},
	{0x2037, 0x8136A736},
	{0x2038, 0x8136A737},
	{0x2039, 0x8136A738},
	{0x203A, 0x8136A739},
	{0x203C, 0x8136A830},
	{0x203D, 0x8136A831},
	{0x203E, 0x8136A832},
	{0x203F, 0x8136A833},
	{0x2040, 0x8136A834},
	{0x2041, 0x8136A835},
	{0x2042, 0x8136A836},
	{0x2043, 0x8136A837},
	{0x2044, 0x8136A838},
	{0x2045, 0x8136A839},
	{0x2046, 0x8136A930},
	{0x2047, 0x8136A931},
	{0x2048, 0x8136A932},
	{0x2049, 0x8136A933},
	{0x204A, 0x8136A934},
	{0x204B, 0x8136A935},
	{0x204C, 0x8136A936},
	{0x204D, 0x8136A937},
	{0x204E, 0x8136A938},
	{0x204F, 0x8136A939},
	{0x2050, 0x8136AA30},
	{0x2051, 0x8136AA31},
	{0x2052, 0x8136AA32},
	{0x2053, 0x8136AA33},
	{0x2054, 0x8136AA34},
	{0x2055, 0x8136AA35},
	{0x2056, 0x8136AA36},
	{0x2057, 0x8136AA37},
	{0x2058, 0x8136AA38},
	{0x2059, 0x8136AA39},
	{0x205A, 0x8136AB30},
	{0x205B, 0x8136AB31},
	{0x205C, 0x8136AB32},
	{0x205D, 0x8136AB33},
	{0x205E, 0x8136AB34},
	{0x205F, 0x8136AB35},
	{0x2060, 0x8136AB36},
	{0x2061, 0x8136AB37},
	{0x2062, 0x8136AB38},
	{0x2063, 0x8136AB39},
	{0x2064, 0x8136AC30},
	{0x2065, 0x8136AC31},
	{0x2066, 0x8136AC32},
	{0x2067, 0x8136AC33},
	{0x2068, 0x8136AC34},
	{0x2069, 0x8136AC35},
	{0x206A, 0x8136AC36},
	{0x206B, 0x8136AC37},
	{0x206C, 0x8136AC38},
	{0x206D, 0x8136AC39},
	{0x206E, 0x8136AD30},
	{0x206F, 0x8136AD31},
	{0x2070, 0x8136AD32},
	{0x2071, 0x8136AD33},
	{0x2072, 0x8136AD34},
	{0x2073, 0x8136AD35},
	{0x2074, 0x8136AD36},
	{0x2075, 0x8136AD37},
	{0x2076, 0x8136AD38},
	{0x2077, 0x8136AD39},
	{0x2078, 0x8136AE30},
	{0x2079, 0x8136AE31},
	{0x207A, 0x8136AE32},
	{0x207B, 0x8136AE33},
	{0x207C, 0x8136AE34},
	{0x207D, 0x8136AE35},
	{0x207E, 0x8136AE36},
	{0x207F, 0x8136AE37},
	{0x2080, 0x8136AE38},
	{0x2081, 0x8136AE39},
	{0x2082, 0x8136AF30},
	{0x2083, 0x8136AF31},
	{0x2084, 0x8136AF32},
	{0x2085, 0x8136AF33},
	{0x2086, 0x8136AF34},
	{0x2087, 0x8136AF35},
	{0x2088, 0x8136AF36},
	{0x2089, 0x8136AF37},
	{0x208A, 0x8136AF38},
	{0x208B, 0x8136AF39},
	{0x208C, 0x8136B030},
	{0x208D, 0x8136B031},
	{0x208E, 0x8136B032},
	{0x208F, 0x8136B033},
	{0x2090, 0x8136B034},
	{0x2091, 0x8136B035},
	{0x2092, 0x8136B036},
	{0x2093, 0x8136B037},
	{0x2094, 0x8136B038},
	{0x2095, 0x8136B039},
	{0x2096, 0x8136B130},
	{0x2097, 0x8136B131},
	{0x2098, 0x8136B132},
	{0x2099, 0x8136B133},
	{0x209A, 0x8136B134},
	{0x209B, 0x8136B135},
	{0x209C, 0x8136B136},
	{0x209D, 0x8136B137},
	{0x209E, 0x8136B138},
	{0x209F, 0x8136B139},
	{0x20A0, 0x8136B230},
	{0x20A1, 0x8136B231},
	{0x20A2, 0x8136B232},
	{0x20A3, 0x8136B233},
	{0x20A4, 0x8136B234},
	{0x20A5, 0x8136B235},
	{0x20A6, 0x8136B236},
	{0x20A7, 0x8136B237},
	{0x20A8, 0x8136B238},
	{0x20A9, 0x8136B239},
	{0x20AA, 0x8136B330},
	{0x20AB, 0x8136B331},
	{0x20AD, 0x8136B332},
	{0x20AE, 0x8136B333},
	{0x20AF, 0x8136B334},
	{0x20B0, 0x8136B335},
	{0x20B1, 0x8136B336},
	{0x20B2, 0x8136B337},
	{0x20B3, 0x8136B338},
	{0x20B4, 0x8136B339},
	{0x20B5, 0x8136B430},
	{0x20B6, 0x8136B431},
	{0x20B7, 0x8136B432},
	{0x20B8, 0x8136B433},
	{0x20B9, 0x8136B434},
	{0x20BA, 0x8136B435},
	{0x20BB, 0x8136B436},
	{0x20BC, 0x8136B437},
	{0x20BD, 0x8136B438},
	{0x20BE, 0x8136B439},
	{0x20BF, 0x8136B530},
	{0x20C0, 0x8136B531},
	{0x20C1, 0x8136B532},
	{0x20C2, 0x8136B533},
	{0x20C3, 0x8136B534},
	{0x20C4, 0x8136B535},
	{0x20C5, 0x8136B536},
	{0x20C6, 0x8136B537},
	{0x20C7, 0x8136B538},
	{0x20C8, 0x8136B539},
	{0x20C9, 0x8136B630},
	{0x20CA, 0x8136B631},
	{0x20CB, 0x8136B632},
	{0x20CC, 0x8136B633},
	{0x20CD, 0x8136B634},
	{0x20CE, 0x8136B635},
	{0x20CF, 0x8136B636},
	{0x20D0, 0x8136B637},
	{0x20D1, 0x8136B638},
	{0x20D2, 0x8136B639},
	{0x20D3, 0x8136B730},
	{0x20D4, 0x8136B731},
	{0x20D5, 0x8136B732},
	{0x20D6, 0x8136B733},
	{0x20D7, 0x8136B734},
	{0x20D8, 0x8136B735},
	{0x20D9, 0x8136B736},
	{0x20DA, 0x8136B737},
	{0x20DB, 0x8136B738},
	{0x20DC, 0x8136B739},
	{0x20DD, 0x8136B830},
	{0x20DE, 0x8136B831},
	{0x20DF, 0x8136B832},
	{0x20E0, 0x8136B833},
	{0x20E1, 0x8136B834},
	{0x20E2, 0x8136B835},
	{0x20E3, 0x8136B836},
	{0x20E4, 0x8136B837},
	{0x20E5, 0x8136B838},
	{0x20E6, 0x8136B839},
	{0x20E7, 0x8136B930},
	{0x20E8, 0x8136B931},
	{0x20E9, 0x8136B932},
	{0x20EA, 0x8136B933},
	{0x20EB, 0x8136B934},
	{0x20EC, 0x8136B935},
	{0x20ED, 0x8136B936},
	{0x20EE, 0x8136B937},
	{0x20EF, 0x8136B938},
	{0x20F0, 0x8136B939},
	{0x20F1, 0x8136BA30},
	{0x20F2, 0x8136BA31},
	{0x20F3, 0x8136BA32},
	{0x20F4, 0x8136BA33},
	{0x20F5, 0x8136BA34},
	{0x20F6, 0x8136BA35},
	{0x20F7, 0x8136BA36},
	{0x20F8, 0x8136BA37},
	{0x20F9, 0x8136BA38},
	{0x20FA, 0x8136BA39},
	{0x20FB, 0x8136BB30},
	{0x20FC, 0x8136BB31},
	{0x20FD, 0x8136BB32},
	{0x20FE, 0x8136BB33},
	{0x20FF, 0x8136BB34},
	{0x2100, 0x8136BB35},
	{0x2101, 0x8136BB36},
	{0x2102, 0x8136BB37},
	{0x2104, 0x8136BB38},
	{0x2106, 0x8136BB39},
	{0x2107, 0x8136BC30},
	{0x2108, 0x8136BC31},
	{0x210A, 0x8136BC32},
	{0x210B, 0x8136BC33},
	{0x210C, 0x8136BC34},
	{0x210D, 0x8136BC35},
	{0x210E, 0x8136BC36},
	{0x210F, 0x8136BC37},
	{0x2110, 0x8136BC38},
	{0x2111, 0x8136BC39},
	{0x2112, 0x8136BD30},
	{0x2113, 0x8136BD31},
	{0x2114, 0x8136BD32},
	{0x2115, 0x8136BD33},
	{0x2117, 0x8136BD34},
	{0x2118, 0x8136BD35},
	{0x2119, 0x8136BD36},
	{0x211A, 0x8136BD37},
	{0x211B, 0x8136BD38},
	{0x211C, 0x8136BD39},
	{0x211D, 0x8136BE30},
	{0x211E, 0x8136BE31},
	{0x211F, 0x8136BE32},
	{0x2120, 0x8136BE33},
	{0x2122, 0x8136BE34},
	{0x2123, 0x8136BE35},
	{0x2124, 0x8136BE36},
	{0x2125, 0x8136BE37},
	{0x2126, 0x8136BE38},
	{0x2127, 0x8136BE39},
	{0x2128, 0x8136BF30},
	{0x2129, 0x8136BF31},
	{0x212A, 0x8136BF32},
	{0x212B, 0x8136BF33},
	{0x212C, 0x8136BF34},
	{0x212D, 0x8136BF35},
	{0x212E, 0x8136BF36},
	{0x212F, 0x8136BF37},
	{0x2130, 0x8136BF38},
	{0x2131, 0x8136BF39},
	{0x2132, 0x8136C030},
	{0x2133, 0x8136C031},
	{0x2134, 0x8136C032},
	{0x2135, 0x8136C033},
	{0x2136, 0x8136C034},
	{0x2137, 0x8136C035},
	{0x2138, 0x8136C036},
	{0x2139, 0x8136C037},
	{0x213A, 0x8136C038},
	{0x213B, 0x8136C039},
	{0x213C, 0x8136C130},
	{0x213D, 0x8136C131},
	{0x213E, 0x8136C132},
	{0x213F, 0x8136C133},
	{0x2140, 0x8136C134},
	{0x2141, 0x8136C135},
	{0x2142, 0x8136C136},
	{0x2143, 0x8136C137},
	{0x2144, 0x8136C138},
	{0x2145, 0x8136C139},
	{0x2146, 0x8136C230},
	{0x2147, 0x8136C231},
	{0x2148, 0x8136C232},
	{0x2149, 0x8136C233},
	{0x214A, 0x8136C234},
	{0x214B, 0x8136C235},
	{0x214C, 0x8136C236},
	{0x214D, 0x8136C237},
	{0x214E, 0x8136C238},
	{0x214F, 0x8136C239},
	{0x2150, 0x8136C330},
	{0x2151, 0x8136C331},
	{0x2152, 0x8136C332},
	{0x2153, 0x8136C333},
	{0x2154, 0x8136C334},
	{0x2155, 0x8136C335},
	{0x2156, 0x8136C336},
	{0x2157, 0x8136C337},
	{0x2158, 0x8136C338},
	{0x2159, 0x8136C339},
	{0x215A, 0x8136C430},
	{0x215B, 0x8136C431},
	{0x215C, 0x8136C432},
	{0x215D, 0x8136C433},
	{0x215E, 0x8136C434},
	{0x215F, 0x8136C435},
	{0x216C, 0x8136C436},
	{0x216D, 0x8136C437},
	{0x216E, 0x8136C438},
	{0x216F, 0x8136C439},
	{0x217A, 0x8136C530},
	{0x217B, 0x8136C531},
	{0x217C, 0x8136C532},
	{0x217D, 0x8136C533},
	{0x217E, 0x8136C534},
	{0x217F, 0x8136C535},
	{0x2180, 0x8136C536},
	{0x2181, 0x8136C537},
	{0x2182, 0x8136C538},
	{0x2183, 0x8136C539},
	{0x2184, 0x8136C630},
	{0x2185, 0x8136C631},
	{0x2186, 0x8136C632},
	{0x2187, 0x8136C633},
	{0x2188, 0x8136C634},
	{0x2189, 0x8136C635},
	{0x218A, 0x8136C636},
	{0x218B, 0x8136C637},
	{0x218C, 0x8136C638},
	{0x218D, 0x8136C639},
	{0x218E, 0x8136C730},
	{0x218F, 0x8136C731},
	{0x2194, 0x8136C732},
	{0x2195, 0x8136C733},
	{0x219A, 0x8136C734},
	{0x219B, 0x8136C735},
	{0x219C, 0x8136C736},
	{0x219D, 0x8136C737},
	{0x219E, 0x8136C738},
	{0x219F, 0x8136C739},
	{0x21A0, 0x8136C830},
	{0x21A1, 0x8136C831},
	{0x21A2, 0x8136C832},
	{0x21A3, 0x8136C833},
	{0x21A4, 0x8136C834},
	{0x21A5, 0x8136C835},
	{0x21A6, 0x8136C836},
	{0x21A7, 0x8136C837},
	{0x21A8, 0x8136C838},
	{0x21A9, 0x8136C839},
	{0x21AA, 0x8136C930},
	{0x21AB, 0x8136C931},
	{0x21AC, 0x8136C932},
	{0x21AD, 0x8136C933},
	{0x21AE, 0x8136C934},
	{0x21AF, 0x8136C935},
	{0x21B0, 0x8136C936},
	{0x21B1, 0x8136C937},
	{0x21B2, 0x8136C938},
	{0x21B3, 0x8136C939},
	{0x21B4, 0x8136CA30},
	{0x21B5, 0x8136CA31},
	{0x21B6, 0x8136CA32},
	{0x21B7, 0x8136CA33},
	{0x21B8, 0x8136CA34},
	{0x21B9, 0x8136CA35},
	{0x21BA, 0x8136CA36},
	{0x21BB, 0x8136CA37},
	{0x21BC, 0x8136CA38},
	{0x21BD, 0x8136CA39},
	{0x21BE, 0x8136CB30},
	{0x21BF, 0x8136CB31},
	{0x21C0, 0x8136CB32},
	{0x21C1, 0x8136CB33},
	{0x21C2, 0x8136CB34},
	{0x21C3, 0x8136CB35},
	{0x21C4, 0x8136CB36},
	{0x21C5, 0x8136CB37},
	{0x21C6, 0x8136CB38},
	{0x21C7, 0x8136CB39},
	{0x21C8, 0x8136CC30},
	{0x21C9, 0x8136CC31},
	{0x21CA, 0x8136CC32},
	{0x21CB, 0x8136CC33},
	{0x21CC, 0x8136CC34},
	{0x21CD, 0x8136CC35},
	{0x21CE, 0x8136CC36},
	{0x21CF, 0x8136CC37},
	{0x21D0, 0x8136CC38},
	{0x21D1, 0x8136CC39},
	{0x21D2, 0x8136CD30},
	{0x21D3, 0x8136CD31},
	{0x21D4, 0x8136CD32},
	{0x21D5, 0x8136CD33},
	{0x21D6, 0x8136CD34},
	{0x21D7, 0x8136CD35},
	{0x21D8, 0x8136CD36},
	{0x21D9, 0x8136CD37},
	{0x21DA, 0x8136CD38},
	{0x21DB, 0x8136CD39},
	{0x21DC, 0x8136CE30},
	{0x21DD, 0x8136CE31},
	{0x21DE, 0x8136CE32},
	{0x21DF, 0x8136CE33},
	{0x21E0, 0x8136CE34},
	{0x21E1, 0x8136CE35},
	{0x21E2, 0x8136CE36},
	{0x21E3, 0x8136CE37},
	{0x21E4, 0x8136CE38},
	{0x21E5, 0x8136CE39},
	{0x21E6, 0x8136CF30},
	{0x21E7, 0x8136CF31},
	{0x21E8, 0x8136CF32},
	{0x21E9, 0x8136CF33},
	{0x21EA, 0x8136CF34},
	{0x21EB, 0x8136CF35},
	{0x21EC, 0x8136CF36},
	{0x21ED, 0x8136CF37},
	{0x21EE, 0x8136CF38},
	{0x21EF, 0x8136CF39},
	{0x21F0, 0x8136D030},
	{0x21F1, 0x8136D031},
	{0x21F2, 0x8136D032},
	{0x21F3, 0x8136D033},
	{0x21F4, 0x8136D034},
	{0x21F5, 0x8136D035},
	{0x21F6, 0x8136D036},
	{0x21F7, 0x8136D037},
	{0x21F8, 0x8136D038},
	{0x21F9, 0x8136D039},
	{0x21FA, 0x8136D130},
	{0x21FB, 0x8136D131},
	{0x21FC, 0x8136D132},
	{0x21FD, 0x8136D133},
	{0x21FE, 0x8136D134},
	{0x21FF, 0x8136D135},
	{0x2200, 0x8136D136},
	{0x2201, 0x8136D137},
	{0x2202, 0x8136D138},
	{0x2203, 0x8136D139},
	{0x2204, 0x8136D230},
	{0x2205, 0x8136D231},
	{0x2206, 0x8136D232},
	{0x2207, 0x8136D233},
	{0x2209, 0x8136D234},
	{0x220A, 0x8136D235},
	{0x220B, 0x8136D236},
	{0x220C, 0x8136D237},
	{0x220D, 0x8136D238},
	{0x220E, 0x8136D239},
	{0x2210, 0x8136D330},
	{0x2212, 0x8136D331},
	{0x2213, 0x8136D332},
	{0x2214, 0x8136D333},
	{0x2216, 0x8136D334},
	{0x2217, 0x8136D335},
	{0x2218, 0x8136D336},
	{0x2219, 0x8136D337},
	{0x221B, 0x8136D338},
	{0x221C, 0x8136D339},
	{0x2221, 0x8136D430},
	{0x2222, 0x8136D431},
	{0x2224, 0x8136D432},
	{0x2226, 0x8136D433},
	{0x222C, 0x8136D434},
	{0x222D, 0x8136D435},
	{0x222F, 0x8136D436},
	{0x2230, 0x8136D437},
	{0x2231, 0x8136D438},
	{0x2232, 0x8136D439},
	{0x2233, 0x8136D530},
	{0x2238, 0x8136D531},
	{0x2239, 0x8136D532},
	{0x223A, 0x8136D533},
	{0x223B, 0x8136D534},
	{0x223C, 0x8136D535},
	{0x223E, 0x8136D536},
	{0x223F, 0x8136D537},
	{0x2240, 0x8136D538},
	{0x2241, 0x8136D539},
	{0x2242, 0x8136D630},
	{0x2243, 0x8136D631},
	{0x2244, 0x8136D632},
	{0x2245, 0x8136D633},
	{0x2246, 0x8136D634},
	{0x2247, 0x8136D635},
	{0x2249, 0x8136D636},
	{0x224A, 0x8136D637},
	{0x224B, 0x8136D638},
	{0x224D, 0x8136D639},
	{0x224E, 0x8136D730},
	{0x224F, 0x8136D731},
	{0x2250, 0x8136D732},
	{0x2251, 0x8136D733},
	{0x2253, 0x8136D734},
	{0x2254, 0x8136D735},
	{0x2255, 0x8136D736},
	{0x2256, 0x8136D737},
	{0x2257, 0x8136D738},
	{0x2258, 0x8136D739},
	{0x2259, 0x8136D830},
	{0x225A, 0x8136D831},
	{0x225B, 0x8136D832},
	{0x225C, 0x8136D833},
	{0x225D, 0x8136D834},
	{0x225E, 0x8136D835},
	{0x225F, 0x8136D836},
	{0x2262, 0x8136D837},
	{0x2263, 0x8136D838},
	{0x2268, 0x8136D839},
	{0x2269, 0x8136D930},
	{0x226A, 0x8136D931},
	{0x226B, 0x8136D932},
	{0x226C, 0x8136D933},
	{0x226D, 0x8136D934},
	{0x2270, 0x8136D935},
	{0x2271, 0x8136D936},
	{0x2272, 0x8136D937},
	{0x2273, 0x8136D938},
	{0x2274, 0x8136D939},
	{0x2275, 0x8136DA30},
	{0x2276, 0x8136DA31},
	{0x2277, 0x8136DA32},
	{0x2278, 0x8136DA33},
	{0x2279, 0x8136DA34},
	{0x227A, 0x8136DA35},
	{0x227B, 0x8136DA36},
	{0x227C, 0x8136DA37},
	{0x227D, 0x8136DA38},
	{0x227E, 0x8136DA39},
	{0x227F, 0x8136DB30},
	{0x2280, 0x8136DB31},
	{0x2281, 0x8136DB32},
	{0x2282, 0x8136DB33},
	{0x2283, 0x8136DB34},
	{0x2284, 0x8136DB35},
	{0x2285, 0x8136DB36},
	{0x2286, 0x8136DB37},
	{0x2287, 0x8136DB38},
	{0x2288, 0x8136DB39},
	{0x2289, 0x8136DC30},
	{0x228A, 0x8136DC31},
	{0x228B, 0x8136DC32},
	{0x228C, 0x8136DC33},
	{0x228D, 0x8136DC34},
	{0x228E, 0x8136DC35},
	{0x228F, 0x8136DC36},
	{0x2290, 0x8136DC37},
	{0x2291, 0x8136DC38},
	{0x2292, 0x8136DC39},
	{0x2293, 0x8136DD30},
	{0x2294, 0x8136DD31},
	{0x2296, 0x8136DD32},
	{0x2297, 0x8136DD33},
	{0x2298, 0x8136DD34},
	{0x229A, 0x8136DD35},
	{0x229B, 0x8136DD36},
	{0x229C, 0x8136DD37},
	{0x229D, 0x8136DD38},
	{0x229E, 0x8136DD39},
	{0x229F, 0x8136DE30},
	{0x22A0, 0x8136DE31},
	{0x22A1, 0x8136DE32},
	{0x22A2, 0x8136DE33},
	{0x22A3, 0x8136DE34},
	{0x22A4, 0x8136DE35},
	{0x22A6, 0x8136DE36},
	{0x22A7, 0x8136DE37},
	{0x22A8, 0x8136DE38},
	{0x22A9, 0x8136DE39},
	{0x22AA, 0x8136DF30},
	{0x22AB, 0x8136DF31},
	{0x22AC, 0x8136DF32},
	{0x22AD, 0x8136DF33},
	{0x22AE, 0x8136DF34},
	{0x22AF, 0x8136DF35},
	{0x22B0, 0x8136DF36},
	{0x22B1, 0x8136DF37},
	{0x22B2, 0x8136DF38},
	{0x22B3, 0x8136DF39},
	{0x22B4, 0x8136E030},
	{0x22B5, 0x8136E031},
	{0x22B6, 0x8136E032},
	{0x22B7, 0x8136E033},
	{0x22B8, 0x8136E034},
	{0x22B9, 0x8136E035},
	{0x22BA, 0x8136E036},
	{0x22BB, 0x8136E037},
	{0x22BC, 0x8136E038},
	{0x22BD, 0x8136E039},
	{0x22BE, 0x8136E130},
	{0x22C0, 0x8136E131},
	{0x22C1, 0x8136E132},
	{0x22C2, 0x8136E133},
	{0x22C3, 0x8136E134},
	{0x22C4, 0x8136E135},
	{0x22C5, 0x8136E136},
	{0x22C6, 0x8136E137},
	{0x22C7, 0x8136E138},
	{0x22C8, 0x8136E139},
	{0x22C9, 0x8136E230},
	{0x22CA, 0x8136E231},
	{0x22CB, 0x8136E232},
	{0x22CC, 0x8136E233},
	{0x22CD, 0x8136E234},
	{0x22CE, 0x8136E235},
	{0x22CF, 0x8136E236},
	{0x22D0, 0x8136E237},
	{0x22D1, 0x8136E238},
	{0x22D2, 0x8136E239},
	{0x22D3, 0x8136E330},
	{0x22D4, 0x8136E331},
	{0x22D5, 0x8136E332},
	{0x22D6, 0x8136E333},
	{0x22D7, 0x8136E334},
	{0x22D8, 0x8136E335},
	{0x22D9, 0x8136E336},
	{0x22DA, 0x8136E337},
	{0x22DB, 0x8136E338},
	{0x22DC, 0x8136E339},
	{0x22DD, 0x8136E430},
	{0x22DE, 0x8136E431},
	{0x22DF, 0x8136E432},
	{0x22E0, 0x8136E433},
	{0x22E1, 0x8136E434},
	{0x22E2, 0x8136E435},
	{0x22E3, 0x8136E436},
	{0x22E4, 0x8136E437},
	{0x22E5, 0x8136E438},
	{0x22E6, 0x8136E439},
	{0x22E7, 0x8136E530},
	{0x22E8, 0x8136E531},
	{0x22E9, 0x8136E532},
	{0x22EA, 0x8136E533},
	{0x22EB, 0x8136E534},
	{0x22EC, 0x8136E535},
	{0x22ED, 0x8136E536},
	{0x22EE, 0x8136E537},
	{0x22EF, 0x8136E538},
	{0x22F0, 0x8136E539},
	{0x22F1, 0x8136E630},
	{0x22F2, 0x8136E631},
	{0x22F3, 0x8136E632},
	{0x22F4, 0x8136E633},
	{0x22F5, 0x8136E634},
	{0x22F6, 0x8136E635},
	{0x22F7, 0x8136E636},
	{0x22F8, 0x8136E637},
	{0x22F9, 0x8136E638},
	{0x22FA, 0x8136E639},
	{0x22FB, 0x8136E730},
	{0x22FC, 0x8136E731},
	{0x22FD, 0x8136E732},
	{0x22FE, 0x8136E733},
	{0x22FF, 0x8136E734},
	{0x2300, 0x8136E735},
	{0x2301, 0x8136E736},
	{0x2302, 0x8136E737},
	{0x2303, 0x8136E738},
	{0x2304, 0x8136E739},
	{0x2305, 0x8136E830},
	{0x2306, 0x8136E831},
	{0x2307, 0x8136E832},
	{0x2308, 0x8136E833},
	{0x2309, 0x8136E834},
	{0x230A, 0x8136E835},
	{0x230B, 0x8136E836},
	{0x230C, 0x8136E837},
	{0x230D, 0x8136E838},
	{0x230E, 0x8136E839},
	{0x230F, 0x8136E930},
	{0x2310, 0x8136E931},
	{0x2311, 0x8136E932},
	{0x2313, 0x8136E933},
	{0x2314, 0x8136E934},
	{0x2315, 0x8136E935},
	{0x2316, 0x8136E936},
	{0x2317, 0x8136E937},
	{0x2318, 0x8136E938},
	{0x2319, 0x8136E939},
	{0x231A, 0x8136EA30},
	{0x231B, 0x8136EA31},
	{0x231C, 0x8136EA32},
	{0x231D, 0x8136EA33},
	{0x231E, 0x8136EA34},
	{0x231F, 0x8136EA35},
	{0x2320, 0x8136EA36},
	{0x2321, 0x8136EA37},
	{0x2322, 0x8136EA38},
	{0x2323, 0x8136EA39},
	{0x2324, 0x8136EB30},
	{0x2325, 0x8136EB31},
	{0x2326, 0x8136EB32},
	{0x2327, 0x8136EB33},
	{0x2328, 0x8136EB34},
	{0x2329, 0x8136EB35},
	{0x232A, 0x8136EB36},
	{0x232B, 0x8136EB37},
	{0x232C, 0x8136EB38},
	{0x232D, 0x8136EB39},
	{0x232E, 0x8136EC30},
	{0x232F, 0x8136EC31},
	{0x2330, 0x8136EC32},
	{0x2331, 0x8136EC33},
	{0x2332, 0x8136EC34},
	{0x2333, 0x8136EC35},
	{0x2334, 0x8136EC36},
	{0x2335, 0x8136EC37},
	{0x2336, 0x8136EC38},
	{0x2337, 0x8136EC39},
	{0x2338, 0x8136ED30},
	{0x2339, 0x8136ED31},
	{0x233A, 0x8136ED32},
	{0x233B, 0x8136ED33},
	{0x233C, 0x8136ED34},
	{0x233D, 0x8136ED35},
	{0x233E, 0x8136ED36},
	{0x233F, 0x8136ED37},
	{0x2340, 0x8136ED38},
	{0x2341, 0x8136ED39},
	{0x2342, 0x8136EE30},
	{0x2343, 0x8136EE31},
	{0x2344, 0x8136EE32},
	{0x2345, 0x8136EE33},
	{0x2346, 0x8136EE34},
	{0x2347, 0x8136EE35},
	{0x2348, 0x8136EE36},
	{0x2349, 0x8136EE37},
	{0x234A, 0x8136EE38},
	{0x234B, 0x8136EE39},
	{0x234C, 0x8136EF30},
	{0x234D, 0x8136EF31},
	{0x234E, 0x8136EF32},
	{0x234F, 0x8136EF33},
	{0x2350, 0x8136EF34},
	{0x2351, 0x8136EF35},
	{0x2352, 0x8136EF36},
	{0x2353, 0x8136EF37},
	{0x2354, 0x8136EF38},
	{0x2355, 0x8136EF39},
	{0x2356, 0x8136F030},
	{0x2357, 0x8136F031},
	{0x2358, 0x8136F032},
	{0x2359, 0x8136F033},
	{0x235A, 0x8136F034},
	{0x235B, 0x8136F035},
	{0x235C, 0x8136F036},
	{0x235D, 0x8136F037},
	{0x235E, 0x8136F038},
	{0x235F, 0x8136F039},
	{0x2360, 0x8136F130},
	{0x2361, 0x8136F131},
	{0x2362, 0x8136F132},
	{0x2363, 0x8136F133},
	{0x2364, 0x8136F134},
	{0x2365, 0x8136F135},
	{0x2366, 0x8136F136},
	{0x2367, 0x8136F137},
	{0x2368, 0x8136F138},
	{0x2369, 0x8136F139},
	{0x236A, 0x8136F230},
	{0x236B, 0x8136F231},
	{0x236C, 0x8136F232},
	{0x236D, 0x8136F233},
	{0x236E, 0x8136F234},
	{0x236F, 0x8136F235},
	{0x2370, 0x8136F236},
	{0x2371, 0x8136F237},
	{0x2372, 0x8136F238},
	{0x2373, 0x8136F239},
	{0x2374, 0x8136F330},
	{0x2375, 0x8136F331},
	{0x2376, 0x8136F332},
	{0x2377, 0x8136F333},
	{0x2378, 0x8136F334},
	{0x2379, 0x8136F335},
	{0x237A, 0x8136F336},
	{0x237B, 0x8136F337},
	{0x237C, 0x8136F338},
	{0x237D, 0x8136F339},
	{0x237E, 0x8136F430},
	{0x237F, 0x8136F431},
	{0x2380, 0x8136F432},
	{0x2381, 0x8136F433},
	{0x2382, 0x8136F434},
	{0x2383, 0x8136F435},
	{0x2384, 0x8136F436},
	{0x2385, 0x8136F437},
	{0x2386, 0x8136F438},
	{0x2387, 0x8136F439},
	{0x2388, 0x8136F530},
	{0x2389, 0x8136F531},
	{0x238A, 0x8136F532},
	{0x238B, 0x8136F533},
	{0x238C, 0x8136F534},
	{0x238D, 0x8136F535},
	{0x238E, 0x8136F536},
	{0x238F, 0x8136F537},
	{0x2390, 0x8136F538},
	{0x2391, 0x8136F539},
	{0x2392, 0x8136F630},
	{0x2393, 0x8136F631},
	{0x2394, 0x8136F632},
	{0x2395, 0x8136F633},
	{0x2396, 0x8136F634},
	{0x2397, 0x8136F635},
	{0x2398, 0x8136F636},
	{0x2399, 0x8136F637},
	{0x239A, 0x8136F638},
	{0x239B, 0x8136F639},
	{0x239C, 0x8136F730},
	{0x239D, 0x8136F731},
	{0x239E, 0x8136F732},
	{0x239F, 0x8136F733},
	{0x23A0, 0x8136F734},
	{0x23A1, 0x8136F735},
	{0x23A2, 0x8136F736},
	{0x23A3, 0x8136F737},
	{0x23A4, 0x8136F738},
	{0x23A5, 0x8136F739},
	{0x23A6, 0x8136F830},
	{0x23A7, 0x8136F831},
	{0x23A8, 0x8136F832},
	{0x23A9, 0x8136F833},
	{0x23AA, 0x8136F834},
	{0x23AB, 0x8136F835},
	{0x23AC, 0x8136F836},
	{0x23AD, 0x8136F837},
	{0x23AE, 0x8136F838},
	{0x23AF, 0x8136F839},
	{0x23B0, 0x8136F930},
	{0x23B1, 0x8136F931},
	{0x23B2, 0x8136F932},
	{0x23B3, 0x8136F933},
	{0x23B4, 0x8136F934},
	{0x23B5, 0x8136F935},
	{0x23B6, 0x8136F936},
	{0x23B7, 0x8136F937},
	{0x23B8, 0x8136F938},
	{0x23B9, 0x8136F939},
	{0x23BA, 0x8136FA30},
	{0x23BB, 0x8136FA31},
	{0x23BC, 0x8136FA32},
	{0x23BD, 0x8136FA33},
	{0x23BE, 0x8136FA34},
	{0x23BF, 0x8136FA35},
	{0x23C0, 0x8136FA36},
	{0x23C1, 0x8136FA37},
	{0x23C2, 0x8136FA38},
	{0x23C3, 0x8136FA39},
	{0x23C4, 0x8136FB30},
	{0x23C5, 0x8136FB31},
	{0x23C6, 0x8136FB32},
	{0x23C7, 0x8136FB33},
	{0x23C8, 0x8136FB34},
	{0x23C9, 0x8136FB35},
	{0x23CA, 0x8136FB36},
	{0x23CB, 0x8136FB37},
	{0x23CC, 0x8136FB38},
	{0x23CD, 0x8136FB39},
	{0x23CE, 0x8136FC30},
	{0x23CF, 0x8136FC31},
	{0x23D0, 0x8136FC32},
	{0x23D1, 0x8136FC33},
	{0x23D2, 0x8136FC34},
	{0x23D3, 0x8136FC35},
	{0x23D4, 0x8136FC36},
	{0x23D5, 0x8136FC37},
	{0x23D6, 0x8136FC38},
	{0x23D7, 0x8136FC39},
	{0x23D8, 0x8136FD30},
	{0x23D9, 0x8136FD31},
	{0x23DA, 0x8136FD32},
	{0x23DB, 0x8136FD33},
	{0x23DC, 0x8136FD34},
	{0x23DD, 0x8136FD35},
	{0x23DE, 0x8136FD36},
	{0x23DF, 0x8136FD37},
	{0x23E0, 0x8136FD38},
	{0x23E1, 0x8136FD39},
	{0x23E2, 0x8136FE30},
	{0x23E3, 0x8136FE31},
	{0x23E4, 0x8136FE32},
	{0x23E5, 0x8136FE33},
	{0x23E6, 0x8136FE34},
	{0x23E7, 0x8136FE35},
	{0x23E8, 0x8136FE36},
	{0x23E9, 0x8136FE37},
	{0x23EA, 0x8136FE38},
	{0x23EB, 0x8136FE39},
	{0x23EC, 0x81378130},
	{0x23ED, 0x81378131},
	{0x23EE, 0x81378132},
	{0x23EF, 0x81378133},
	{0x23F0, 0x81378134},
	{0x23F1, 0x81378135},
	{0x23F2, 0x81378136},
	{0x23F3, 0x81378137},
	{0x23F4, 0x81378138},
	{0x23F5, 0x81378139},
	{0x23F6, 0x81378230},
	{0x23F7, 0x81378231},
	{0x23F8, 0x81378232},
	{0x23F9, 0x81378233},
	{0x23FA, 0x81378234},
	{0x23FB, 0x81378235},
	{0x23FC, 0x81378236},
	{0x23FD, 0x81378237},
	{0x23FE, 0x81378238},
	{0x23FF, 0x81378239},
	{0x2400, 0x81378330},
	{0x2401, 0x81378331},
	{0x2402, 0x81378332},
	{0x2403, 0x81378333},
	{0x2404, 0x81378334},
	{0x2405, 0x81378335},
	{0x2406, 0x81378336},
	{0x2407, 0x81378337},
	{0x2408, 0x81378338},
	{0x2409, 0x81378339},
	{0x240A, 0x81378430},
	{0x240B, 0x81378431},
	{0x240C, 0x81378432},
	{0x240D, 0x81378433},
	{0x240E, 0x81378434},
	{0x240F, 0x81378435},
	{0x2410, 0x81378436},
	{0x2411, 0x81378437},
	{0x2412, 0x81378438},
	{0x2413, 0x81378439},
	{0x2414, 0x81378530},
	{0x2415, 0x81378531},
	{0x2416, 0x81378532},
	{0x2417, 0x81378533},
	{0x2418, 0x81378534},
	{0x2419, 0x81378535},
	{0x241A, 0x81378536},
	{0x241B, 0x81378537},
	{0x241C, 0x81378538},
	{0x241D, 0x81378539},
	{0x241E, 0x81378630},
	{0x241F, 0x81378631},
	{0x2420, 0x81378632},
	{0x2421, 0x81378633},
	{0x2422, 0x81378634},
	{0x2423, 0x81378635},
	{0x2424, 0x81378636},
	{0x2425, 0x81378637},
	{0x2426, 0x81378638},
	{0x2427, 0x81378639},
	{0x2428, 0x81378730},
	{0x2429, 0x81378731},
	{0x242A, 0x81378732},
	{0x242B, 0x81378733},
	{0x242C, 0x81378734},
	{0x242D, 0x81378735},
	{0x242E, 0x81378736},
	{0x242F, 0x81378737},
	{0x2430, 0x81378738},
	{0x2431, 0x81378739},
	{0x2432, 0x81378830},
	{0x2433, 0x81378831},
	{0x2434, 0x81378832},
	{0x2435, 0x81378833},
	{0x2436, 0x81378834},
	{0x2437, 0x81378835},
	{0x2438, 0x81378836},
	{0x2439, 0x81378837},
	{0x243A, 0x81378838},
	{0x243B, 0x81378839},
	{0x243C, 0x81378930},
	{0x243D, 0x81378931},
	{0x243E, 0x81378932},
	{0x243F, 0x81378933},
	{0x2440, 0x81378934},
	{0x2441, 0x81378935},
	{0x2442, 0x81378936},
	{0x2443, 0x81378937},
	{0x2444, 0x81378938},
	{0x2445, 0x81378939},
	{0x2446, 0x81378A30},
	{0x2447, 0x81378A31},
	{0x2448, 0x81378A32},
	{0x2449, 0x81378A33},
	{0x244A, 0x81378A34},
	{0x244B, 0x81378A35},
	{0x244C, 0x81378A36},
	{0x244D, 0x81378A37},
	{0x244E, 0x81378A38},
	{0x244F, 0x81378A39},
	{0x2450, 0x81378B30},
	{0x2451, 0x81378B31},
	{0x2452, 0x81378B32},
	{0x2453, 0x81378B33},
	{0x2454, 0x81378B34},
	{0x2455, 0x81378B35},
	{0x2456, 0x81378B36},
	{0x2457, 0x81378B37},
	{0x2458, 0x81378B38},
	{0x2459, 0x81378B39},
	{0x245A, 0x81378C30},
	{0x245B, 0x81378C31},
	{0x245C, 0x81378C32},
	{0x245D, 0x81378C33},
	{0x245E, 0x81378C34},
	{0x245F, 0x81378C35},
	{0x246A, 0x81378C36},
	{0x246B, 0x81378C37},
	{0x246C, 0x81378C38},
	{0x246D, 0x81378C39},
	{0x246E, 0x81378D30},
	{0x246F, 0x81378D31},
	{0x2470, 0x81378D32},
	{0x2471, 0x81378D33},
	{0x2472, 0x81378D34},
	{0x2473, 0x81378D35},
	{0x249C, 0x81378D36},
	{0x249D, 0x81378D37},
	{0x249E, 0x81378D38},
	{0x249F, 0x81378D39},
	{0x24A0, 0x81378E30},
	{0x24A1, 0x81378E31},
	{0x24A2, 0x81378E32},
	{0x24A3, 0x81378E33},
	{0x24A4, 0x81378E34},
	{0x24A5, 0x81378E35},
	{0x24A6, 0x81378E36},
	{0x24A7, 0x81378E37},
	{0x24A8, 0x81378E38},
	{0x24A9, 0x81378E39},
	{0x24AA, 0x81378F30},
	{0x24AB, 0x81378F31},
	{0x24AC, 0x81378F32},
	{0x24AD, 0x81378F33},
	{0x24AE, 0x81378F34},
	{0x24AF, 0x81378F35},
	{0x24B0, 0x81378F36},
	{0x24B1, 0x81378F37},
	{0x24B2, 0x81378F38},
	{0x24B3, 0x81378F39},
	{0x24B4, 0x81379030},
	{0x24B5, 0x81379031},
	{0x24B6, 0x81379032},
	{0x24B7, 0x81379033},
	{0x24B8, 0x81379034},
	{0x24B9, 0x81379035},
	{0x24BA, 0x81379036},
	{0x24BB, 0x81379037},
	{0x24BC, 0x81379038},
	{0x24BD, 0x81379039},
	{0x24BE, 0x81379130},
	{0x24BF, 0x81379131},
	{0x24C0, 0x81379132},
	{0x24C1, 0x81379133},
	{0x24C2, 0x81379134},
	{0x24C3, 0x81379135},
	{0x24C4, 0x81379136},
	{0x24C5, 0x81379137},
	{0x24C6, 0x81379138},
	{0x24C7, 0x81379139},
	{0x24C8, 0x81379230},
	{0x24C9, 0x81379231},
	{0x24CA, 0x81379232},
	{0x24CB, 0x81379233},
	{0x24CC, 0x81379234},
	{0x24CD, 0x81379235},
	{0x24CE, 0x81379236},
	{0x24CF, 0x81379237},
	{0x24D0, 0x81379238},
	{0x24D1, 0x81379239},
	{0x24D2, 0x81379330},
	{0x24D3, 0x81379331},
	{0x24D4, 0x81379332},
	{0x24D5, 0x81379333},
	{0x24D6, 0x81379334},
	{0x24D7, 0x81379335},
	{0x24D8, 0x81379336},
	{0x24D9, 0x81379337},
	{0x24DA, 0x81379338},
	{0x24DB, 0x81379339},
	{0x24DC, 0x81379430},
	{0x24DD, 0x81379431},
	{0x24DE, 0x81379432},
	{0x24DF, 0x81379433},
	{0x24E0, 0x81379434},
	{0x24E1, 0x81379435},
	{0x24E2, 0x81379436},
	{0x24E3, 0x81379437},
	{0x24E4, 0x81379438},
	{0x24E5, 0x81379439},
	{0x24E6, 0x81379530},
	{0x24E7, 0x81379531},
	{0x24E8, 0x81379532},
	{0x24E9, 0x81379533},
	{0x24EA, 0x81379534},
	{0x24EB, 0x81379535},
	{0x24EC, 0x81379536},
	{0x24ED, 0x81379537},
	{0x24EE, 0x81379538},
	{0x24EF, 0x81379539},
	{0x24F0, 0x81379630},
	{0x24F1, 0x81379631},
	{0x24F2, 0x81379632},
	{0x24F3, 0x81379633},
	{0x24F4, 0x81379634},
	{0x24F5, 0x81379635},
	{0x24F6, 0x81379636},
	{0x24F7, 0x81379637},
	{0x24F8, 0x81379638},
	{0x24F9, 0x81379639},
	{0x24FA, 0x81379730},
	{0x24FB, 0x81379731},
	{0x24FC, 0x81379732},
	{0x24FD, 0x81379733},
	{0x24FE, 0x81379734},
	{0x24FF, 0x81379735},
	{0x254C, 0x81379736},
	{0x254D, 0x81379737},
	{0x254E, 0x81379738},
	{0x254F, 0x81379739},
	{0x2574, 0x81379830},
	{0x2575, 0x81379831},
	{0x2576, 0x81379832},
	{0x2577, 0x81379833},
	{0x2578, 0x81379834},
	{0x2579, 0x81379835},
	{0x257A, 0x81379836},
	{0x257B, 0x81379837},
	{0x257C, 0x81379838},
	{0x257D, 0x81379839},
	{0x257E, 0x81379930},
	{0x257F, 0x81379931},
	{0x2580, 0x81379932},
	{0x2590, 0x81379933},
	{0x2591, 0x81379934},
	{0x2592, 0x81379935},
	{0x2596, 0x81379936},
	{0x2597, 0x81379937},
	{0x2598, 0x81379938},
	{0x2599, 0x81379939},
	{0x259A, 0x81379A30},
	{0x259B, 0x81379A31},
	{0x259C, 0x81379A32},
	{0x259D, 0x81379A33},
	{0x259E, 0x81379A34},
	{0x259F, 0x81379A35},
	{0x25A2, 0x81379A36},
	{0x25A3, 0x81379A37},
	{0x25A4, 0x81379A38},
	{0x25A5, 0x81379A39},
	{0x25A6, 0x81379B30},
	{0x25A7, 0x81379B31},
	{0x25A8, 0x81379B32},
	{0x25A9, 0x81379B33},
	{0x25AA, 0x81379B34},
	{0x25AB, 0x81379B35},
	{0x25AC, 0x81379B36},
	{0x25AD, 0x81379B37},
	{0x25AE, 0x81379B38},
	{0x25AF, 0x81379B39},
	{0x25B0, 0x81379C30},
	{0x25B1, 0x81379C31},
	{0x25B4, 0x81379C32},
	{0x25B5, 0x81379C33},
	{0x25B6, 0x81379C34},
	{0x25B7, 0x81379C35},
	{0x25B8, 0x81379C36},
	{0x25B9, 0x81379C37},
	{0x25BA, 0x81379C38},
	{0x25BB, 0x81379C39},
	{0x25BE, 0x81379D30},
	{0x25BF, 0x81379D31},
	{0x25C0, 0x81379D32},
	{0x25C1, 0x81379D33},
	{0x25C2, 0x81379D34},
	{0x25C3, 0x81379D35},
	{0x25C4, 0x81379D36},
	{0x25C5, 0x81379D37},
	{0x25C8, 0x81379D38},
	{0x25C9, 0x81379D39},
	{0x25CA, 0x81379E30},
	{0x25CC, 0x81379E31},
	{0x25CD, 0x81379E32},
	{0x25D0, 0x81379E33},
	{0x25D1, 0x81379E34},
	{0x25D2, 0x81379E35},
	{0x25D3, 0x81379E36},
	{0x25D4, 0x81379E37},
	{0x25D5, 0x81379E38},
	{0x25D6, 0x81379E39},
	{0x25D7, 0x81379F30},
	{0x25D8, 0x81379F31},
	{0x25D9, 0x81379F32},
	{0x25DA, 0x81379F33},
	{0x25DB, 0x81379F34},
	{0x25DC, 0x81379F35},
	{0x25DD, 0x81379F36},
	{0x25DE, 0x81379F37},
	{0x25DF, 0x81379F38},
	{0x25E0, 0x81379F39},
	{0x25E1, 0x8137A030},
	{0x25E6, 0x8137A031},
	{0x25E7, 0x8137A032},
	{0x25E8, 0x8137A033},
	{0x25E9, 0x8137A034},
	{0x25EA, 0x8137A035},
	{0x25EB, 0x8137A036},
	{0x25EC, 0x8137A037},
	{0x25ED, 0x8137A038},
	{0x25EE, 0x8137A039},
	{0x25EF, 0x8137A130},
	{0x25F0, 0x8137A131},
	{0x25F1, 0x8137A132},
	{0x25F2, 0x8137A133},
	{0x25F3, 0x8137A134},
	{0x25F4, 0x8137A135},
	{0x25F5, 0x8137A136},
	{0x25F6, 0x8137A137},
	{0x25F7, 0x8137A138},
	{0x25F8, 0x8137A139},
	{0x25F9, 0x8137A230},
	{0x25FA, 0x8137A231},
	{0x25FB, 0x8137A232},
	{0x25FC, 0x8137A233},
	{0x25FD, 0x8137A234},
	{0x25FE, 0x8137A235},
	{0x25FF, 0x8137A236},
	{0x2600, 0x8137A237},
	{0x2601, 0x8137A238},
	{0x2602, 0x8137A239},
	{0x2603, 0x8137A330},
	{0x2604, 0x8137A331},
	{0x2607, 0x8137A332},
	{0x2608, 0x8137A333},
	{0x260A, 0x8137A334},
	{0x260B, 0x8137A335},
	{0x260C, 0x8137A336},
	{0x260D, 0x8137A337},
	{0x260E, 0x8137A338},
	{0x260F, 0x8137A339},
	{0x2610, 0x8137A430},
	{0x2611, 0x8137A431},
	{0x2612, 0x8137A432},
	{0x2613, 0x8137A433},
	{0x2614, 0x8137A434},
	{0x2615, 0x8137A435},
	{0x2616, 0x8137A436},
	{0x2617, 0x8137A437},
	{0x2618, 0x8137A438},
	{0x2619, 0x8137A439},
	{0x261A, 0x8137A530},
	{0x261B, 0x8137A531},
	{0x261C, 0x8137A532},
	{0x261D, 0x8137A533},
	{0x261E, 0x8137A534},
	{0x261F, 0x8137A535},
	{0x2620, 0x8137A536},
	{0x2621, 0x8137A537},
	{0x2622, 0x8137A538},
	{0x2623, 0x8137A539},
	{0x2624, 0x8137A630},
	{0x2625, 0x8137A631},
	{0x2626, 0x8137A632},
	{0x2627, 0x8137A633},
	{0x2628, 0x8137A634},
	{0x2629, 0x8137A635},
	{0x262A, 0x8137A636},
	{0x262B, 0x8137A637},
	{0x262C, 0x8137A638},
	{0x262D, 0x8137A639},
	{0x262E, 0x8137A730},
	{0x262F, 0x8137A731},
	{0x2630, 0x8137A732},
	{0x2631, 0x8137A733},
	{0x2632, 0x8137A734},
	{0x2633, 0x8137A735},
	{0x2634, 0x8137A736},
	{0x2635, 0x8137A737},
	{0x2636, 0x8137A738},
	{0x2637, 0x8137A739},
	{0x2638, 0x8137A830},
	{0x2639, 0x8137A831},
	{0x263A, 0x8137A832},
	{0x263B, 0x8137A833},
	{0x263C, 0x8137A834},
	{0x263D, 0x8137A835},
	{0x263E, 0x8137A836},
	{0x263F, 0x8137A837},
	{0x2641, 0x8137A838},
	{0x2E82, 0x8138FD39},
	{0x2E83, 0x8138FE30},
	{0x2E85, 0x8138FE31},
	{0x2E86, 0x8138FE32},
	{0x2E87, 0x8138FE33},
	{0x2E89, 0x8138FE34},
	{0x2E8A, 0x8138FE35},
	{0x2E8D, 0x8138FE36},
	{0x2E8E, 0x8138FE37},
	{0x2E8F, 0x8138FE38},
	{0x2E90, 0x8138FE39},
	{0x2E91, 0x81398130},
	{0x2E92, 0x81398131},
	{0x2E93, 0x81398132},
	{0x2E94, 0x81398133},
	{0x2E95, 0x81398134},
	{0x2E96, 0x81398135},
	{0x2E98, 0x81398136},
	{0x2E99, 0x81398137},
	{0x2E9A, 0x81398138},
	{0x2E9B, 0x81398139},
	{0x2E9C, 0x81398230},
	{0x2E9D, 0x81398231},
	{0x2E9E, 0x81398232},
	{0x2E9F, 0x81398233},
	{0x2EA0, 0x81398234},
	{0x2EA1, 0x81398235},
	{0x2EA2, 0x81398236},
	{0x2EA3, 0x81398237},
	{0x2EA4, 0x81398238},
	{0x2EA5, 0x81398239},
	{0x2EA6, 0x81398330},
	{0x2EA8, 0x81398331},
	{0x2EA9, 0x81398332},
	{0x2EAB, 0x81398333},
	{0x2EAC, 0x81398334},
	{0x2EAD, 0x81398335},
	{0x2EAF, 0x81398336},
	{0x2EB0, 0x81398337},
	{0x2EB1, 0x81398338},
	{0x2EB2, 0x81398339},
	{0x2EB4, 0x81398430},
	{0x2EB5, 0x81398431},
	{0x2EB8, 0x81398432},
	{0x2EB9, 0x81398433},
	{0x2EBA, 0x81398434},
	{0x2EBC, 0x81398435},
	{0x2EBD, 0x81398436},
	{0x2EBE, 0x81398437},
	{0x2EBF, 0x81398438},
	{0x2EC0, 0x81398439},
	{0x2EC1, 0x81398530},
	{0x2EC2, 0x81398531},
	{0x2EC3, 0x81398532},
	{0x2EC4, 0x81398533},
	{0x2EC5, 0x81398534},
	{0x2EC6, 0x81398535},
	{0x2EC7, 0x81398536},
	{0x2EC8, 0x81398537},
	{0x2EC9, 0x81398538},
	{0x2ECB, 0x81398539},
	{0x2ECC, 0x81398630},
	{0x2ECD, 0x81398631},
	{0x2ECE, 0x81398632},
	{0x2ECF, 0x81398633},
	{0x2ED0, 0x81398634},
	{0x2ED1, 0x81398635},
	{0x2ED2, 0x81398636},
	{0x2ED3, 0x81398637},
	{0x2ED4, 0x81398638},
	{0x2ED5, 0x81398639},
	{0x2ED6, 0x81398730},
	{0x2ED7, 0x81398731},
	{0x2ED8, 0x81398732},
	{0x2ED9, 0x81398733},
	{0x2EDA, 0x81398734},
	{0x2EDB, 0x81398735},
	{0x2EDC, 0x81398736},
	{0x2EDD, 0x81398737},
	{0x2EDE, 0x81398738},
	{0x2EDF, 0x81398739},
	{0x2EE0, 0x81398830},
	{0x2EE1, 0x81398831},
	{0x2EE2, 0x81398832},
	{0x2EE3, 0x81398833},
	{0x2EE4, 0x81398834},
	{0x2EE5, 0x81398835},
	{0x2EE6, 0x81398836},
	{0x2EE7, 0x81398837},
	{0x2EE8, 0x81398838},
	{0x2EE9, 0x81398839},
	{0x2EEA, 0x81398930},
	{0x2EEB, 0x81398931},
	{0x2EEC, 0x81398932},
	{0x2EED, 0x81398933},
	{0x2EEE, 0x81398934},
	{0x2EEF, 0x81398935},
	{0x2EF0, 0x81398936},
	{0x2EF1, 0x81398937},
	{0x2EF2, 0x81398938},
	{0x2EF3, 0x81398939},
	{0x2EF4, 0x81398A30},
	{0x2EF5, 0x81398A31},
	{0x2EF6, 0x81398A32},
	{0x2EF7, 0x81398A33},
	{0x2EF8, 0x81398A34},
	{0x2EF9, 0x81398A35},
	{0x2EFA, 0x81398A36},
	{0x2EFB, 0x81398A37},
	{0x2EFC, 0x81398A38},
	{0x2EFD, 0x81398A39},
	{0x2EFE, 0x81398B30},
	{0x2EFF, 0x81398B31},
	{0x2F00, 0x81398B32},
	{0x2F01, 0x81398B33},
	{0x2F02, 0x81398B34},
	{0x2F03, 0x81398B35},
	{0x2F04, 0x81398B36},
	{0x2F05, 0x81398B37},
	{0x2F06, 0x81398B38},
	{0x2F07, 0x81398B39},
	{0x2F08, 0x81398C30},
	{0x2F09, 0x81398C31},
	{0x2F0A, 0x81398C32},
	{0x2F0B, 0x81398C33},
	{0x2F0C, 0x81398C34},
	{0x2F0D, 0x81398C35},
	{0x2F0E, 0x81398C36},
	{0x2F0F, 0x81398C37},
	{0x2F10, 0x81398C38},
	{0x2F11, 0x81398C39},
	{0x2F12, 0x81398D30},
	{0x2F13, 0x81398D31},
	{0x2F14, 0x81398D32},
	{0x2F15, 0x81398D33},
	{0x2F16, 0x81398D34},
	{0x2F17, 0x81398D35},
	{0x2F18, 0x81398D36},
	{0x2F19, 0x81398D37},
	{0x2F1A, 0x81398D38},
	{0x2F1B, 0x81398D39},
	{0x2F1C, 0x81398E30},
	{0x2F1D, 0x81398E31},
	{0x2F1E, 0x81398E32},
	{0x2F1F, 0x81398E33},
	{0x2F20, 0x81398E34},
	{0x2F21, 0x81398E35},
	{0x2F22, 0x81398E36},
	{0x2F23, 0x81398E37},
	{0x2F24, 0x81398E38},
	{0x2F25, 0x81398E39},
	{0x2F26, 0x81398F30},
	{0x2F27, 0x81398F31},
	{0x2F28, 0x81398F32},
	{0x2F29, 0x81398F33},
	{0x2F2A, 0x81398F34},
	{0x2F2B, 0x81398F35},
	{0x2F2C, 0x81398F36},
	{0x2F2D, 0x81398F37},
	{0x2F2E, 0x81398F38},
	{0x2F2F, 0x81398F39},
	{0x2F30, 0x81399030},
	{0x2F31, 0x81399031},
	{0x2F32, 0x81399032},
	{0x2F33, 0x81399033},
	{0x2F34, 0x81399034},
	{0x2F35, 0x81399035},
	{0x2F36, 0x81399036},
	{0x2F37, 0x81399037},
	{0x2F38, 0x81399038},
	{0x2F39, 0x81399039},
	{0x2F3A, 0x81399130},
	{0x2F3B, 0x81399131},
	{0x2F3C, 0x81399132},
	{0x2F3D, 0x81399133},
	{0x2F3E, 0x81399134},
	{0x2F3F, 0x81399135},
	{0x2F40, 0x81399136},
	{0x2F41, 0x81399137},
	{0x2F42, 0x81399138},
	{0x2F43, 0x81399139},
	{0x2F44, 0x81399230},
	{0x2F45, 0x81399231},
	{0x2F46, 0x81399232},
	{0x2F47, 0x81399233},
	{0x2F48, 0x81399234},
	{0x2F49, 0x81399235},
	{0x2F4A, 0x81399236},
	{0x2F4B, 0x81399237},
	{0x2F4C, 0x81399238},
	{0x2F4D, 0x81399239},
	{0x2F4E, 0x81399330},
	{0x2F4F, 0x81399331},
	{0x2F50, 0x81399332},
	{0x2F51, 0x81399333},
	{0x2F52, 0x81399334},
	{0x2F53, 0x81399335},
	{0x2F54, 0x81399336},
	{0x2F55, 0x81399337},
	{0x2F56, 0x81399338},
	{0x2F57, 0x81399339},
	{0x2F58, 0x81399430},
	{0x2F59, 0x81399431},
	{0x2F5A, 0x81399432},
	{0x2F5B, 0x81399433},
	{0x2F5C, 0x81399434},
	{0x2F5D, 0x81399435},
	{0x2F5E, 0x81399436},
	{0x2F5F, 0x81399437},
	{0x2F60, 0x81399438},
	{0x2F61, 0x81399439},
	{0x2F62, 0x81399530},
	{0x2F63, 0x81399531},
	{0x2F64, 0x81399532},
	{0x2F65, 0x81399533},
	{0x2F66, 0x81399534},
	{0x2F67, 0x81399535},
	{0x2F68, 0x81399536},
	{0x2F69, 0x81399537},
	{0x2F6A, 0x81399538},
	{0x2F6B, 0x81399539},
	{0x2F6C, 0x81399630},
	{0x2F6D, 0x81399631},
	{0x2F6E, 0x81399632},
	{0x2F6F, 0x81399633},
	{0x2F70, 0x81399634},
	{0x2F71, 0x81399635},
	{0x2F72, 0x81399636},
	{0x2F73, 0x81399637},
	{0x2F74, 0x81399638},
	{0x2F75, 0x81399639},
	{0x2F76, 0x81399730},
	{0x2F77, 0x81399731},
	{0x2F78, 0x81399732},
	{0x2F79, 0x81399733},
	{0x2F7A, 0x81399734},
	{0x2F7B, 0x81399735},
	{0x2F7C, 0x81399736},
	{0x2F7D, 0x81399737},
	{0x2F7E, 0x81399738},
	{0x2F7F, 0x81399739},
	{0x2F80, 0x81399830},
	{0x2F81, 0x81399831},
	{0x2F82, 0x81399832},
	{0x2F83, 0x81399833},
	{0x2F84, 0x81399834},
	{0x2F85, 0x81399835},
	{0x2F86, 0x81399836},
	{0x2F87, 0x81399837},
	{0x2F88, 0x81399838},
	{0x2F89, 0x81399839},
	{0x2F8A, 0x81399930},
	{0x2F8B, 0x81399931},
	{0x2F8C, 0x81399932},
	{0x2F8D, 0x81399933},
	{0x2F8E, 0x81399934},
	{0x2F8F, 0x81399935},
	{0x2F90, 0x81399936},
	{0x2F91, 0x81399937},
	{0x2F92, 0x81399938},
	{0x2F93, 0x81399939},
	{0x2F94, 0x81399A30},
	{0x2F95, 0x81399A31},
	{0x2F96, 0x81399A32},
	{0x2F97, 0x81399A33},
	{0x2F98, 0x81399A34},
	{0x2F99, 0x81399A35},
	{0x2F9A, 0x81399A36},
	{0x2F9B, 0x81399A37},
	{0x2F9C, 0x81399A38},
	{0x2F9D, 0x81399A39},
	{0x2F9E, 0x81399B30},
	{0x2F9F, 0x81399B31},
	{0x2FA0, 0x81399B32},
	{0x2FA1, 0x81399B33},
	{0x2FA2, 0x81399B34},
	{0x2FA3, 0x81399B35},
	{0x2FA4, 0x81399B36},
	{0x2FA5, 0x81399B37},
	{0x2FA6, 0x81399B38},
	{0x2FA7, 0x81399B39},
	{0x2FA8, 0x81399C30},
	{0x2FA9, 0x81399C31},
	{0x2FAA, 0x81399C32},
	{0x2FAB, 0x81399C33},
	{0x2FAC, 0x81399C34},
	{0x2FAD, 0x81399C35},
	{0x2FAE, 0x81399C36},
	{0x2FAF, 0x81399C37},
	{0x2FB0, 0x81399C38},
	{0x2FB1, 0x81399C39},
	{0x2FB2, 0x81399D30},
	{0x2FB3, 0x81399D31},
	{0x2FB4, 0x81399D32},
	{0x2FB5, 0x81399D33},
	{0x2FB6, 0x81399D34},
	{0x2FB7, 0x81399D35},
	{0x2FB8, 0x81399D36},
	{0x2FB9, 0x81399D37},
	{0x2FBA, 0x81399D38},
	{0x2FBB, 0x81399D39},
	{0x2FBC, 0x81399E30},
	{0x2FBD, 0x81399E31},
	{0x2FBE, 0x81399E32},
	{0x2FBF, 0x81399E33},
	{0x2FC0, 0x81399E34},
	{0x2FC1, 0x81399E35},
	{0x2FC2, 0x81399E36},
	{0x2FC3, 0x81399E37},
	{0x2FC4, 0x81399E38},
	{0x2FC5, 0x81399E39},
	{0x2FC6, 0x81399F30},
	{0x2FC7, 0x81399F31},
	{0x2FC8, 0x81399F32},
	{0x2FC9, 0x81399F33},
	{0x2FCA, 0x81399F34},
	{0x2FCB, 0x81399F35},
	{0x2FCC, 0x81399F36},
	{0x2FCD, 0x81399F37},
	{0x2FCE, 0x81399F38},
	{0x2FCF, 0x81399F39},
	{0x2FD0, 0x8139A030},
	{0x2FD1, 0x8139A031},
	{0x2FD2, 0x8139A032},
	{0x2FD3, 0x8139A033},
	{0x2FD4, 0x8139A034},
	{0x2FD5, 0x8139A035},
	{0x2FD6, 0x8139A036},
	{0x2FD7, 0x8139A037},
	{0x2FD8, 0x8139A038},
	{0x2FD9, 0x8139A039},
	{0x2FDA, 0x8139A130},
	{0x2FDB, 0x8139A131},
	{0x2FDC, 0x8139A132},
	{0x2FDD, 0x8139A133},
	{0x2FDE, 0x8139A134},
	{0x2FDF, 0x8139A135},
	{0x2FE0, 0x8139A136},
	{0x2FE1, 0x8139A137},
	{0x2FE2, 0x8139A138},
	{0x2FE3, 0x8139A139},
	{0x2FE4, 0x8139A230},
	{0x2FE5, 0x8139A231},
	{0x2FE6, 0x8139A232},
	{0x2FE7, 0x8139A233},
	{0x2FE8, 0x8139A234},
	{0x2FE9, 0x8139A235},
	{0x2FEA, 0x8139A236},
	{0x2FEB, 0x8139A237},
	{0x2FEC, 0x8139A238},
	{0x2FED, 0x8139A239},
	{0x2FEE, 0x8139A330},
	{0x2FEF, 0x8139A331},
	{0x2FFC, 0x8139A332},
	{0x2FFD, 0x8139A333},
	{0x2FFE, 0x8139A334},
	{0x2FFF, 0x8139A335},
	{0x3004, 0x8139A336},
	{0x3018, 0x8139A337},
	{0x3019, 0x8139A338},
	{0x301A, 0x8139A339},
	{0x301B, 0x8139A430},
	{0x301C, 0x8139A431},
	{0x301F, 0x8139A432},
	{0x3020, 0x8139A433},
	{0x302A, 0x8139A434},
	{0x302B, 0x8139A435},
	{0x302C, 0x8139A436},
	{0x302D, 0x8139A437},
	{0x302E, 0x8139A438},
	{0x302F, 0x8139A439},
	{0x3030, 0x8139A530},
	{0x3031, 0x8139A531},
	{0x3032, 0x8139A532},
	{0x3033, 0x8139A533},
	{0x3034, 0x8139A534},
	{0x3035, 0x8139A535},
	{0x3036, 0x8139A536},
	{0x3037, 0x8139A537},
	{0x3038, 0x8139A538},
	{0x3039, 0x8139A539},
	{0x303A, 0x8139A630},
	{0x303B, 0x8139A631},
	{0x303C, 0x8139A632},
	{0x303D, 0x8139A633},
	{0x303F, 0x8139A634},
	{0x3040, 0x8139A635},
	{0x3094, 0x8139A636},
	{0x3095, 0x8139A637},
	{0x3096, 0x8139A638},
	{0x3097, 0x8139A639},
	{0x3098, 0x8139A730},
	{0x3099, 0x8139A731},
	{0x309A, 0x8139A732},
	{0x309F, 0x8139A733},
	{0x30A0, 0x8139A734},
	{0x30F7, 0x8139A735},
	{0x30F8, 0x8139A736},
	{0x30F9, 0x8139A737},
	{0x30FA, 0x8139A738},
	{0x30FB, 0x8139A739},
	{0x30FF, 0x8139A830},
	{0x3100, 0x8139A831},
	{0x3101, 0x8139A832},
	{0x3102, 0x8139A833},
	{0x3103, 0x8139A834},
	{0x3104, 0x8139A835},
	{0x312A, 0x8139A836},
	{0x312B, 0x8139A837},
	{0x312C, 0x8139A838},
	{0x312D, 0x8139A839},
	{0x312E, 0x8139A930},
	{0x312F, 0x8139A931},
	{0x3130, 0x8139A932},
	{0x3131, 0x8139A933},
	{0x3132, 0x8139A934},
	{0x3133, 0x8139A935},
	{0x3134, 0x8139A936},
	{0x3135, 0x8139A937},
	{0x3136, 0x8139A938},
	{0x3137, 0x8139A939},
	{0x3138, 0x8139AA30},
	{0x3139, 0x8139AA31},
	{0x313A, 0x8139AA32},
	{0x313B, 0x8139AA33},
	{0x313C, 0x8139AA34},
	{0x313D, 0x8139AA35},
	{0x313E, 0x8139AA36},
	{0x313F, 0x8139AA37},
	{0x3140, 0x8139AA38},
	{0x3141, 0x8139AA39},
	{0x3142, 0x8139AB30},
	{0x3143, 0x8139AB31},
	{0x3144, 0x8139AB32},
	{0x3145, 0x8139AB33},
	{0x3146, 0x8139AB34},
	{0x3147, 0x8139AB35},
	{0x3148, 0x8139AB36},
	{0x3149, 0x8139AB37},
	{0x314A, 0x8139AB38},
	{0x314B, 0x8139AB39},
	{0x314C, 0x8139AC30},
	{0x314D, 0x8139AC31},
	{0x314E, 0x8139AC32},
	{0x314F, 0x8139AC33},
	{0x3150, 0x8139AC34},
	{0x3151, 0x8139AC35},
	{0x3152, 0x8139AC36},
	{0x3153, 0x8139AC37},
	{0x3154, 0x8139AC38},
	{0x3155, 0x8139AC39},
	{0x3156, 0x8139AD30},
	{0x3157, 0x8139AD31},
	{0x3158, 0x8139AD32},
	{0x3159, 0x8139AD33},
	{0x315A, 0x8139AD34},
	{0x315B, 0x8139AD35},
	{0x315C, 0x8139AD36},
	{0x315D, 0x8139AD37},
	{0x315E, 0x8139AD38},
	{0x315F, 0x8139AD39},
	{0x3160, 0x8139AE30},
	{0x3161, 0x8139AE31},
	{0x3162, 0x8139AE32},
	{0x3163, 0x8139AE33},
	{0x3164, 0x8139AE34},
	{0x3165, 0x8139AE35},
	{0x3166, 0x8139AE36},
	{0x3167, 0x8139AE37},
	{0x3168, 0x8139AE38},
	{0x3169, 0x8139AE39},
	{0x316A, 0x8139AF30},
	{0x316B, 0x8139AF31},
	{0x316C, 0x8139AF32},
	{0x316D, 0x8139AF33},
	{0x316E, 0x8139AF34},
	{0x316F, 0x8139AF35},
	{0x3170, 0x8139AF36},
	{0x3171, 0x8139AF37},
	{0x3172, 0x8139AF38},
	{0x3173, 0x8139AF39},
	{0x3174, 0x8139B030},
	{0x3175, 0x8139B031},
	{0x3176, 0x8139B032},
	{0x3177, 0x8139B033},
	{0x3178, 0x8139B034},
	{0x3179, 0x8139B035},
	{0x317A, 0x8139B036},
	{0x317B, 0x8139B037},
	{0x317C, 0x8139B038},
	{0x317D, 0x8139B039},
	{0x317E, 0x8139B130},
	{0x317F, 0x8139B131},
	{0x3180, 0x8139B132},
	{0x3181, 0x8139B133},
	{0x3182, 0x8139B134},
	{0x3183, 0x8139B135},
	{0x3184, 0x8139B136},
	{0x3185, 0x8139B137},
	{0x3186, 0x8139B138},
	{0x3187, 0x8139B139},
	{0x3188, 0x8139B230},
	{0x3189, 0x8139B231},
	{0x318A, 0x8139B232},
	{0x318B, 0x8139B233},
	{0x318C, 0x8139B234},
	{0x318D, 0x8139B235},
	{0x318E, 0x8139B236},
	{0x318F, 0x8139B237},
	{0x3190, 0x8139B238},
	{0x3191, 0x8139B239},
	{0x3192, 0x8139B330},
	{0x3193, 0x8139B331},
	{0x3194, 0x8139B332},
	{0x3195, 0x8139B333},
	{0x3196, 0x8139B334},
	{0x3197, 0x8139B335},
	{0x3198, 0x8139B336},
	{0x3199, 0x8139B337},
	{0x319A, 0x8139B338},
	{0x319B, 0x8139B339},
	{0x319C, 0x8139B430},
	{0x319D, 0x8139B431},
	{0x319E, 0x8139B432},
	{0x319F, 0x8139B433},
	{0x31A0, 0x8139B434},
	{0x31A1, 0x8139B435},
	{0x31A2, 0x8139B436},
	{0x31A3, 0x8139B437},
	{0x31A4, 0x8139B438},
	{0x31A5, 0x8139B439},
	{0x31A6, 0x8139B530},
	{0x31A7, 0x8139B531},
	{0x31A8, 0x8139B532},
	{0x31A9, 0x8139B533},
	{0x31AA, 0x8139B534},
	{0x31AB, 0x8139B535},
	{0x31AC, 0x8139B536},
	{0x31AD, 0x8139B537},
	{0x31AE, 0x8139B538},
	{0x31AF, 0x8139B539},
	{0x31B0, 0x8139B630},
	{0x31B1, 0x8139B631},
	{0x31B2, 0x8139B632},
	{0x31B3, 0x8139B633},
	{0x31B4, 0x8139B634},
	{0x31B5, 0x8139B635},
	{0x31B6, 0x8139B636},
	{0x31B7, 0x8139B637},
	{0x31B8, 0x8139B638},
	{0x31B9, 0x8139B639},
	{0x31BA, 0x8139B730},
	{0x31BB, 0x8139B731},
	{0x31BC, 0x8139B732},
	{0x31BD, 0x8139B733},
	{0x31BE, 0x8139B734},
	{0x31BF, 0x8139B735},
	{0x31C0, 0x8139B736},
	{0x31C1, 0x8139B737},
	{0x31C2, 0x8139B738},
	{0x31C3, 0x8139B739},
	{0x31C4, 0x8139B830},
	{0x31C5, 0x8139B831},
	{0x31C6, 0x8139B832},
	{0x31C7, 0x8139B833},
	{0x31C8, 0x8139B834},
	{0x31C9, 0x8139B835},
	{0x31CA, 0x8139B836},
	{0x31CB, 0x8139B837},
	{0x31CC, 0x8139B838},
	{0x31CD, 0x8139B839},
	{0x31CE, 0x8139B930},
	{0x31CF, 0x8139B931},
	{0x31D0, 0x8139B932},
	{0x31D1, 0x8139B933},
	{0x31D2, 0x8139B934},
	{0x31D3, 0x8139B935},
	{0x31D4, 0x8139B936},
	{0x31D5, 0x8139B937},
	{0x31D6, 0x8139B938},
	{0x31D7, 0x8139B939},
	{0x31D8, 0x8139BA30},
	{0x31D9, 0x8139BA31},
	{0x31DA, 0x8139BA32},
	{0x31DB, 0x8139BA33},
	{0x31DC, 0x8139BA34},
	{0x31DD, 0x8139BA35},
	{0x31DE, 0x8139BA36},
	{0x31DF, 0x8139BA37},
	{0x31E0, 0x8139BA38},
	{0x31E1, 0x8139BA39},
	{0x31E2, 0x8139BB30},
	{0x31E3, 0x8139BB31},
	{0x31E4, 0x8139BB32},
	{0x31E5, 0x8139BB33},
	{0x31E6, 0x8139BB34},
	{0x31E7, 0x8139BB35},
	{0x31E8, 0x8139BB36},
	{0x31E9, 0x8139BB37},
	{0x31EA, 0x8139BB38},
	{0x31EB, 0x8139BB39},
	{0x31EC, 0x8139BC30},
	{0x31ED, 0x8139BC31},
	{0x31EE, 0x8139BC32},
	{0x31EF, 0x8139BC33},
	{0x31F0, 0x8139BC34},
	{0x31F1, 0x8139BC35},
	{0x31F2, 0x8139BC36},
	{0x31F3, 0x8139BC37},
	{0x31F4, 0x8139BC38},
	{0x31F5, 0x8139BC39},
	{0x31F6, 0x8139BD30},
	{0x31F7, 0x8139BD31},
	{0x31F8, 0x8139BD32},
	{0x31F9, 0x8139BD33},
	{0x31FA, 0x8139BD34},
	{0x31FB, 0x8139BD35},
	{0x31FC, 0x8139BD36},
	{0x31FD, 0x8139BD37},
	{0x31FE, 0x8139BD38},
	{0x31FF, 0x8139BD39},
	{0x3200, 0x8139BE30},
	{0x3201, 0x8139BE31},
	{0x3202, 0x8139BE32},
	{0x3203, 0x8139BE33},
	{0x3204, 0x8139BE34},
	{0x3205, 0x8139BE35},
	{0x3206, 0x8139BE36},
	{0x3207, 0x8139BE37},
	{0x3208, 0x8139BE38},
	{0x3209, 0x8139BE39},
	{0x320A, 0x8139BF30},
	{0x320B, 0x8139BF31},
	{0x320C, 0x8139BF32},
	{0x320D, 0x8139BF33},
	{0x320E, 0x8139BF34},
	{0x320F, 0x8139BF35},
	{0x3210, 0x8139BF36},
	{0x3211, 0x8139BF37},
	{0x3212, 0x8139BF38},
	{0x3213, 0x8139BF39},
	{0x3214, 0x8139C030},
	{0x3215, 0x8139C031},
	{0x3216, 0x8139C032},
	{0x3217, 0x8139C033},
	{0x3218, 0x8139C034},
	{0x3219, 0x8139C035},
	{0x321A, 0x8139C036},
	{0x321B, 0x8139C037},
	{0x321C, 0x8139C038},
	{0x321D, 0x8139C039},
	{0x321E, 0x8139C130},
	{0x321F, 0x8139C131},
	{0x322A, 0x8139C132},
	{0x322B, 0x8139C133},
	{0x322C, 0x8139C134},
	{0x322D, 0x8139C135},
	{0x322E, 0x8139C136},
	{0x322F, 0x8139C137},
	{0x3230, 0x8139C138},
	{0x3232, 0x8139C139},
	{0x3233, 0x8139C230},
	{0x3234, 0x8139C231},
	{0x3235, 0x8139C232},
	{0x3236, 0x8139C233},
	{0x3237, 0x8139C234},
	{0x3238, 0x8139C235},
	{0x3239, 0x8139C236},
	{0x323A, 0x8139C237},
	{0x323B, 0x8139C238},
	{0x323C, 0x8139C239},
	{0x323D, 0x8139C330},
	{0x323E, 0x8139C331},
	{0x323F, 0x8139C332},
	{0x3240, 0x8139C333},
	{0x3241, 0x8139C334},
	{0x3242, 0x8139C335},
	{0x3243, 0x8139C336},
	{0x3244, 0x8139C337},
	{0x3245, 0x8139C338},
	{0x3246, 0x8139C339},
	{0x3247, 0x8139C430},
	{0x3248, 0x8139C431},
	{0x3249, 0x8139C432},
	{0x324A, 0x8139C433},
	{0x324B, 0x8139C434},
	{0x324C, 0x8139C435},
	{0x324D, 0x8139C436},
	{0x324E, 0x8139C437},
	{0x324F, 0x8139C438},
	{0x3250, 0x8139C439},
	{0x3251, 0x8139C530},
	{0x3252, 0x8139C531},
	{0x3253, 0x8139C532},
	{0x3254, 0x8139C533},
	{0x3255, 0x8139C534},
	{0x3256, 0x8139C535},
	{0x3257, 0x8139C536},
	{0x3258, 0x8139C537},
	{0x3259, 0x8139C538},
	{0x325A, 0x8139C539},
	{0x325B, 0x8139C630},
	{0x325C, 0x8139C631},
	{0x325D, 0x8139C632},
	{0x325E, 0x8139C633},
	{0x325F, 0x8139C634},
	{0x3260, 0x8139C635},
	{0x3261, 0x8139C636},
	{0x3262, 0x8139C637},
	{0x3263, 0x8139C638},
	{0x3264, 0x8139C639},
	{0x3265, 0x8139C730},
	{0x3266, 0x8139C731},
	{0x3267, 0x8139C732},
	{0x3268, 0x8139C733},
	{0x3269, 0x8139C734},
	{0x326A, 0x8139C735},
	{0x326B, 0x8139C736},
	{0x326C, 0x8139C737},
	{0x326D, 0x8139C738},
	{0x326E, 0x8139C739},
	{0x326F, 0x8139C830},
	{0x3270, 0x8139C831},
	{0x3271, 0x8139C832},
	{0x3272, 0x8139C833},
	{0x3273, 0x8139C834},
	{0x3274, 0x8139C835},
	{0x3275, 0x8139C836},
	{0x3276, 0x8139C837},
	{0x3277, 0x8139C838},
	{0x3278, 0x8139C839},
	{0x3279, 0x8139C930},
	{0x327A, 0x8139C931},
	{0x327B, 0x8139C932},
	{0x327C, 0x8139C933},
	{0x327D, 0x8139C934},
	{0x327E, 0x8139C935},
	{0x327F, 0x8139C936},
	{0x3280, 0x8139C937},
	{0x3281, 0x8139C938},
	{0x3282, 0x8139C939},
	{0x3283, 0x8139CA30},
	{0x3284, 0x8139CA31},
	{0x3285, 0x8139CA32},
	{0x3286, 0x8139CA33},
	{0x3287, 0x8139CA34},
	{0x3288, 0x8139CA35},
	{0x3289, 0x8139CA36},
	{0x328A, 0x8139CA37},
	{0x328B, 0x8139CA38},
	{0x328C, 0x8139CA39},
	{0x328D, 0x8139CB30},
	{0x328E, 0x8139CB31},
	{0x328F, 0x8139CB32},
	{0x3290, 0x8139CB33},
	{0x3291, 0x8139CB34},
	{0x3292, 0x8139CB35},
	{0x3293, 0x8139CB36},
	{0x3294, 0x8139CB37},
	{0x3295, 0x8139CB38},
	{0x3296, 0x8139CB39},
	{0x3297, 0x8139CC30},
	{0x3298, 0x8139CC31},
	{0x3299, 0x8139CC32},
	{0x329A, 0x8139CC33},
	{0x329B, 0x8139CC34},
	{0x329C, 0x8139CC35},
	{0x329D, 0x8139CC36},
	{0x329E, 0x8139CC37},
	{0x329F, 0x8139CC38},
	{0x32A0, 0x8139CC39},
	{0x32A1, 0x8139CD30},
	{0x32A2, 0x8139CD31},
	{0x32A4, 0x8139CD32},
	{0x32A5, 0x8139CD33},
	{0x32A6, 0x8139CD34},
	{0x32A7, 0x8139CD35},
	{0x32A8, 0x8139CD36},
	{0x32A9, 0x8139CD37},
	{0x32AA, 0x8139CD38},
	{0x32AB, 0x8139CD39},
	{0x32AC, 0x8139CE30},
	{0x32AD, 0x8139CE31},
	{0x32AE, 0x8139CE32},
	{0x32AF, 0x8139CE33},
	{0x32B0, 0x8139CE34},
	{0x32B1, 0x8139CE35},
	{0x32B2, 0x8139CE36},
	{0x32B3, 0x8139CE37},
	{0x32B4, 0x8139CE38},
	{0x32B5, 0x8139CE39},
	{0x32B6, 0x8139CF30},
	{0x32B7, 0x8139CF31},
	{0x32B8, 0x8139CF32},
	{0x32B9, 0x8139CF33},
	{0x32BA, 0x8139CF34},
	{0x32BB, 0x8139CF35},
	{0x32BC, 0x8139CF36},
	{0x32BD, 0x8139CF37},
	{0x32BE, 0x8139CF38},
	{0x32BF, 0x8139CF39},
	{0x32C0, 0x8139D030},
	{0x32C1, 0x8139D031},
	{0x32C2, 0x8139D032},
	{0x32C3, 0x8139D033},
	{0x32C4, 0x8139D034},
	{0x32C5, 0x8139D035},
	{0x32C6, 0x8139D036},
	{0x32C7, 0x8139D037},
	{0x32C8, 0x8139D038},
	{0x32C9, 0x8139D039},
	{0x32CA, 0x8139D130},
	{0x32CB, 0x8139D131},
	{0x32CC, 0x8139D132},
	{0x32CD, 0x8139D133},
	{0x32CE, 0x8139D134},
	{0x32CF, 0x8139D135},
	{0x32D0, 0x8139D136},
	{0x32D1, 0x8139D137},
	{0x32D2, 0x8139D138},
	{0x32D3, 0x8139D139},
	{0x32D4, 0x8139D230},
	{0x32D5, 0x8139D231},
	{0x32D6, 0x8139D232},
	{0x32D7, 0x8139D233},
	{0x32D8, 0x8139D234},
	{0x32D9, 0x8139D235},
	{0x32DA, 0x8139D236},
	{0x32DB, 0x8139D237},
	{0x32DC, 0x8139D238},
	{0x32DD, 0x8139D239},
	{0x32DE, 0x8139D330},
	{0x32DF, 0x8139D331},
	{0x32E0, 0x8139D332},
	{0x32E1, 0x8139D333},
	{0x32E2, 0x8139D334},
	{0x32E3, 0x8139D335},
	{0x32E4, 0x8139D336},
	{0x32E5, 0x8139D337},
	{0x32E6, 0x8139D338},
	{0x32E7, 0x8139D339},
	{0x32E8, 0x8139D430},
	{0x32E9, 0x8139D431},
	{0x32EA, 0x8139D432},
	{0x32EB, 0x8139D433},
	{0x32EC, 0x8139D434},
	{0x32ED, 0x8139D435},
	{0x32EE, 0x8139D436},
	{0x32EF, 0x8139D437},
	{0x32F0, 0x8139D438},
	{0x32F1, 0x8139D439},
	{0x32F2, 0x8139D530},
	{0x32F3, 0x8139D531},
	{0x32F4, 0x8139D532},
	{0x32F5, 0x8139D533},
	{0x32F6, 0x8139D534},
	{0x32F7, 0x8139D535},
	{0x32F8, 0x8139D536},
	{0x32F9, 0x8139D537},
	{0x32FA, 0x8139D538},
	{0x32FB, 0x8139D539},
	{0x32FC, 0x8139D630},
	{0x32FD, 0x8139D631},
	{0x32FE, 0x8139D632},
	{0x32FF, 0x8139D633},
	{0x3300, 0x8139D634},
	{0x3301, 0x8139D635},
	{0x3302, 0x8139D636},
	{0x3303, 0x8139D637},
	{0x3304, 0x8139D638},
	{0x3305, 0x8139D639},
	{0x3306, 0x8139D730},
	{0x3307, 0x8139D731},
	{0x3308, 0x8139D732},
	{0x3309, 0x8139D733},
	{0x330A, 0x8139D734},
	{0x330B, 0x8139D735},
	{0x330C, 0x8139D736},
	{0x330D, 0x8139D737},
	{0x330E, 0x8139D738},
	{0x330F, 0x8139D739},
	{0x3310, 0x8139D830},
	{0x3311, 0x8139D831},
	{0x3312, 0x8139D832},
	{0x3313, 0x8139D833},
	{0x3314, 0x8139D834},
	{0x3315, 0x8139D835},
	{0x3316, 0x8139D836},
	{0x3317, 0x8139D837},
	{0x3318, 0x8139D838},
	{0x3319, 0x8139D839},
	{0x331A, 0x8139D930},
	{0x331B, 0x8139D931},
	{0x331C, 0x8139D932},
	{0x331D, 0x8139D933},
	{0x331E, 0x8139D934},
	{0x331F, 0x8139D935},
	{0x3320, 0x8139D936},
	{0x3321, 0x8139D937},
	{0x3322, 0x8139D938},
	{0x3323, 0x8139D939},
	{0x3324, 0x8139DA30},
	{0x3325, 0x8139DA31},
	{0x3326, 0x8139DA32},
	{0x3327, 0x8139DA33},
	{0x3328, 0x8139DA34},
	{0x3329, 0x8139DA35},
	{0x332A, 0x8139DA36},
	{0x332B, 0x8139DA37},
	{0x332C, 0x8139DA38},
	{0x332D, 0x8139DA39},
	{0x332E, 0x8139DB30},
	{0x332F, 0x8139DB31},
	{0x3330, 0x8139DB32},
	{0x3331, 0x8139DB33},
	{0x3332, 0x8139DB34},
	{0x3333, 0x8139DB35},
	{0x3334, 0x8139DB36},
	{0x3335, 0x8139DB37},
	{0x3336, 0x8139DB38},
	{0x3337, 0x8139DB39},
	{0x3338, 0x8139DC30},
	{0x3339, 0x8139DC31},
	{0x333A, 0x8139DC32},
	{0x333B, 0x8139DC33},
	{0x333C, 0x8139DC34},
	{0x333D, 0x8139DC35},
	{0x333E, 0x8139DC36},
	{0x333F, 0x8139DC37},
	{0x3340, 0x8139DC38},
	{0x3341, 0x8139DC39},
	{0x3342, 0x8139DD30},
	{0x3343, 0x8139DD31},
	{0x3344, 0x8139DD32},
	{0x3345, 0x8139DD33},
	{0x3346, 0x8139DD34},
	{0x3347, 0x8139DD35},
	{0x3348, 0x8139DD36},
	{0x3349, 0x8139DD37},
	{0x334A, 0x8139DD38},
	{0x334B, 0x8139DD39},
	{0x334C, 0x8139DE30},
	{0x334D, 0x8139DE31},
	{0x334E, 0x8139DE32},
	{0x334F, 0x8139DE33},
	{0x3350, 0x8139DE34},
	{0x3351, 0x8139DE35},
	{0x3352, 0x8139DE36},
	{0x3353, 0x8139DE37},
	{0x3354, 0x8139DE38},
	{0x3355, 0x8139DE39},
	{0x3356, 0x8139DF30},
	{0x3357, 0x8139DF31},
	{0x3358, 0x8139DF32},
	{0x3359, 0x8139DF33},
	{0x335A, 0x8139DF34},
	{0x335B, 0x8139DF35},
	{0x335C, 0x8139DF36},
	{0x335D, 0x8139DF37},
	{0x335E, 0x8139DF38},
	{0x335F, 0x8139DF39},
	{0x3360, 0x8139E030},
	{0x3361, 0x8139E031},
	{0x3362, 0x8139E032},
	{0x3363, 0x8139E033},
	{0x3364, 0x8139E034},
	{0x3365, 0x8139E035},
	{0x3366, 0x8139E036},
	{0x3367, 0x8139E037},
	{0x3368, 0x8139E038},
	{0x3369, 0x8139E039},
	{0x336A, 0x8139E130},
	{0x336B, 0x8139E131},
	{0x336C, 0x8139E132},
	{0x336D, 0x8139E133},
	{0x336E, 0x8139E134},
	{0x336F, 0x8139E135},
	{0x3370, 0x8139E136},
	{0x3371, 0x8139E137},
	{0x3372, 0x8139E138},
	{0x3373, 0x8139E139},
	{0x3374, 0x8139E230},
	{0x3375, 0x8139E231},
	{0x3376, 0x8139E232},
	{0x3377, 0x8139E233},
	{0x3378, 0x8139E234},
	{0x3379, 0x8139E235},
	{0x337A, 0x8139E236},
	{0x337B, 0x8139E237},
	{0x337C, 0x8139E238},
	{0x337D, 0x8139E239},
	{0x337E, 0x8139E330},
	{0x337F, 0x8139E331},
	{0x3380, 0x8139E332},
	{0x3381, 0x8139E333},
	{0x3382, 0x8139E334},
	{0x3383, 0x8139E335},
	{0x3384, 0x8139E336},
	{0x3385, 0x8139E337},
	{0x3386, 0x8139E338},
	{0x3387, 0x8139E339},
	{0x3388, 0x8139E430},
	{0x3389, 0x8139E431},
	{0x338A, 0x8139E432},
	{0x338B, 0x8139E433},
	{0x338C, 0x8139E434},
	{0x338D, 0x8139E435},
	{0x3390, 0x8139E436},
	{0x3391, 0x8139E437},
	{0x3392, 0x8139E438},
	{0x3393, 0x8139E439},
	{0x3394, 0x8139E530},
	{0x3395, 0x8139E531},
	{0x3396, 0x8139E532},
	{0x3397, 0x8139E533},
	{0x3398, 0x8139E534},
	{0x3399, 0x8139E535},
	{0x339A, 0x8139E536},
	{0x339B, 0x8139E537},
	{0x339F, 0x8139E538},
	{0x33A0, 0x8139E539},
	{0x33A2, 0x8139E630},
	{0x33A3, 0x8139E631},
	{0x33A4, 0x8139E632},
	{0x33A5, 0x8139E633},
	{0x33A6, 0x8139E634},
	{0x33A7, 0x8139E635},
	{0x33A8, 0x8139E636},
	{0x33A9, 0x8139E637},
	{0x33AA, 0x8139E638},
	{0x33AB, 0x8139E639},
	{0x33AC, 0x8139E730},
	{0x33AD, 0x8139E731},
	{0x33AE, 0x8139E732},
	{0x33AF, 0x8139E733},
	{0x33B0, 0x8139E734},
	{0x33B1, 0x8139E735},
	{0x33B2, 0x8139E736},
	{0x33B3, 0x8139E737},
	{0x33B4, 0x8139E738},
	{0x33B5, 0x8139E739},
	{0x33B6, 0x8139E830},
	{0x33B7, 0x8139E831},
	{0x33B8, 0x8139E832},
	{0x33B9, 0x8139E833},
	{0x33BA, 0x8139E834},
	{0x33BB, 0x8139E835},
	{0x33BC, 0x8139E836},
	{0x33BD, 0x8139E837},
	{0x33BE, 0x8139E838},
	{0x33BF, 0x8139E839},
	{0x33C0, 0x8139E930},
	{0x33C1, 0x8139E931},
	{0x33C2, 0x8139E932},
	{0x33C3, 0x8139E933},
	{0x33C5, 0x8139E934},
	{0x33C6, 0x8139E935},
	{0x33C7, 0x8139E936},
	{0x33C8, 0x8139E937},
	{0x33C9, 0x8139E938},
	{0x33CA, 0x8139E939},
	{0x33CB, 0x8139EA30},
	{0x33CC, 0x8139EA31},
	{0x33CD, 0x8139EA32},
	{0x33CF, 0x8139EA33},
	{0x33D0, 0x8139EA34},
	{0x33D3, 0x8139EA35},
	{0x33D4, 0x8139EA36},
	{0x33D6, 0x8139EA37},
	{0x33D7, 0x8139EA38},
	{0x33D8, 0x8139EA39},
	{0x33D9, 0x8139EB30},
	{0x33DA, 0x8139EB31},
	{0x33DB, 0x8139EB32},
	{0x33DC, 0x8139EB33},
	{0x33DD, 0x8139EB34},
	{0x33DE, 0x8139EB35},
	{0x33DF, 0x8139EB36},
	{0x33E0, 0x8139EB37},
	{0x33E1, 0x8139EB38},
	{0x33E2, 0x8139EB39},
	{0x33E3, 0x8139EC30},
	{0x33E4, 0x8139EC31},
	{0x33E5, 0x8139EC32},
	{0x33E6, 0x8139EC33},
	{0x33E7, 0x8139EC34},
	{0x33E8, 0x8139EC35},
	{0x33E9, 0x8139EC36},
	{0x33EA, 0x8139EC37},
	{0x33EB, 0x8139EC38},
	{0x33EC, 0x8139EC39},
	{0x33ED, 0x8139ED30},
	{0x33EE, 0x8139ED31},
	{0x33EF, 0x8139ED32},
	{0x33F0, 0x8139ED33},
	{0x33F1, 0x8139ED34},
	{0x33F2, 0x8139ED35},
	{0x33F3, 0x8139ED36},
	{0x33F4, 0x8139ED37},
	{0x33F5, 0x8139ED38},
	{0x33F6, 0x8139ED39},
	{0x33F7, 0x8139EE30},
	{0x33F8, 0x8139EE31},
	{0x33F9, 0x8139EE32},
	{0x33FA, 0x8139EE33},
	{0x33FB, 0x8139EE34},
	{0x33FC, 0x8139EE35},
	{0x33FD, 0x8139EE36},
	{0x33FE, 0x8139EE37},
	{0x33FF, 0x8139EE38},
	{0x3400, 0x8139EE39},
	{0x3401, 0x8139EF30},
	{0x3402, 0x8139EF31},
	{0x3403, 0x8139EF32},
	{0x3404, 0x8139EF33},
	{0x3405, 0x8139EF34},
	{0x3406, 0x8139EF35},
	{0x3407, 0x8139EF36},
	{0x3408, 0x8139EF37},
	{0x3409, 0x8139EF38},
	{0x340A, 0x8139EF39},
	{0x340B, 0x8139F030},
	{0x340C, 0x8139F031},
	{0x340D, 0x8139F032},
	{0x340E, 0x8139F033},
	{0x340F, 0x8139F034},
	{0x3410, 0x8139F035},
	{0x3411, 0x8139F036},
	{0x3412, 0x8139F037},
	{0x3413, 0x8139F038},
	{0x3414, 0x8139F039},
	{0x3415, 0x8139F130},
	{0x3416, 0x8139F131},
	{0x3417, 0x8139F132},
	{0x3418, 0x8139F133},
	{0x3419, 0x8139F134},
	{0x341A, 0x8139F135},
	{0x341B, 0x8139F136},
	{0x341C, 0x8139F137},
	{0x341D, 0x8139F138},
	{0x341E, 0x8139F139},
	{0x341F, 0x8139F230},
	{0x3420, 0x8139F231},
	{0x3421, 0x8139F232},
	{0x3422, 0x8139F233},
	{0x3423, 0x8139F234},
	{0x3424, 0x8139F235},
	{0x3425, 0x8139F236},
	{0x3426, 0x8139F237},
	{0x3427, 0x8139F238},
	{0x3428, 0x8139F239},
	{0x3429, 0x8139F330},
	{0x342A, 0x8139F331},
	{0x342B, 0x8139F332},
	{0x342C, 0x8139F333},
	{0x342D, 0x8139F334},
	{0x342E, 0x8139F335},
	{0x342F, 0x8139F336},
	{0x3430, 0x8139F337},
	{0x3431, 0x8139F338},
	{0x3432, 0x8139F339},
	{0x3433, 0x8139F430},
	{0x3434, 0x8139F431},
	{0x3435, 0x8139F432},
	{0x3436, 0x8139F433},
	{0x3437, 0x8139F434},
	{0x3438, 0x8139F435},
	{0x3439, 0x8139F436},
	{0x343A, 0x8139F437},
	{0x343B, 0x8139F438},
	{0x343C, 0x8139F439},
	{0x343D, 0x8139F530},
	{0x343E, 0x8139F531},
	{0x343F, 0x8139F532},
	{0x3440, 0x8139F533},
	{0x3441, 0x8139F534},
	{0x3442, 0x8139F535},
	{0x3443, 0x8139F536},
	{0x3444, 0x8139F537},
	{0x3445, 0x8139F538},
	{0x3446, 0x8139F539},
	{0x3448, 0x8139F630},
	{0x3449, 0x8139F631},
	{0x344A, 0x8139F632},
	{0x344B, 0x8139F633},
	{0x344C, 0x8139F634},
	{0x344D, 0x8139F635},
	{0x344E, 0x8139F636},
	{0x344F, 0x8139F637},
	{0x3450, 0x8139F638},
	{0x3451, 0x8139F639},
	{0x3452, 0x8139F730},
	{0x3453, 0x8139F731},
	{0x3454, 0x8139F732},
	{0x3455, 0x8139F733},
	{0x3456, 0x8139F734},
	{0x3457, 0x8139F735},
	{0x3458, 0x8139F736},
	{0x3459, 0x8139F737},
	{0x345A, 0x8139F738},
	{0x345B, 0x8139F739},
	{0x345C, 0x8139F830},
	{0x345D, 0x8139F831},
	{0x345E, 0x8139F832},
	{0x345F, 0x8139F833},
	{0x3460, 0x8139F834},
	{0x3461, 0x8139F835},
	{0x3462, 0x8139F836},
	{0x3463, 0x8139F837},
	{0x3464, 0x8139F838},
	{0x3465, 0x8139F839},
	{0x3466, 0x8139F930},
	{0x3467, 0x8139F931},
	{0x3468, 0x8139F932},
	{0x3469, 0x8139F933},
	{0x346A, 0x8139F934},
	{0x346B, 0x8139F935},
	{0x346C, 0x8139F936},
	{0x346D, 0x8139F937},
	{0x346E, 0x8139F938},
	{0x346F, 0x8139F939},
	{0x3470, 0x8139FA30},
	{0x3471, 0x8139FA31},
	{0x3472, 0x8139FA32},
	{0x3474, 0x8139FA33},
	{0x3475, 0x8139FA34},
	{0x3476, 0x8139FA35},
	{0x3477, 0x8139FA36},
	{0x3478, 0x8139FA37},
	{0x3479, 0x8139FA38},
	{0x347A, 0x8139FA39},
	{0x347B, 0x8139FB30},
	{0x347C, 0x8139FB31},
	{0x347D, 0x8139FB32},
	{0x347E, 0x8139FB33},
	{0x347F, 0x8139FB34},
	{0x3480, 0x8139FB35},
	{0x3481, 0x8139FB36},
	{0x3482, 0x8139FB37},
	{0x3483, 0x8139FB38},
	{0x3484, 0x8139FB39},
	{0x3485, 0x8139FC30},
	{0x3486, 0x8139FC31},
	{0x3487, 0x8139FC32},
	{0x3488, 0x8139FC33},
	{0x3489, 0x8139FC34},
	{0x348A, 0x8139FC35},
	{0x348B, 0x8139FC36},
	{0x348C, 0x8139FC37},
	{0x348D, 0x8139FC38},
	{0x348E, 0x8139FC39},
	{0x348F, 0x8139FD30},
	{0x3490, 0x8139FD31},
	{0x3491, 0x8139FD32},
	{0x3492, 0x8139FD33},
	{0x3493, 0x8139FD34},
	{0x3494, 0x8139FD35},
	{0x3495, 0x8139FD36},
	{0x3496, 0x8139FD37},
	{0x3497, 0x8139FD38},
	{0x3498, 0x8139FD39},
	{0x3499, 0x8139FE30},
	{0x349A, 0x8139FE31},
	{0x349B, 0x8139FE32},
	{0x349C, 0x8139FE33},
	{0x349D, 0x8139FE34},
	{0x349E, 0x8139FE35},
	{0x349F, 0x8139FE36},
	{0x34A0, 0x8139FE37},
	{0x34A1, 0x8139FE38},
	{0x34A2, 0x8139FE39},
	{0x34A3, 0x82308130},
	{0x34A4, 0x82308131},
	{0x34A5, 0x82308132},
	{0x34A6, 0x82308133},
	{0x34A7, 0x82308134},
	{0x34A8, 0x82308135},
	{0x34A9, 0x82308136},
	{0x34AA, 0x82308137},
	{0x34AB, 0x82308138},
	{0x34AC, 0x82308139},
	{0x34AD, 0x82308230},
	{0x34AE, 0x82308231},
	{0x34AF, 0x82308232},
	{0x34B0, 0x82308233},
	{0x34B1, 0x82308234},
	{0x34B2, 0x82308235},
	{0x34B3, 0x82308236},
	{0x34B4, 0x82308237},
	{0x34B5, 0x82308238},
	{0x34B6, 0x82308239},
	{0x34B7, 0x82308330},
	{0x34B8, 0x82308331},
	{0x34B9, 0x82308332},
	{0x34BA, 0x82308333},
	{0x34BB, 0x82308334},
	{0x34BC, 0x82308335},
	{0x34BD, 0x82308336},
	{0x34BE, 0x82308337},
	{0x34BF, 0x82308338},
	{0x34C0, 0x82308339},
	{0x34C1, 0x82308430},
	{0x34C2, 0x82308431},
	{0x34C3, 0x82308432},
	{0x34C4, 0x82308433},
	{0x34C5, 0x82308434},
	{0x34C6, 0x82308435},
	{0x34C7, 0x82308436},
	{0x34C8, 0x82308437},
	{0x34C9, 0x82308438},
	{0x34CA, 0x82308439},
	{0x34CB, 0x82308530},
	{0x34CC, 0x82308531},
	{0x34CD, 0x82308532},
	{0x34CE, 0x82308533},
	{0x34CF, 0x82308534},
	{0x34D0, 0x82308535},
	{0x34D1, 0x82308536},
	{0x34D2, 0x82308537},
	{0x34D3, 0x82308538},
	{0x34D4, 0x82308539},
	{0x34D5, 0x82308630},
	{0x34D6, 0x82308631},
	{0x34D7, 0x82308632},
	{0x34D8, 0x82308633},
	{0x34D9, 0x82308634},
	{0x34DA, 0x82308635},
	{0x34DB, 0x82308636},
	{0x34DC, 0x82308637},
	{0x34DD, 0x82308638},
	{0x34DE, 0x82308639},
	{0x34DF, 0x82308730},
	{0x34E0, 0x82308731},
	{0x34E1, 0x82308732},
	{0x34E2, 0x82308733},
	{0x34E3, 0x82308734},
	{0x34E4, 0x82308735},
	{0x34E5, 0x82308736},
	{0x34E6, 0x82308737},
	{0x34E7, 0x82308738},
	{0x34E8, 0x82308739},
	{0x34E9, 0x82308830},
	{0x34EA, 0x82308831},
	{0x34EB, 0x82308832},
	{0x34EC, 0x82308833},
	{0x34ED, 0x82308834},
	{0x34EE, 0x82308835},
	{0x34EF, 0x82308836},
	{0x34F0, 0x82308837},
	{0x34F1, 0x82308838},
	{0x34F2, 0x82308839},
	{0x34F3, 0x82308930},
	{0x34F4, 0x82308931},
	{0x34F5, 0x82308932},
	{0x34F6, 0x82308933},
	{0x34F7, 0x82308934},
	{0x34F8, 0x82308935},
	{0x34F9, 0x82308936},
	{0x34FA, 0x82308937},
	{0x34FB, 0x82308938},
	{0x34FC, 0x82308939},
	{0x34FD, 0x82308A30},
	{0x34FE, 0x82308A31},
	{0x34FF, 0x82308A32},
	{0x3500, 0x82308A33},
	{0x3501, 0x82308A34},
	{0x3502, 0x82308A35},
	{0x3503, 0x82308A36},
	{0x3504, 0x82308A37},
	{0x3505, 0x82308A38},
	{0x3506, 0x82308A39},
	{0x3507, 0x82308B30},
	{0x3508, 0x82308B31},
	{0x3509, 0x82308B32},
	{0x350A, 0x82308B33},
	{0x350B, 0x82308B34},
	{0x350C, 0x82308B35},
	{0x350D, 0x82308B36},
	{0x350E, 0x82308B37},
	{0x350F, 0x82308B38},
	{0x3510, 0x82308B39},
	{0x3511, 0x82308C30},
	{0x3512, 0x82308C31},
	{0x3513, 0x82308C32},
	{0x3514, 0x82308C33},
	{0x3515, 0x82308C34},
	{0x3516, 0x82308C35},
	{0x3517, 0x82308C36},
	{0x3518, 0x82308C37},
	{0x3519, 0x82308C38},
	{0x351A, 0x82308C39},
	{0x351B, 0x82308D30},
	{0x351C, 0x82308D31},
	{0x351D, 0x82308D32},
	{0x351E, 0x82308D33},
	{0x351F, 0x82308D34},
	{0x3520, 0x82308D35},
	{0x3521, 0x82308D36},
	{0x3522, 0x82308D37},
	{0x3523, 0x82308D38},
	{0x3524, 0x82308D39},
	{0x3525, 0x82308E30},
	{0x3526, 0x82308E31},
	{0x3527, 0x82308E32},
	{0x3528, 0x82308E33},
	{0x3529, 0x82308E34},
	{0x352A, 0x82308E35},
	{0x352B, 0x82308E36},
	{0x352C, 0x82308E37},
	{0x352D, 0x82308E38},
	{0x352E, 0x82308E39},
	{0x352F, 0x82308F30},
	{0x3530, 0x82308F31},
	{0x3531, 0x82308F32},
	{0x3532, 0x82308F33},
	{0x3533, 0x82308F34},
	{0x3534, 0x82308F35},
	{0x3535, 0x82308F36},
	{0x3536, 0x82308F37},
	{0x3537, 0x82308F38},
	{0x3538, 0x82308F39},
	{0x3539, 0x82309030},
	{0x353A, 0x82309031},
	{0x353B, 0x82309032},
	{0x353C, 0x82309033},
	{0x353D, 0x82309034},
	{0x353E, 0x82309035},
	{0x353F, 0x82309036},
	{0x3540, 0x82309037},
	{0x3541, 0x82309038},
	{0x3542, 0x82309039},
	{0x3543, 0x82309130},
	{0x3544, 0x82309131},
	{0x3545, 0x82309132},
	{0x3546, 0x82309133},
	{0x3547, 0x82309134},
	{0x3548, 0x82309135},
	{0x3549, 0x82309136},
	{0x354A, 0x82309137},
	{0x354B, 0x82309138},
	{0x354C, 0x82309139},
	{0x354D, 0x82309230},
	{0x354E, 0x82309231},
	{0x354F, 0x82309232},
	{0x3550, 0x82309233},
	{0x3551, 0x82309234},
	{0x3552, 0x82309235},
	{0x3553, 0x82309236},
	{0x3554, 0x82309237},
	{0x3555, 0x82309238},
	{0x3556, 0x82309239},
	{0x3557, 0x82309330},
	{0x3558, 0x82309331},
	{0x3559, 0x82309332},
	{0x355A, 0x82309333},
	{0x355B, 0x82309334},
	{0x355C, 0x82309335},
	{0x355D, 0x82309336},
	{0x355E, 0x82309337},
	{0x355F, 0x82309338},
	{0x3560, 0x82309339},
	{0x3561, 0x82309430},
	{0x3562, 0x82309431},
	{0x3563, 0x82309432},
	{0x3564, 0x82309433},
	{0x3565, 0x82309434},
	{0x3566, 0x82309435},
	{0x3567, 0x82309436},
	{0x3568, 0x82309437},
	{0x3569, 0x82309438},
	{0x356A, 0x82309439},
	{0x356B, 0x82309530},
	{0x356C, 0x82309531},
	{0x356D, 0x82309532},
	{0x356E, 0x82309533},
	{0x356F, 0x82309534},
	{0x3570, 0x82309535},
	{0x3571, 0x82309536},
	{0x3572, 0x82309537},
	{0x3573, 0x82309538},
	{0x3574, 0x82309539},
	{0x3575, 0x82309630},
	{0x3576, 0x82309631},
	{0x3577, 0x82309632},
	{0x3578, 0x82309633},
	{0x3579, 0x82309634},
	{0x357A, 0x82309635},
	{0x357B, 0x82309636},
	{0x357C, 0x82309637},
	{0x357D, 0x82309638},
	{0x357E, 0x82309639},
	{0x357F, 0x82309730},
	{0x3580, 0x82309731},
	{0x3581, 0x82309732},
	{0x3582, 0x82309733},
	{0x3583, 0x82309734},
	{0x3584, 0x82309735},
	{0x3585, 0x82309736},
	{0x3586, 0x82309737},
	{0x3587, 0x82309738},
	{0x3588, 0x82309739},
	{0x3589, 0x82309830},
	{0x358A, 0x82309831},
	{0x358B, 0x82309832},
	{0x358C, 0x82309833},
	{0x358D, 0x82309834},
	{0x358E, 0x82309835},
	{0x358F, 0x82309836},
	{0x3590, 0x82309837},
	{0x3591, 0x82309838},
	{0x3592, 0x82309839},
	{0x3593, 0x82309930},
	{0x3594, 0x82309931},
	{0x3595, 0x82309932},
	{0x3596, 0x82309933},
	{0x3597, 0x82309934},
	{0x3598, 0x82309935},
	{0x3599, 0x82309936},
	{0x359A, 0x82309937},
	{0x359B, 0x82309938},
	{0x359C, 0x82309939},
	{0x359D, 0x82309A30},
	{0x359F, 0x82309A31},
	{0x35A0, 0x82309A32},
	{0x35A1, 0x82309A33},
	{0x35A2, 0x82309A34},
	{0x35A3, 0x82309A35},
	{0x35A4, 0x82309A36},
	{0x35A5, 0x82309A37},
	{0x35A6, 0x82309A38},
	{0x35A7, 0x82309A39},
	{0x35A8, 0x82309B30},
	{0x35A9, 0x82309B31},
	{0x35AA, 0x82309B32},
	{0x35AB, 0x82309B33},
	{0x35AC, 0x82309B34},
	{0x35AD, 0x82309B35},
	{0x35AE, 0x82309B36},
	{0x35AF, 0x82309B37},
	{0x35B0, 0x82309B38},
	{0x35B1, 0x82309B39},
	{0x35B2, 0x82309C30},
	{0x35B3, 0x82309C31},
	{0x35B4, 0x82309C32},
	{0x35B5, 0x82309C33},
	{0x35B6, 0x82309C34},
	{0x35B7, 0x82309C35},
	{0x35B8, 0x82309C36},
	{0x35B9, 0x82309C37},
	{0x35BA, 0x82309C38},
	{0x35BB, 0x82309C39},
	{0x35BC, 0x82309D30},
	{0x35BD, 0x82309D31},
	{0x35BE, 0x82309D32},
	{0x35BF, 0x82309D33},
	{0x35C0, 0x82309D34},
	{0x35C1, 0x82309D35},
	{0x35C2, 0x82309D36},
	{0x35C3, 0x82309D37},
	{0x35C4, 0x82309D38},
	{0x35C5, 0x82309D39},
	{0x35C6, 0x82309E30},
	{0x35C7, 0x82309E31},
	{0x35C8, 0x82309E32},
	{0x35C9, 0x82309E33},
	{0x35CA, 0x82309E34},
	{0x35CB, 0x82309E35},
	{0x35CC, 0x82309E36},
	{0x35CD, 0x82309E37},
	{0x35CE, 0x82309E38},
	{0x35CF, 0x82309E39},
	{0x35D0, 0x82309F30},
	{0x35D1, 0x82309F31},
	{0x35D2, 0x82309F32},
	{0x35D3, 0x82309F33},
	{0x35D4, 0x82309F34},
	{0x35D5, 0x82309F35},
	{0x35D6, 0x82309F36},
	{0x35D7, 0x82309F37},
	{0x35D8, 0x82309F38},
	{0x35D9, 0x82309F39},
	{0x35DA, 0x8230A030},
	{0x35DB, 0x8230A031},
	{0x35DC, 0x8230A032},
	{0x35DD, 0x8230A033},
	{0x35DE, 0x8230A034},
	{0x35DF, 0x8230A035},
	{0x35E0, 0x8230A036},
	{0x35E1, 0x8230A037},
	{0x35E2, 0x8230A038},
	{0x35E3, 0x8230A039},
	{0x35E4, 0x8230A130},
	{0x35E5, 0x8230A131},
	{0x35E6, 0x8230A132},
	{0x35E7, 0x8230A133},
	{0x35E8, 0x8230A134},
	{0x35E9, 0x8230A135},
	{0x35EA, 0x8230A136},
	{0x35EB, 0x8230A137},
	{0x35EC, 0x8230A138},
	{0x35ED, 0x8230A139},
	{0x35EE, 0x8230A230},
	{0x35EF, 0x8230A231},
	{0x35F0, 0x8230A232},
	{0x35F1, 0x8230A233},
	{0x35F2, 0x8230A234},
	{0x35F3, 0x8230A235},
	{0x35F4, 0x8230A236},
	{0x35F5, 0x8230A237},
	{0x35F6, 0x8230A238},
	{0x35F7, 0x8230A239},
	{0x35F8, 0x8230A330},
	{0x35F9, 0x8230A331},
	{0x35FA, 0x8230A332},
	{0x35FB, 0x8230A333},
	{0x35FC, 0x8230A334},
	{0x35FD, 0x8230A335},
	{0x35FE, 0x8230A336},
	{0x35FF, 0x8230A337},
	{0x3600, 0x8230A338},
	{0x3601, 0x8230A339},
	{0x3602, 0x8230A430},
	{0x3603, 0x8230A431},
	{0x3604, 0x8230A432},
	{0x3605, 0x8230A433},
	{0x3606, 0x8230A434},
	{0x3607, 0x8230A435},
	{0x3608, 0x8230A436},
	{0x3609, 0x8230A437},
	{0x360A, 0x8230A438},
	{0x360B, 0x8230A439},
	{0x360C, 0x8230A530},
	{0x360D, 0x8230A531},
	{0x360F, 0x8230A532},
	{0x3610, 0x8230A533},
	{0x3611, 0x8230A534},
	{0x3612, 0x8230A535},
	{0x3613, 0x8230A536},
	{0x3614, 0x8230A537},
	{0x3615, 0x8230A538},
	{0x3616, 0x8230A539},
	{0x3617, 0x8230A630},
	{0x3618, 0x8230A631},
	{0x3619, 0x8230A632},
	{0x3919, 0x8230F238},
	{0x391A, 0x8230F239},
	{0x391B, 0x8230F330},
	{0x391C, 0x8230F331},
	{0x391D, 0x8230F332},
	{0x391E, 0x8230F333},
	{0x391F, 0x8230F334},
	{0x3920, 0x8230F335},
	{0x3921, 0x8230F336},
	{0x3922, 0x8230F337},
	{0x3923, 0x8230F338},
	{0x3924, 0x8230F339},
	{0x3925, 0x8230F430},
	{0x3926, 0x8230F431},
	{0x3927, 0x8230F432},
	{0x3928, 0x8230F433},
	{0x3929, 0x8230F434},
	{0x392A, 0x8230F435},
	{0x392B, 0x8230F436},
	{0x392C, 0x8230F437},
	{0x392D, 0x8230F438},
	{0x392E, 0x8230F439},
	{0x392F, 0x8230F530},
	{0x3930, 0x8230F531},
	{0x3931, 0x8230F532},
	{0x3932, 0x8230F533},
	{0x3933, 0x8230F534},
	{0x3934, 0x8230F535},
	{0x3935, 0x8230F536},
	{0x3936, 0x8230F537},
	{0x3937, 0x8230F538},
	{0x3938, 0x8230F539},
	{0x3939, 0x8230F630},
	{0x393A, 0x8230F631},
	{0x393B, 0x8230F632},
	{0x393C, 0x8230F633},
	{0x393D, 0x8230F634},
	{0x393E, 0x8230F635},
	{0x393F, 0x8230F636},
	{0x3940, 0x8230F637},
	{0x3941, 0x8230F638},
	{0x3942, 0x8230F639},
	{0x3943, 0x8230F730},
	{0x3944, 0x8230F731},
	{0x3945, 0x8230F732},
	{0x3946, 0x8230F733},
	{0x3947, 0x8230F734},
	{0x3948, 0x8230F735},
	{0x3949, 0x8230F736},
	{0x394A, 0x8230F737},
	{0x394B, 0x8230F738},
	{0x394C, 0x8230F739},
	{0x394D, 0x8230F830},
	{0x394E, 0x8230F831},
	{0x394F, 0x8230F832},
	{0x3950, 0x8230F833},
	{0x3951, 0x8230F834},
	{0x3952, 0x8230F835},
	{0x3953, 0x8230F836},
	{0x3954, 0x8230F837},
	{0x3955, 0x8230F838},
	{0x3956, 0x8230F839},
	{0x3957, 0x8230F930},
	{0x3958, 0x8230F931},
	{0x3959, 0x8230F932},
	{0x395A, 0x8230F933},
	{0x395B, 0x8230F934},
	{0x395C, 0x8230F935},
	{0x395D, 0x8230F936},
	{0x395E, 0x8230F937},
	{0x395F, 0x8230F938},
	{0x3960, 0x8230F939},
	{0x3961, 0x8230FA30},
	{0x3962, 0x8230FA31},
	{0x3963, 0x8230FA32},
	{0x3964, 0x8230FA33},
	{0x3965, 0x8230FA34},
	{0x3966, 0x8230FA35},
	{0x3967, 0x8230FA36},
	{0x3968, 0x8230FA37},
	{0x3969, 0x8230FA38},
	{0x396A, 0x8230FA39},
	{0x396B, 0x8230FB30},
	{0x396C, 0x8230FB31},
	{0x396D, 0x8230FB32},
	{0x396F, 0x8230FB33},
	{0x3970, 0x8230FB34},
	{0x3971, 0x8230FB35},
	{0x3972, 0x8230FB36},
	{0x3973, 0x8230FB37},
	{0x3974, 0x8230FB38},
	{0x3975, 0x8230FB39},
	{0x3976, 0x8230FC30},
	{0x3977, 0x8230FC31},
	{0x3978, 0x8230FC32},
	{0x3979, 0x8230FC33},
	{0x397A, 0x8230FC34},
	{0x397B, 0x8230FC35},
	{0x397C, 0x8230FC36},
	{0x397D, 0x8230FC37},
	{0x397E, 0x8230FC38},
	{0x397F, 0x8230FC39},
	{0x3980, 0x8230FD30},
	{0x3981, 0x8230FD31},
	{0x3982, 0x8230FD32},
	{0x3983, 0x8230FD33},
	{0x3984, 0x8230FD34},
	{0x3985, 0x8230FD35},
	{0x3986, 0x8230FD36},
	{0x3987, 0x8230FD37},
	{0x3988, 0x8230FD38},
	{0x3989, 0x8230FD39},
	{0x398A, 0x8230FE30},
	{0x398B, 0x8230FE31},
	{0x398C, 0x8230FE32},
	{0x398D, 0x8230FE33},
	{0x398E, 0x8230FE34},
	{0x398F, 0x8230FE35},
	{0x3990, 0x8230FE36},
	{0x3991, 0x8230FE37},
	{0x3992, 0x8230FE38},
	{0x3993, 0x8230FE39},
	{0x3994, 0x82318130},
	{0x3995, 0x82318131},
	{0x3996, 0x82318132},
	{0x3997, 0x82318133},
	{0x3998, 0x82318134},
	{0x3999, 0x82318135},
	{0x399A, 0x82318136},
	{0x399B, 0x82318137},
	{0x399C, 0x82318138},
	{0x399D, 0x82318139},
	{0x399E, 0x82318230},
	{0x399F, 0x82318231},
	{0x39A0, 0x82318232},
	{0x39A1, 0x82318233},
	{0x39A2, 0x82318234},
	{0x39A3, 0x82318235},
	{0x39A4, 0x82318236},
	{0x39A5, 0x82318237},
	{0x39A6, 0x82318238},
	{0x39A7, 0x82318239},
	{0x39A8, 0x82318330},
	{0x39A9, 0x82318331},
	{0x39AA, 0x82318332},
	{0x39AB, 0x82318333},
	{0x39AC, 0x82318334},
	{0x39AD, 0x82318335},
	{0x39AE, 0x82318336},
	{0x39AF, 0x82318337},
	{0x39B0, 0x82318338},
	{0x39B1, 0x82318339},
	{0x39B2, 0x82318430},
	{0x39B3, 0x82318431},
	{0x39B4, 0x82318432},
	{0x39B5, 0x82318433},
	{0x39B6, 0x82318434},
	{0x39B7, 0x82318435},
	{0x39B8, 0x82318436},
	{0x39B9, 0x82318437},
	{0x39BA, 0x82318438},
	{0x39BB, 0x82318439},
	{0x39BC, 0x82318530},
	{0x39BD, 0x82318531},
	{0x39BE, 0x82318532},
	{0x39BF, 0x82318533},
	{0x39C0, 0x82318534},
	{0x39C1, 0x82318535},
	{0x39C2, 0x82318536},
	{0x39C3, 0x82318537},
	{0x39C4, 0x82318538},
	{0x39C5, 0x82318539},
	{0x39C6, 0x82318630},
	{0x39C7, 0x82318631},
	{0x39C8, 0x82318632},
	{0x39C9, 0x82318633},
	{0x39CA, 0x82318634},
	{0x39CB, 0x82318635},
	{0x39CC, 0x82318636},
	{0x39CD, 0x82318637},
	{0x39CE, 0x82318638},
	{0x39D1, 0x82318639},
	{0x39D2, 0x82318730},
	{0x39D3, 0x82318731},
	{0x39D4, 0x82318732},
	{0x39D5, 0x82318733},
	{0x39D6, 0x82318734},
	{0x39D7, 0x82318735},
	{0x39D8, 0x82318736},
	{0x39D9, 0x82318737},
	{0x39DA, 0x82318738},
	{0x39DB, 0x82318739},
	{0x39DC, 0x82318830},
	{0x39DD, 0x82318831},
	{0x39DE, 0x82318832},
	{0x39E0, 0x82318833},
	{0x39E1, 0x82318834},
	{0x39E2, 0x82318835},
	{0x39E3, 0x82318836},
	{0x39E4, 0x82318837},
	{0x39E5, 0x82318838},
	{0x39E6, 0x82318839},
	{0x39E7, 0x82318930},
	{0x39E8, 0x82318931},
	{0x39E9, 0x82318932},
	{0x39EA, 0x82318933},
	{0x39EB, 0x82318934},
	{0x39EC, 0x82318935},
	{0x39ED, 0x82318936},
	{0x39EE, 0x82318937},
	{0x39EF, 0x82318938},
	{0x39F0, 0x82318939},
	{0x39F1, 0x82318A30},
	{0x39F2, 0x82318A31},
	{0x39F3, 0x82318A32},
	{0x39F4, 0x82318A33},
	{0x39F5, 0x82318A34},
	{0x39F6, 0x82318A35},
	{0x39F7, 0x82318A36},
	{0x39F8, 0x82318A37},
	{0x39F9, 0x82318A38},
	{0x39FA, 0x82318A39},
	{0x39FB, 0x82318B30},
	{0x39FC, 0x82318B31},
	{0x39FD, 0x82318B32},
	{0x39FE, 0x82318B33},
	{0x39FF, 0x82318B34},
	{0x3A00, 0x82318B35},
	{0x3A01, 0x82318B36},
	{0x3A02, 0x82318B37},
	{0x3A03, 0x82318B38},
	{0x3A04, 0x82318B39},
	{0x3A05, 0x82318C30},
	{0x3A06, 0x82318C31},
	{0x3A07, 0x82318C32},
	{0x3A08, 0x82318C33},
	{0x3A09, 0x82318C34},
	{0x3A0A, 0x82318C35},
	{0x3A0B, 0x82318C36},
	{0x3A0C, 0x82318C37},
	{0x3A0D, 0x82318C38},
	{0x3A0E, 0x82318C39},
	{0x3A0F, 0x82318D30},
	{0x3A10, 0x82318D31},
	{0x3A11, 0x82318D32},
	{0x3A12, 0x82318D33},
	{0x3A13, 0x82318D34},
	{0x3A14, 0x82318D35},
	{0x3A15, 0x82318D36},
	{0x3A16, 0x82318D37},
	{0x3A17, 0x82318D38},
	{0x3A18, 0x82318D39},
	{0x3A19, 0x82318E30},
	{0x3A1A, 0x82318E31},
	{0x3A1B, 0x82318E32},
	{0x3A1C, 0x82318E33},
	{0x3A1D, 0x82318E34},
	{0x3A1E, 0x82318E35},
	{0x3A1F, 0x82318E36},
	{0x3A20, 0x82318E37},
	{0x3A21, 0x82318E38},
	{0x3A22, 0x82318E39},
	{0x3A23, 0x82318F30},
	{0x3A24, 0x82318F31},
	{0x3A25, 0x82318F32},
	{0x3A26, 0x82318F33},
	{0x3A27, 0x82318F34},
	{0x3A28, 0x82318F35},
	{0x3A29, 0x82318F36},
	{0x3A2A, 0x82318F37},
	{0x3A2B, 0x82318F38},
	{0x3A2C, 0x82318F39},
	{0x3A2D, 0x82319030},
	{0x3A2E, 0x82319031},
	{0x3A2F, 0x82319032},
	{0x3A30, 0x82319033},
	{0x3A31, 0x82319034},
	{0x3A32, 0x82319035},
	{0x3A33, 0x82319036},
	{0x3A34, 0x82319037},
	{0x3A35, 0x82319038},
	{0x3A36, 0x82319039},
	{0x3A37, 0x82319130},
	{0x3A38, 0x82319131},
	{0x3A39, 0x82319132},
	{0x3A3A, 0x82319133},
	{0x3A3B, 0x82319134},
	{0x3A3C, 0x82319135},
	{0x3A3D, 0x82319136},
	{0x3A3E, 0x82319137},
	{0x3A3F, 0x82319138},
	{0x3A40, 0x82319139},
	{0x3A41, 0x82319230},
	{0x3A42, 0x82319231},
	{0x3A43, 0x82319232},
	{0x3A44, 0x82319233},
	{0x3A45, 0x82319234},
	{0x3A46, 0x82319235},
	{0x3A47, 0x82319236},
	{0x3A48, 0x82319237},
	{0x3A49, 0x82319238},
	{0x3A4A, 0x82319239},
	{0x3A4B, 0x82319330},
	{0x3A4C, 0x82319331},
	{0x3A4D, 0x82319332},
	{0x3A4E, 0x82319333},
	{0x3A4F, 0x82319334},
	{0x3A50, 0x82319335},
	{0x3A51, 0x82319336},
	{0x3A52, 0x82319337},
	{0x3A53, 0x82319338},
	{0x3A54, 0x82319339},
	{0x3A55, 0x82319430},
	{0x3A56, 0x82319431},
	{0x3A57, 0x82319432},
	{0x3A58, 0x82319433},
	{0x3A59, 0x82319434},
	{0x3A5A, 0x82319435},
	{0x3A5B, 0x82319436},
	{0x3A5C, 0x82319437},
	{0x3A5D, 0x82319438},
	{0x3A5E, 0x82319439},
	{0x3A5F, 0x82319530},
	{0x3A60, 0x82319531},
	{0x3A61, 0x82319532},
	{0x3A62, 0x82319533},
	{0x3A63, 0x82319534},
	{0x3A64, 0x82319535},
	{0x3A65, 0x82319536},
	{0x3A66, 0x82319537},
	{0x3A67, 0x82319538},
	{0x3A68, 0x82319539},
	{0x3A69, 0x82319630},
	{0x3A6A, 0x82319631},
	{0x3A6B, 0x82319632},
	{0x3A6C, 0x82319633},
	{0x3A6D, 0x82319634},
	{0x3A6E, 0x82319635},
	{0x3A6F, 0x82319636},
	{0x3A70, 0x82319637},
	{0x3A71, 0x82319638},
	{0x3A72, 0x82319639},
	{0x3A74, 0x82319730},
	{0x3A75, 0x82319731},
	{0x3A76, 0x82319732},
	{0x3A77, 0x82319733},
	{0x3A78, 0x82319734},
	{0x3A79, 0x82319735},
	{0x3A7A, 0x82319736},
	{0x3A7B, 0x82319737},
	{0x3A7C, 0x82319738},
	{0x3A7D, 0x82319739},
	{0x3A7E, 0x82319830},
	{0x3A7F, 0x82319831},
	{0x3A80, 0x82319832},
	{0x3A81, 0x82319833},
	{0x3A82, 0x82319834},
	{0x3A83, 0x82319835},
	{0x3A84, 0x82319836},
	{0x3A85, 0x82319837},
	{0x3A86, 0x82319838},
	{0x3A87, 0x82319839},
	{0x3A88, 0x82319930},
	{0x3A89, 0x82319931},
	{0x3A8A, 0x82319932},
	{0x3A8B, 0x82319933},
	{0x3A8C, 0x82319934},
	{0x3A8D, 0x82319935},
	{0x3A8E, 0x82319936},
	{0x3A8F, 0x82319937},
	{0x3A90, 0x82319938},
	{0x3A91, 0x82319939},
	{0x3A92, 0x82319A30},
	{0x3A93, 0x82319A31},
	{0x3A94, 0x82319A32},
	{0x3A95, 0x82319A33},
	{0x3A96, 0x82319A34},
	{0x3A97, 0x82319A35},
	{0x3A98, 0x82319A36},
	{0x3A99, 0x82319A37},
	{0x3A9A, 0x82319A38},
	{0x3A9B, 0x82319A39},
	{0x3A9C, 0x82319B30},
	{0x3A9D, 0x82319B31},
	{0x3A9E, 0x82319B32},
	{0x3A9F, 0x82319B33},
	{0x3AA0, 0x82319B34},
	{0x3AA1, 0x82319B35},
	{0x3AA2, 0x82319B36},
	{0x3AA3, 0x82319B37},
	{0x3AA4, 0x82319B38},
	{0x3AA5, 0x82319B39},
	{0x3AA6, 0x82319C30},
	{0x3AA7, 0x82319C31},
	{0x3AA8, 0x82319C32},
	{0x3AA9, 0x82319C33},
	{0x3AAA, 0x82319C34},
	{0x3AAB, 0x82319C35},
	{0x3AAC, 0x82319C36},
	{0x3AAD, 0x82319C37},
	{0x3AAE, 0x82319C38},
	{0x3AAF, 0x82319C39},
	{0x3AB0, 0x82319D30},
	{0x3AB1, 0x82319D31},
	{0x3AB2, 0x82319D32},
	{0x3AB3, 0x82319D33},
	{0x3AB4, 0x82319D34},
	{0x3AB5, 0x82319D35},
	{0x3AB6, 0x82319D36},
	{0x3AB7, 0x82319D37},
	{0x3AB8, 0x82319D38},
	{0x3AB9, 0x82319D39},
	{0x3ABA, 0x82319E30},
	{0x3ABB, 0x82319E31},
	{0x3ABC, 0x82319E32},
	{0x3ABD, 0x82319E33},
	{0x3ABE, 0x82319E34},
	{0x3ABF, 0x82319E35},
	{0x3AC0, 0x82319E36},
	{0x3AC1, 0x82319E37},
	{0x3AC2, 0x82319E38},
	{0x3AC3, 0x82319E39},
	{0x3AC4, 0x82319F30},
	{0x3AC5, 0x82319F31},
	{0x3AC6, 0x82319F32},
	{0x3AC7, 0x82319F33},
	{0x3AC8, 0x82319F34},
	{0x3AC9, 0x82319F35},
	{0x3ACA, 0x82319F36},
	{0x3ACB, 0x82319F37},
	{0x3ACC, 0x82319F38},
	{0x3ACD, 0x82319F39},
	{0x3ACE, 0x8231A030},
	{0x3ACF, 0x8231A031},
	{0x3AD0, 0x8231A032},
	{0x3AD1, 0x8231A033},
	{0x3AD2, 0x8231A034},
	{0x3AD3, 0x8231A035},
	{0x3AD4, 0x8231A036},
	{0x3AD5, 0x8231A037},
	{0x3AD6, 0x8231A038},
	{0x3AD7, 0x8231A039},
	{0x3AD8, 0x8231A130},
	{0x3AD9, 0x8231A131},
	{0x3ADA, 0x8231A132},
	{0x3ADB, 0x8231A133},
	{0x3ADC, 0x8231A134},
	{0x3ADD, 0x8231A135},
	{0x3ADE, 0x8231A136},
	{0x3ADF, 0x8231A137},
	{0x3AE0, 0x8231A138},
	{0x3AE1, 0x8231A139},
	{0x3AE2, 0x8231A230},
	{0x3AE3, 0x8231A231},
	{0x3AE4, 0x8231A232},
	{0x3AE5, 0x8231A233},
	{0x3AE6, 0x8231A234},
	{0x3AE7, 0x8231A235},
	{0x3AE8, 0x8231A236},
	{0x3AE9, 0x8231A237},
	{0x3AEA, 0x8231A238},
	{0x3AEB, 0x8231A239},
	{0x3AEC, 0x8231A330},
	{0x3AED, 0x8231A331},
	{0x3AEE, 0x8231A332},
	{0x3AEF, 0x8231A333},
	{0x3AF0, 0x8231A334},
	{0x3AF1, 0x8231A335},
	{0x3AF2, 0x8231A336},
	{0x3AF3, 0x8231A337},
	{0x3AF4, 0x8231A338},
	{0x3AF5, 0x8231A339},
	{0x3AF6, 0x8231A430},
	{0x3AF7, 0x8231A431},
	{0x3AF8, 0x8231A432},
	{0x3AF9, 0x8231A433},
	{0x3AFA, 0x8231A434},
	{0x3AFB, 0x8231A435},
	{0x3AFC, 0x8231A436},
	{0x3AFD, 0x8231A437},
	{0x3AFE, 0x8231A438},
	{0x3AFF, 0x8231A439},
	{0x3B00, 0x8231A530},
	{0x3B01, 0x8231A531},
	{0x3B02, 0x8231A532},
	{0x3B03, 0x8231A533},
	{0x3B04, 0x8231A534},
	{0x3B05, 0x8231A535},
	{0x3B06, 0x8231A536},
	{0x3B07, 0x8231A537},
	{0x3B08, 0x8231A538},
	{0x3B09, 0x8231A539},
	{0x3B0A, 0x8231A630},
	{0x3B0B, 0x8231A631},
	{0x3B0C, 0x8231A632},
	{0x3B0D, 0x8231A633},
	{0x3B0E, 0x8231A634},
	{0x3B0F, 0x8231A635},
	{0x3B10, 0x8231A636},
	{0x3B11, 0x8231A637},
	{0x3B12, 0x8231A638},
	{0x3B13, 0x8231A639},
	{0x3B14, 0x8231A730},
	{0x3B15, 0x8231A731},
	{0x3B16, 0x8231A732},
	{0x3B17, 0x8231A733},
	{0x3B18, 0x8231A734},
	{0x3B19, 0x8231A735},
	{0x3B1A, 0x8231A736},
	{0x3B1B, 0x8231A737},
	{0x3B1C, 0x8231A738},
	{0x3B1D, 0x8231A739},
	{0x3B1E, 0x8231A830},
	{0x3B1F, 0x8231A831},
	{0x3B20, 0x8231A832},
	{0x3B21, 0x8231A833},
	{0x3B22, 0x8231A834},
	{0x3B23, 0x8231A835},
	{0x3B24, 0x8231A836},
	{0x3B25, 0x8231A837},
	{0x3B26, 0x8231A838},
	{0x3B27, 0x8231A839},
	{0x3B28, 0x8231A930},
	{0x3B29, 0x8231A931},
	{0x3B2A, 0x8231A932},
	{0x3B2B, 0x8231A933},
	{0x3B2C, 0x8231A934},
	{0x3B2D, 0x8231A935},
	{0x3B2E, 0x8231A936},
	{0x3B2F, 0x8231A937},
	{0x3B30, 0x8231A938},
	{0x3B31, 0x8231A939},
	{0x3B32, 0x8231AA30},
	{0x3B33, 0x8231AA31},
	{0x3B34, 0x8231AA32},
	{0x3B35, 0x8231AA33},
	{0x3B36, 0x8231AA34},
	{0x3B37, 0x8231AA35},
	{0x3B38, 0x8231AA36},
	{0x3B39, 0x8231AA37},
	{0x3B3A, 0x8231AA38},
	{0x3B3B, 0x8231AA39},
	{0x3B3C, 0x8231AB30},
	{0x3B3D, 0x8231AB31},
	{0x3B3E, 0x8231AB32},
	{0x3B3F, 0x8231AB33},
	{0x3B40, 0x8231AB34},
	{0x3B41, 0x8231AB35},
	{0x3B42, 0x8231AB36},
	{0x3B43, 0x8231AB37},
	{0x3B44, 0x8231AB38},
	{0x3B45, 0x8231AB39},
	{0x3B46, 0x8231AC30},
	{0x3B47, 0x8231AC31},
	{0x3B48, 0x8231AC32},
	{0x3B49, 0x8231AC33},
	{0x3B4A, 0x8231AC34},
	{0x3B4B, 0x8231AC35},
	{0x3B4C, 0x8231AC36},
	{0x3B4D, 0x8231AC37},
	{0x3B4F, 0x8231AC38},
	{0x3B50, 0x8231AC39},
	{0x3B51, 0x8231AD30},
	{0x3B52, 0x8231AD31},
	{0x3B53, 0x8231AD32},
	{0x3B54, 0x8231AD33},
	{0x3B55, 0x8231AD34},
	{0x3B56, 0x8231AD35},
	{0x3B57, 0x8231AD36},
	{0x3B58, 0x8231AD37},
	{0x3B59, 0x8231AD38},
	{0x3B5A, 0x8231AD39},
	{0x3B5B, 0x8231AE30},
	{0x3B5C, 0x8231AE31},
	{0x3B5D, 0x8231AE32},
	{0x3B5E, 0x8231AE33},
	{0x3B5F, 0x8231AE34},
	{0x3B60, 0x8231AE35},
	{0x3B61, 0x8231AE36},
	{0x3B62, 0x8231AE37},
	{0x3B63, 0x8231AE38},
	{0x3B64, 0x8231AE39},
	{0x3B65, 0x8231AF30},
	{0x3B66, 0x8231AF31},
	{0x3B67, 0x8231AF32},
	{0x3B68, 0x8231AF33},
	{0x3B69, 0x8231AF34},
	{0x3B6A, 0x8231AF35},
	{0x3B6B, 0x8231AF36},
	{0x3B6C, 0x8231AF37},
	{0x3B6D, 0x8231AF38},
	{0x3B6E, 0x8231AF39},
	{0x3B6F, 0x8231B030},
	{0x3B70, 0x8231B031},
	{0x3B71, 0x8231B032},
	{0x3B72, 0x8231B033},
	{0x3B73, 0x8231B034},
	{0x3B74, 0x8231B035},
	{0x3B75, 0x8231B036},
	{0x3B76, 0x8231B037},
	{0x3B77, 0x8231B038},
	{0x3B78, 0x8231B039},
	{0x3B79, 0x8231B130},
	{0x3B7A, 0x8231B131},
	{0x3B7B, 0x8231B132},
	{0x3B7C, 0x8231B133},
	{0x3B7D, 0x8231B134},
	{0x3B7E, 0x8231B135},
	{0x3B7F, 0x8231B136},
	{0x3B80, 0x8231B137},
	{0x3B81, 0x8231B138},
	{0x3B82, 0x8231B139},
	{0x3B83, 0x8231B230},
	{0x3B84, 0x8231B231},
	{0x3B85, 0x8231B232},
	{0x3B86, 0x8231B233},
	{0x3B87, 0x8231B234},
	{0x3B88, 0x8231B235},
	{0x3B89, 0x8231B236},
	{0x3B8A, 0x8231B237},
	{0x3B8B, 0x8231B238},
	{0x3B8C, 0x8231B239},
	{0x3B8D, 0x8231B330},
	{0x3B8E, 0x8231B331},
	{0x3B8F, 0x8231B332},
	{0x3B90, 0x8231B333},
	{0x3B91, 0x8231B334},
	{0x3B92, 0x8231B335},
	{0x3B93, 0x8231B336},
	{0x3B94, 0x8231B337},
	{0x3B95, 0x8231B338},
	{0x3B96, 0x8231B339},
	{0x3B97, 0x8231B430},
	{0x3B98, 0x8231B431},
	{0x3B99, 0x8231B432},
	{0x3B9A, 0x8231B433},
	{0x3B9B, 0x8231B434},
	{0x3B9C, 0x8231B435},
	{0x3B9D, 0x8231B436},
	{0x3B9E, 0x8231B437},
	{0x3B9F, 0x8231B438},
	{0x3BA0, 0x8231B439},
	{0x3BA1, 0x8231B530},
	{0x3BA2, 0x8231B531},
	{0x3BA3, 0x8231B532},
	{0x3BA4, 0x8231B533},
	{0x3BA5, 0x8231B534},
	{0x3BA6, 0x8231B535},
	{0x3BA7, 0x8231B536},
	{0x3BA8, 0x8231B537},
	{0x3BA9, 0x8231B538},
	{0x3BAA, 0x8231B539},
	{0x3BAB, 0x8231B630},
	{0x3BAC, 0x8231B631},
	{0x3BAD, 0x8231B632},
	{0x3BAE, 0x8231B633},
	{0x3BAF, 0x8231B634},
	{0x3BB0, 0x8231B635},
	{0x3BB1, 0x8231B636},
	{0x3BB2, 0x8231B637},
	{0x3BB3, 0x8231B638},
	{0x3BB4, 0x8231B639},
	{0x3BB5, 0x8231B730},
	{0x3BB6, 0x8231B731},
	{0x3BB7, 0x8231B732},
	{0x3BB8, 0x8231B733},
	{0x3BB9, 0x8231B734},
	{0x3BBA, 0x8231B735},
	{0x3BBB, 0x8231B736},
	{0x3BBC, 0x8231B737},
	{0x3BBD, 0x8231B738},
	{0x3BBE, 0x8231B739},
	{0x3BBF, 0x8231B830},
	{0x3BC0, 0x8231B831},
	{0x3BC1, 0x8231B832},
	{0x3BC2, 0x8231B833},
	{0x3BC3, 0x8231B834},
	{0x3BC4, 0x8231B835},
	{0x3BC5, 0x8231B836},
	{0x3BC6, 0x8231B837},
	{0x3BC7, 0x8231B838},
	{0x3BC8, 0x8231B839},
	{0x3BC9, 0x8231B930},
	{0x3BCA, 0x8231B931},
	{0x3BCB, 0x8231B932},
	{0x3BCC, 0x8231B933},
	{0x3BCD, 0x8231B934},
	{0x3BCE, 0x8231B935},
	{0x3BCF, 0x8231B936},
	{0x3BD0, 0x8231B937},
	{0x3BD1, 0x8231B938},
	{0x3BD2, 0x8231B939},
	{0x3BD3, 0x8231BA30},
	{0x3BD4, 0x8231BA31},
	{0x3BD5, 0x8231BA32},
	{0x3BD6, 0x8231BA33},
	{0x3BD7, 0x8231BA34},
	{0x3BD8, 0x8231BA35},
	{0x3BD9, 0x8231BA36},
	{0x3BDA, 0x8231BA37},
	{0x3BDB, 0x8231BA38},
	{0x3BDC, 0x8231BA39},
	{0x3BDD, 0x8231BB30},
	{0x3BDE, 0x8231BB31},
	{0x3BDF, 0x8231BB32},
	{0x3BE0, 0x8231BB33},
	{0x3BE1, 0x8231BB34},
	{0x3BE2, 0x8231BB35},
	{0x3BE3, 0x8231BB36},
	{0x3BE4, 0x8231BB37},
	{0x3BE5, 0x8231BB38},
	{0x3BE6, 0x8231BB39},
	{0x3BE7, 0x8231BC30},
	{0x3BE8, 0x8231BC31},
	{0x3BE9, 0x8231BC32},
	{0x3BEA, 0x8231BC33},
	{0x3BEB, 0x8231BC34},
	{0x3BEC, 0x8231BC35},
	{0x3BED, 0x8231BC36},
	{0x3BEE, 0x8231BC37},
	{0x3BEF, 0x8231BC38},
	{0x3BF0, 0x8231BC39},
	{0x3BF1, 0x8231BD30},
	{0x3BF2, 0x8231BD31},
	{0x3BF3, 0x8231BD32},
	{0x3BF4, 0x8231BD33},
	{0x3BF5, 0x8231BD34},
	{0x3BF6, 0x8231BD35},
	{0x3BF7, 0x8231BD36},
	{0x3BF8, 0x8231BD37},
	{0x3BF9, 0x8231BD38},
	{0x3BFA, 0x8231BD39},
	{0x3BFB, 0x8231BE30},
	{0x3BFC, 0x8231BE31},
	{0x3BFD, 0x8231BE32},
	{0x3BFE, 0x8231BE33},
	{0x3BFF, 0x8231BE34},
	{0x3C00, 0x8231BE35},
	{0x3C01, 0x8231BE36},
	{0x3C02, 0x8231BE37},
	{0x3C03, 0x8231BE38},
	{0x3C04, 0x8231BE39},
	{0x3C05, 0x8231BF30},
	{0x3C06, 0x8231BF31},
	{0x3C07, 0x8231BF32},
	{0x3C08, 0x8231BF33},
	{0x3C09, 0x8231BF34},
	{0x3C0A, 0x8231BF35},
	{0x3C0B, 0x8231BF36},
	{0x3C0C, 0x8231BF37},
	{0x3C0D, 0x8231BF38},
	{0x3C0E, 0x8231BF39},
	{0x3C0F, 0x8231C030},
	{0x3C10, 0x8231C031},
	{0x3C11, 0x8231C032},
	{0x3C12, 0x8231C033},
	{0x3C13, 0x8231C034},
	{0x3C14, 0x8231C035},
	{0x3C15, 0x8231C036},
	{0x3C16, 0x8231C037},
	{0x3C17, 0x8231C038},
	{0x3C18, 0x8231C039},
	{0x3C19, 0x8231C130},
	{0x3C1A, 0x8231C131},
	{0x3C1B, 0x8231C132},
	{0x3C1C, 0x8231C133},
	{0x3C1D, 0x8231C134},
	{0x3C1E, 0x8231C135},
	{0x3C1F, 0x8231C136},
	{0x3C20, 0x8231C137},
	{0x3C21, 0x8231C138},
	{0x3C22, 0x8231C139},
	{0x3C23, 0x8231C230},
	{0x3C24, 0x8231C231},
	{0x3C25, 0x8231C232},
	{0x3C26, 0x8231C233},
	{0x3C27, 0x8231C234},
	{0x3C28, 0x8231C235},
	{0x3C29, 0x8231C236},
	{0x3C2A, 0x8231C237},
	{0x3C2B, 0x8231C238},
	{0x3C2C, 0x8231C239},
	{0x3C2D, 0x8231C330},
	{0x3C2E, 0x8231C331},
	{0x3C2F, 0x8231C332},
	{0x3C30, 0x8231C333},
	{0x3C31, 0x8231C334},
	{0x3C32, 0x8231C335},
	{0x3C33, 0x8231C336},
	{0x3C34, 0x8231C337},
	{0x3C35, 0x8231C338},
	{0x3C36, 0x8231C339},
	{0x3C37, 0x8231C430},
	{0x3C38, 0x8231C431},
	{0x3C39, 0x8231C432},
	{0x3C3A, 0x8231C433},
	{0x3C3B, 0x8231C434},
	{0x3C3C, 0x8231C435},
	{0x3C3D, 0x8231C436},
	{0x3C3E, 0x8231C437},
	{0x3C3F, 0x8231C438},
	{0x3C40, 0x8231C439},
	{0x3C41, 0x8231C530},
	{0x3C42, 0x8231C531},
	{0x3C43, 0x8231C532},
	{0x3C44, 0x8231C533},
	{0x3C45, 0x8231C534},
	{0x3C46, 0x8231C535},
	{0x3C47, 0x8231C536},
	{0x3C48, 0x8231C537},
	{0x3C49, 0x8231C538},
	{0x3C4A, 0x8231C539},
	{0x3C4B, 0x8231C630},
	{0x3C4C, 0x8231C631},
	{0x3C4D, 0x8231C632},
	{0x3C4E, 0x8231C633},
	{0x3C4F, 0x8231C634},
	{0x3C50, 0x8231C635},
	{0x3C51, 0x8231C636},
	{0x3C52, 0x8231C637},
	{0x3C53, 0x8231C638},
	{0x3C54, 0x8231C639},
	{0x3C55, 0x8231C730},
	{0x3C56, 0x8231C731},
	{0x3C57, 0x8231C732},
	{0x3C58, 0x8231C733},
	{0x3C59, 0x8231C734},
	{0x3C5A, 0x8231C735},
	{0x3C5B, 0x8231C736},
	{0x3C5C, 0x8231C737},
	{0x3C5D, 0x8231C738},
	{0x3C5E, 0x8231C739},
	{0x3C5F, 0x8231C830},
	{0x3C60, 0x8231C831},
	{0x3C61, 0x8231C832},
	{0x3C62, 0x8231C833},
	{0x3C63, 0x8231C834},
	{0x3C64, 0x8231C835},
	{0x3C65, 0x8231C836},
	{0x3C66, 0x8231C837},
	{0x3C67, 0x8231C838},
	{0x3C68, 0x8231C839},
	{0x3C69, 0x8231C930},
	{0x3C6A, 0x8231C931},
	{0x3C6B, 0x8231C932},
	{0x3C6C, 0x8231C933},
	{0x3C6D, 0x8231C934},
	{0x3C6F, 0x8231C935},
	{0x3C70, 0x8231C936},
	{0x3C71, 0x8231C937},
	{0x3C72, 0x8231C938},
	{0x3C73, 0x8231C939},
	{0x3C74, 0x8231CA30},
	{0x3C75, 0x8231CA31},
	{0x3C76, 0x8231CA32},
	{0x3C77, 0x8231CA33},
	{0x3C78, 0x8231CA34},
	{0x3C79, 0x8231CA35},
	{0x3C7A, 0x8231CA36},
	{0x3C7B, 0x8231CA37},
	{0x3C7C, 0x8231CA38},
	{0x3C7D, 0x8231CA39},
	{0x3C7E, 0x8231CB30},
	{0x3C7F, 0x8231CB31},
	{0x3C80, 0x8231CB32},
	{0x3C81, 0x8231CB33},
	{0x3C82, 0x8231CB34},
	{0x3C83, 0x8231CB35},
	{0x3C84, 0x8231CB36},
	{0x3C85, 0x8231CB37},
	{0x3C86, 0x8231CB38},
	{0x3C87, 0x8231CB39},
	{0x3C88, 0x8231CC30},
	{0x3C89, 0x8231CC31},
	{0x3C8A, 0x8231CC32},
	{0x3C8B, 0x8231CC33},
	{0x3C8C, 0x8231CC34},
	{0x3C8D, 0x8231CC35},
	{0x3C8E, 0x8231CC36},
	{0x3C8F, 0x8231CC37},
	{0x3C90, 0x8231CC38},
	{0x3C91, 0x8231CC39},
	{0x3C92, 0x8231CD30},
	{0x3C93, 0x8231CD31},
	{0x3C94, 0x8231CD32},
	{0x3C95, 0x8231CD33},
	{0x3C96, 0x8231CD34},
	{0x3C97, 0x8231CD35},
	{0x3C98, 0x8231CD36},
	{0x3C99, 0x8231CD37},
	{0x3C9A, 0x8231CD38},
	{0x3C9B, 0x8231CD39},
	{0x3C9C, 0x8231CE30},
	{0x3C9D, 0x8231CE31},
	{0x3C9E, 0x8231CE32},
	{0x3C9F, 0x8231CE33},
	{0x3CA0, 0x8231CE34},
	{0x3CA1, 0x8231CE35},
	{0x3CA2, 0x8231CE36},
	{0x3CA3, 0x8231CE37},
	{0x3CA4, 0x8231CE38},
	{0x3CA5, 0x8231CE39},
	{0x3CA6, 0x8231CF30},
	{0x3CA7, 0x8231CF31},
	{0x3CA8, 0x8231CF32},
	{0x3CA9, 0x8231CF33},
	{0x3CAA, 0x8231CF34},
	{0x3CAB, 0x8231CF35},
	{0x3CAC, 0x8231CF36},
	{0x3CAD, 0x8231CF37},
	{0x3CAE, 0x8231CF38},
	{0x3CAF, 0x8231CF39},
	{0x3CB0, 0x8231D030},
	{0x3CB1, 0x8231D031},
	{0x3CB2, 0x8231D032},
	{0x3CB3, 0x8231D033},
	{0x3CB4, 0x8231D034},
	{0x3CB5, 0x8231D035},
	{0x3CB6, 0x8231D036},
	{0x3CB7, 0x8231D037},
	{0x3CB8, 0x8231D038},
	{0x3CB9, 0x8231D039},
	{0x3CBA, 0x8231D130},
	{0x3CBB, 0x8231D131},
	{0x3CBC, 0x8231D132},
	{0x3CBD, 0x8231D133},
	{0x3CBE, 0x8231D134},
	{0x3CBF, 0x8231D135},
	{0x3CC0, 0x8231D136},
	{0x3CC1, 0x8231D137},
	{0x3CC2, 0x8231D138},
	{0x3CC3, 0x8231D139},
	{0x3CC4, 0x8231D230},
	{0x3CC5, 0x8231D231},
	{0x3CC6, 0x8231D232},
	{0x3CC7, 0x8231D233},
	{0x3CC8, 0x8231D234},
	{0x3CC9, 0x8231D235},
	{0x3CCA, 0x8231D236},
	{0x3CCB, 0x8231D237},
	{0x3CCC, 0x8231D238},
	{0x3CCD, 0x8231D239},
	{0x3CCE, 0x8231D330},
	{0x3CCF, 0x8231D331},
	{0x3CD0, 0x8231D332},
	{0x3CD1, 0x8231D333},
	{0x3CD2, 0x8231D334},
	{0x3CD3, 0x8231D335},
	{0x3CD4, 0x8231D336},
	{0x3CD5, 0x8231D337},
	{0x3CD6, 0x8231D338},
	{0x3CD7, 0x8231D339},
	{0x3CD8, 0x8231D430},
	{0x3CD9, 0x8231D431},
	{0x3CDA, 0x8231D432},
	{0x3CDB, 0x8231D433},
	{0x3CDC, 0x8231D434},
	{0x3CDD, 0x8231D435},
	{0x3CDE, 0x8231D436},
	{0x3CDF, 0x8231D437},
	{0x4057, 0x8232AF33},
	{0x4058, 0x8232AF34},
	{0x4059, 0x8232AF35},
	{0x405A, 0x8232AF36},
	{0x405B, 0x8232AF37},
	{0x405C, 0x8232AF38},
	{0x405D, 0x8232AF39},
	{0x405E, 0x8232B030},
	{0x405F, 0x8232B031},
	{0x4060, 0x8232B032},
	{0x4061, 0x8232B033},
	{0x4062, 0x8232B034},
	{0x4063, 0x8232B035},
	{0x4064, 0x8232B036},
	{0x4065, 0x8232B037},
	{0x4066, 0x8232B038},
	{0x4067, 0x8232B039},
	{0x4068, 0x8232B130},
	{0x4069, 0x8232B131},
	{0x406A, 0x8232B132},
	{0x406B, 0x8232B133},
	{0x406C, 0x8232B134},
	{0x406D, 0x8232B135},
	{0x406E, 0x8232B136},
	{0x406F, 0x8232B137},
	{0x4070, 0x8232B138},
	{0x4071, 0x8232B139},
	{0x4072, 0x8232B230},
	{0x4073, 0x8232B231},
	{0x4074, 0x8232B232},
	{0x4075, 0x8232B233},
	{0x4076, 0x8232B234},
	{0x4077, 0x8232B235},
	{0x4078, 0x8232B236},
	{0x4079, 0x8232B237},
	{0x407A, 0x8232B238},
	{0x407B, 0x8232B239},
	{0x407C, 0x8232B330},
	{0x407D, 0x8232B331},
	{0x407E, 0x8232B332},
	{0x407F, 0x8232B333},
	{0x4080, 0x8232B334},
	{0x4081, 0x8232B335},
	{0x4082, 0x8232B336},
	{0x4083, 0x8232B337},
	{0x4084, 0x8232B338},
	{0x4085, 0x8232B339},
	{0x4086, 0x8232B430},
	{0x4087, 0x8232B431},
	{0x4088, 0x8232B432},
	{0x4089, 0x8232B433},
	{0x408A, 0x8232B434},
	{0x408B, 0x8232B435},
	{0x408C, 0x8232B436},
	{0x408D, 0x8232B437},
	{0x408E, 0x8232B438},
	{0x408F, 0x8232B439},
	{0x4090, 0x8232B530},
	{0x4091, 0x8232B531},
	{0x4092, 0x8232B532},
	{0x4093, 0x8232B533},
	{0x4094, 0x8232B534},
	{0x4095, 0x8232B535},
	{0x4096, 0x8232B536},
	{0x4097, 0x8232B537},
	{0x4098, 0x8232B538},
	{0x4099, 0x8232B539},
	{0x409A, 0x8232B630},
	{0x409B, 0x8232B631},
	{0x409C, 0x8232B632},
	{0x409D, 0x8232B633},
	{0x409E, 0x8232B634},
	{0x409F, 0x8232B635},
	{0x40A0, 0x8232B636},
	{0x40A1, 0x8232B637},
	{0x40A2, 0x8232B638},
	{0x40A3, 0x8232B639},
	{0x40A4, 0x8232B730},
	{0x40A5, 0x8232B731},
	{0x40A6, 0x8232B732},
	{0x40A7, 0x8232B733},
	{0x40A8, 0x8232B734},
	{0x40A9, 0x8232B735},
	{0x40AA, 0x8232B736},
	{0x40AB, 0x8232B737},
	{0x40AC, 0x8232B738},
	{0x40AD, 0x8232B739},
	{0x40AE, 0x8232B830},
	{0x40AF, 0x8232B831},
	{0x40B0, 0x8232B832},
	{0x40B1, 0x8232B833},
	{0x40B2, 0x8232B834},
	{0x40B3, 0x8232B835},
	{0x40B4, 0x8232B836},
	{0x40B5, 0x8232B837},
	{0x40B6, 0x8232B838},
	{0x40B7, 0x8232B839},
	{0x40B8, 0x8232B930},
	{0x40B9, 0x8232B931},
	{0x40BA, 0x8232B932},
	{0x40BB, 0x8232B933},
	{0x40BC, 0x8232B934},
	{0x40BD, 0x8232B935},
	{0x40BE, 0x8232B936},
	{0x40BF, 0x8232B937},
	{0x40C0, 0x8232B938},
	{0x40C1, 0x8232B939},
	{0x40C2, 0x8232BA30},
	{0x40C3, 0x8232BA31},
	{0x40C4, 0x8232BA32},
	{0x40C5, 0x8232BA33},
	{0x40C6, 0x8232BA34},
	{0x40C7, 0x8232BA35},
	{0x40C8, 0x8232BA36},
	{0x40C9, 0x8232BA37},
	{0x40CA, 0x8232BA38},
	{0x40CB, 0x8232BA39},
	{0x40CC, 0x8232BB30},
	{0x40CD, 0x8232BB31},
	{0x40CE, 0x8232BB32},
	{0x40CF, 0x8232BB33},
	{0x40D0, 0x8232BB34},
	{0x40D1, 0x8232BB35},
	{0x40D2, 0x8232BB36},
	{0x40D3, 0x8232BB37},
	{0x40D4, 0x8232BB38},
	{0x40D5, 0x8232BB39},
	{0x40D6, 0x8232BC30},
	{0x40D7, 0x8232BC31},
	{0x40D8, 0x8232BC32},
	{0x40D9, 0x8232BC33},
	{0x40DA, 0x8232BC34},
	{0x40DB, 0x8232BC35},
	{0x40DC, 0x8232BC36},
	{0x40DD, 0x8232BC37},
	{0x40DE, 0x8232BC38},
	{0x40DF, 0x8232BC39},
	{0x40E0, 0x8232BD30},
	{0x40E1, 0x8232BD31},
	{0x40E2, 0x8232BD32},
	{0x40E3, 0x8232BD33},
	{0x40E4, 0x8232BD34},
	{0x40E5, 0x8232BD35},
	{0x40E6, 0x8232BD36},
	{0x40E7, 0x8232BD37},
	{0x40E8, 0x8232BD38},
	{0x40E9, 0x8232BD39},
	{0x40EA, 0x8232BE30},
	{0x40EB, 0x8232BE31},
	{0x40EC, 0x8232BE32},
	{0x40ED, 0x8232BE33},
	{0x40EE, 0x8232BE34},
	{0x40EF, 0x8232BE35},
	{0x40F0, 0x8232BE36},
	{0x40F1, 0x8232BE37},
	{0x40F2, 0x8232BE38},
	{0x40F3, 0x8232BE39},
	{0x40F4, 0x8232BF30},
	{0x40F5, 0x8232BF31},
	{0x40F6, 0x8232BF32},
	{0x40F7, 0x8232BF33},
	{0x40F8, 0x8232BF34},
	{0x40F9, 0x8232BF35},
	{0x40FA, 0x8232BF36},
	{0x40FB, 0x8232BF37},
	{0x40FC, 0x8232BF38},
	{0x40FD, 0x8232BF39},
	{0x40FE, 0x8232C030},
	{0x40FF, 0x8232C031},
	{0x4100, 0x8232C032},
	{0x4101, 0x8232C033},
	{0x4102, 0x8232C034},
	{0x4103, 0x8232C035},
	{0x4104, 0x8232C036},
	{0x4105, 0x8232C037},
	{0x4106, 0x8232C038},
	{0x4107, 0x8232C039},
	{0x4108, 0x8232C130},
	{0x4109, 0x8232C131},
	{0x410A, 0x8232C132},
	{0x410B, 0x8232C133},
	{0x410C, 0x8232C134},
	{0x410D, 0x8232C135},
	{0x410E, 0x8232C136},
	{0x410F, 0x8232C137},
	{0x4110, 0x8232C138},
	{0x4111, 0x8232C139},
	{0x4112, 0x8232C230},
	{0x4113, 0x8232C231},
	{0x4114, 0x8232C232},
	{0x4115, 0x8232C233},
	{0x4116, 0x8232C234},
	{0x4117, 0x8232C235},
	{0x4118, 0x8232C236},
	{0x4119, 0x8232C237},
	{0x411A, 0x8232C238},
	{0x411B, 0x8232C239},
	{0x411C, 0x8232C330},
	{0x411D, 0x8232C331},
	{0x411E, 0x8232C332},
	{0x411F, 0x8232C333},
	{0x4120, 0x8232C334},
	{0x4121, 0x8232C335},
	{0x4122, 0x8232C336},
	{0x4123, 0x8232C337},
	{0x4124, 0x8232C338},
	{0x4125, 0x8232C339},
	{0x4126, 0x8232C430},
	{0x4127, 0x8232C431},
	{0x4128, 0x8232C432},
	{0x4129, 0x8232C433},
	{0x412A, 0x8232C434},
	{0x412B, 0x8232C435},
	{0x412C, 0x8232C436},
	{0x412D, 0x8232C437},
	{0x412E, 0x8232C438},
	{0x412F, 0x8232C439},
	{0x4130, 0x8232C530},
	{0x4131, 0x8232C531},
	{0x4132, 0x8232C532},
	{0x4133, 0x8232C533},
	{0x4134, 0x8232C534},
	{0x4135, 0x8232C535},
	{0x4136, 0x8232C536},
	{0x4137, 0x8232C537},
	{0x4138, 0x8232C538},
	{0x4139, 0x8232C539},
	{0x413A, 0x8232C630},
	{0x413B, 0x8232C631},
	{0x413C, 0x8232C632},
	{0x413D, 0x8232C633},
	{0x413E, 0x8232C634},
	{0x413F, 0x8232C635},
	{0x4140, 0x8232C636},
	{0x4141, 0x8232C637},
	{0x4142, 0x8232C638},
	{0x4143, 0x8232C639},
	{0x4144, 0x8232C730},
	{0x4145, 0x8232C731},
	{0x4146, 0x8232C732},
	{0x4147, 0x8232C733},
	{0x4148, 0x8232C734},
	{0x4149, 0x8232C735},
	{0x414A, 0x8232C736},
	{0x414B, 0x8232C737},
	{0x414C, 0x8232C738},
	{0x414D, 0x8232C739},
	{0x414E, 0x8232C830},
	{0x414F, 0x8232C831},
	{0x4150, 0x8232C832},
	{0x4151, 0x8232C833},
	{0x4152, 0x8232C834},
	{0x4153, 0x8232C835},
	{0x4154, 0x8232C836},
	{0x4155, 0x8232C837},
	{0x4156, 0x8232C838},
	{0x4157, 0x8232C839},
	{0x4158, 0x8232C930},
	{0x4159, 0x8232C931},
	{0x415A, 0x8232C932},
	{0x415B, 0x8232C933},
	{0x415C, 0x8232C934},
	{0x415D, 0x8232C935},
	{0x415E, 0x8232C936},
	{0x4338, 0x8232F838},
	{0x4339, 0x8232F839},
	{0x433A, 0x8232F930},
	{0x433B, 0x8232F931},
	{0x433C, 0x8232F932},
	{0x433D, 0x8232F933},
	{0x433E, 0x8232F934},
	{0x433F, 0x8232F935},
	{0x4340, 0x8232F936},
	{0x4341, 0x8232F937},
	{0x4342, 0x8232F938},
	{0x4343, 0x8232F939},
	{0x4344, 0x8232FA30},
	{0x4345, 0x8232FA31},
	{0x4346, 0x8232FA32},
	{0x4347, 0x8232FA33},
	{0x4348, 0x8232FA34},
	{0x4349, 0x8232FA35},
	{0x434A, 0x8232FA36},
	{0x434B, 0x8232FA37},
	{0x434C, 0x8232FA38},
	{0x434D, 0x8232FA39},
	{0x434E, 0x8232FB30},
	{0x434F, 0x8232FB31},
	{0x4350, 0x8232FB32},
	{0x4351, 0x8232FB33},
	{0x4352, 0x8232FB34},
	{0x4353, 0x8232FB35},
	{0x4354, 0x8232FB36},
	{0x4355, 0x8232FB37},
	{0x4356, 0x8232FB38},
	{0x4357, 0x8232FB39},
	{0x4358, 0x8232FC30},
	{0x4359, 0x8232FC31},
	{0x435A, 0x8232FC32},
	{0x435B, 0x8232FC33},
	{0x435C, 0x8232FC34},
	{0x435D, 0x8232FC35},
	{0x435E, 0x8232FC36},
	{0x435F, 0x8232FC37},
	{0x4360, 0x8232FC38},
	{0x4361, 0x8232FC39},
	{0x4362, 0x8232FD30},
	{0x4363, 0x8232FD31},
	{0x4364, 0x8232FD32},
	{0x4365, 0x8232FD33},
	{0x4366, 0x8232FD34},
	{0x4367, 0x8232FD35},
	{0x4368, 0x8232FD36},
	{0x4369, 0x8232FD37},
	{0x436A, 0x8232FD38},
	{0x436B, 0x8232FD39},
	{0x436C, 0x8232FE30},
	{0x436D, 0x8232FE31},
	{0x436E, 0x8232FE32},
	{0x436F, 0x8232FE33},
	{0x4370, 0x8232FE34},
	{0x4371, 0x8232FE35},
	{0x4372, 0x8232FE36},
	{0x4373, 0x8232FE37},
	{0x4374, 0x8232FE38},
	{0x4375, 0x8232FE39},
	{0x4376, 0x82338130},
	{0x4377, 0x82338131},
	{0x4378, 0x82338132},
	{0x4379, 0x82338133},
	{0x437A, 0x82338134},
	{0x437B, 0x82338135},
	{0x437C, 0x82338136},
	{0x437D, 0x82338137},
	{0x437E, 0x82338138},
	{0x437F, 0x82338139},
	{0x4380, 0x82338230},
	{0x4381, 0x82338231},
	{0x4382, 0x82338232},
	{0x4383, 0x82338233},
	{0x4384, 0x82338234},
	{0x4385, 0x82338235},
	{0x4386, 0x82338236},
	{0x4387, 0x82338237},
	{0x4388, 0x82338238},
	{0x4389, 0x82338239},
	{0x438A, 0x82338330},
	{0x438B, 0x82338331},
	{0x438C, 0x82338332},
	{0x438D, 0x82338333},
	{0x438E, 0x82338334},
	{0x438F, 0x82338335},
	{0x4390, 0x82338336},
	{0x4391, 0x82338337},
	{0x4392, 0x82338338},
	{0x4393, 0x82338339},
	{0x4394, 0x82338430},
	{0x4395, 0x82338431},
	{0x4396, 0x82338432},
	{0x4397, 0x82338433},
	{0x4398, 0x82338434},
	{0x4399, 0x82338435},
	{0x439A, 0x82338436},
	{0x439B, 0x82338437},
	{0x439C, 0x82338438},
	{0x439D, 0x82338439},
	{0x439E, 0x82338530},
	{0x439F, 0x82338531},
	{0x43A0, 0x82338532},
	{0x43A1, 0x82338533},
	{0x43A2, 0x82338534},
	{0x43A3, 0x82338535},
	{0x43A4, 0x82338536},
	{0x43A5, 0x82338537},
	{0x43A6, 0x82338538},
	{0x43A7, 0x82338539},
	{0x43A8, 0x82338630},
	{0x43A9, 0x82338631},
	{0x43AA, 0x82338632},
	{0x43AB, 0x82338633},
	{0x43AD, 0x82338634},
	{0x43AE, 0x82338635},
	{0x43AF, 0x82338636},
	{0x43B0, 0x82338637},
	{0x43B2, 0x82338638},
	{0x43B3, 0x82338639},
	{0x43B4, 0x82338730},
	{0x43B5, 0x82338731},
	{0x43B6, 0x82338732},
	{0x43B7, 0x82338733},
	{0x43B8, 0x82338734},
	{0x43B9, 0x82338735},
	{0x43BA, 0x82338736},
	{0x43BB, 0x82338737},
	{0x43BC, 0x82338738},
	{0x43BD, 0x82338739},
	{0x43BE, 0x82338830},
	{0x43BF, 0x82338831},
	{0x43C0, 0x82338832},
	{0x43C1, 0x82338833},
	{0x43C2, 0x82338834},
	{0x43C3, 0x82338835},
	{0x43C4, 0x82338836},
	{0x43C5, 0x82338837},
	{0x43C6, 0x82338838},
	{0x43C7, 0x82338839},
	{0x43C8, 0x82338930},
	{0x43C9, 0x82338931},
	{0x43CA, 0x82338932},
	{0x43CB, 0x82338933},
	{0x43CC, 0x82338934},
	{0x43CD, 0x82338935},
	{0x43CE, 0x82338936},
	{0x43CF, 0x82338937},
	{0x43D0, 0x82338938},
	{0x43D1, 0x82338939},
	{0x43D2, 0x82338A30},
	{0x43D3, 0x82338A31},
	{0x43D4, 0x82338A32},
	{0x43D5, 0x82338A33},
	{0x43D6, 0x82338A34},
	{0x43D7, 0x82338A35},
	{0x43D8, 0x82338A36},
	{0x43D9, 0x82338A37},
	{0x43DA, 0x82338A38},
	{0x43DB, 0x82338A39},
	{0x43DC, 0x82338B30},
	{0x43DE, 0x82338B31},
	{0x43DF, 0x82338B32},
	{0x43E0, 0x82338B33},
	{0x43E1, 0x82338B34},
	{0x43E2, 0x82338B35},
	{0x43E3, 0x82338B36},
	{0x43E4, 0x82338B37},
	{0x43E5, 0x82338B38},
	{0x43E6, 0x82338B39},
	{0x43E7, 0x82338C30},
	{0x43E8, 0x82338C31},
	{0x43E9, 0x82338C32},
	{0x43EA, 0x82338C33},
	{0x43EB, 0x82338C34},
	{0x43EC, 0x82338C35},
	{0x43ED, 0x82338C36},
	{0x43EE, 0x82338C37},
	{0x43EF, 0x82338C38},
	{0x43F0, 0x82338C39},
	{0x43F1, 0x82338D30},
	{0x43F2, 0x82338D31},
	{0x43F3, 0x82338D32},
	{0x43F4, 0x82338D33},
	{0x43F5, 0x82338D34},
	{0x43F6, 0x82338D35},
	{0x43F7, 0x82338D36},
	{0x43F8, 0x82338D37},
	{0x43F9, 0x82338D38},
	{0x43FA, 0x82338D39},
	{0x43FB, 0x82338E30},
	{0x43FC, 0x82338E31},
	{0x43FD, 0x82338E32},
	{0x43FE, 0x82338E33},
	{0x43FF, 0x82338E34},
	{0x4400, 0x82338E35},
	{0x4401, 0x82338E36},
	{0x4402, 0x82338E37},
	{0x4403, 0x82338E38},
	{0x4404, 0x82338E39},
	{0x4405, 0x82338F30},
	{0x4406, 0x82338F31},
	{0x4407, 0x82338F32},
	{0x4408, 0x82338F33},
	{0x4409, 0x82338F34},
	{0x440A, 0x82338F35},
	{0x440B, 0x82338F36},
	{0x440C, 0x82338F37},
	{0x440D, 0x82338F38},
	{0x440E, 0x82338F39},
	{0x440F, 0x82339030},
	{0x4410, 0x82339031},
	{0x4411, 0x82339032},
	{0x4412, 0x82339033},
	{0x4413, 0x82339034},
	{0x4414, 0x82339035},
	{0x4415, 0x82339036},
	{0x4416, 0x82339037},
	{0x4417, 0x82339038},
	{0x4418, 0x82339039},
	{0x4419, 0x82339130},
	{0x441A, 0x82339131},
	{0x441B, 0x82339132},
	{0x441C, 0x82339133},
	{0x441D, 0x82339134},
	{0x441E, 0x82339135},
	{0x441F, 0x82339136},
	{0x4420, 0x82339137},
	{0x4421, 0x82339138},
	{0x4422, 0x82339139},
	{0x4423, 0x82339230},
	{0x4424, 0x82339231},
	{0x4425, 0x82339232},
	{0x4426, 0x82339233},
	{0x4427, 0x82339234},
	{0x4428, 0x82339235},
	{0x4429, 0x82339236},
	{0x442A, 0x82339237},
	{0x442B, 0x82339238},
	{0x442C, 0x82339239},
	{0x442D, 0x82339330},
	{0x442E, 0x82339331},
	{0x442F, 0x82339332},
	{0x4430, 0x82339333},
	{0x4431, 0x82339334},
	{0x4432, 0x82339335},
	{0x4433, 0x82339336},
	{0x4434, 0x82339337},
	{0x4435, 0x82339338},
	{0x4436, 0x82339339},
	{0x4437, 0x82339430},
	{0x4438, 0x82339431},
	{0x4439, 0x82339432},
	{0x443A, 0x82339433},
	{0x443B, 0x82339434},
	{0x443C, 0x82339435},
	{0x443D, 0x82339436},
	{0x443E, 0x82339437},
	{0x443F, 0x82339438},
	{0x4440, 0x82339439},
	{0x4441, 0x82339530},
	{0x4442, 0x82339531},
	{0x4443, 0x82339532},
	{0x4444, 0x82339533},
	{0x4445, 0x82339534},
	{0x4446, 0x82339535},
	{0x4447, 0x82339536},
	{0x4448, 0x82339537},
	{0x4449, 0x82339538},
	{0x444A, 0x82339539},
	{0x444B, 0x82339630},
	{0x444C, 0x82339631},
	{0x444D, 0x82339632},
	{0x444E, 0x82339633},
	{0x444F, 0x82339634},
	{0x4450, 0x82339635},
	{0x4451, 0x82339636},
	{0x4452, 0x82339637},
	{0x4453, 0x82339638},
	{0x4454, 0x82339639},
	{0x4455, 0x82339730},
	{0x4456, 0x82339731},
	{0x4457, 0x82339732},
	{0x4458, 0x82339733},
	{0x4459, 0x82339734},
	{0x445A, 0x82339735},
	{0x445B, 0x82339736},
	{0x445C, 0x82339737},
	{0x445D, 0x82339738},
	{0x445E, 0x82339739},
	{0x445F, 0x82339830},
	{0x4460, 0x82339831},
	{0x4461, 0x82339832},
	{0x4462, 0x82339833},
	{0x4463, 0x82339834},
	{0x4464, 0x82339835},
	{0x4465, 0x82339836},
	{0x4466, 0x82339837},
	{0x4467, 0x82339838},
	{0x4468, 0x82339839},
	{0x4469, 0x82339930},
	{0x446A, 0x82339931},
	{0x446B, 0x82339932},
	{0x446C, 0x82339933},
	{0x446D, 0x82339934},
	{0x446E, 0x82339935},
	{0x446F, 0x82339936},
	{0x4470, 0x82339937},
	{0x4471, 0x82339938},
	{0x4472, 0x82339939},
	{0x4473, 0x82339A30},
	{0x4474, 0x82339A31},
	{0x4475, 0x82339A32},
	{0x4476, 0x82339A33},
	{0x4477, 0x82339A34},
	{0x4478, 0x82339A35},
	{0x4479, 0x82339A36},
	{0x447A, 0x82339A37},
	{0x447B, 0x82339A38},
	{0x447C, 0x82339A39},
	{0x447D, 0x82339B30},
	{0x447E, 0x82339B31},
	{0x447F, 0x82339B32},
	{0x4480, 0x82339B33},
	{0x4481, 0x82339B34},
	{0x4482, 0x82339B35},
	{0x4483, 0x82339B36},
	{0x4484, 0x82339B37},
	{0x4485, 0x82339B38},
	{0x4486, 0x82339B39},
	{0x4487, 0x82339C30},
	{0x4488, 0x82339C31},
	{0x4489, 0x82339C32},
	{0x448A, 0x82339C33},
	{0x448B, 0x82339C34},
	{0x448C, 0x82339C35},
	{0x448D, 0x82339C36},
	{0x448E, 0x82339C37},
	{0x448F, 0x82339C38},
	{0x4490, 0x82339C39},
	{0x4491, 0x82339D30},
	{0x4492, 0x82339D31},
	{0x4493, 0x82339D32},
	{0x4494, 0x82339D33},
	{0x4495, 0x82339D34},
	{0x4496, 0x82339D35},
	{0x4497, 0x82339D36},
	{0x4498, 0x82339D37},
	{0x4499, 0x82339D38},
	{0x449A, 0x82339D39},
	{0x449B, 0x82339E30},
	{0x449C, 0x82339E31},
	{0x449D, 0x82339E32},
	{0x449E, 0x82339E33},
	{0x449F, 0x82339E34},
	{0x44A0, 0x82339E35},
	{0x44A1, 0x82339E36},
	{0x44A2, 0x82339E37},
	{0x44A3, 0x82339E38},
	{0x44A4, 0x82339E39},
	{0x44A5, 0x82339F30},
	{0x44A6, 0x82339F31},
	{0x44A7, 0x82339F32},
	{0x44A8, 0x82339F33},
	{0x44A9, 0x82339F34},
	{0x44AA, 0x82339F35},
	{0x44AB, 0x82339F36},
	{0x44AC, 0x82339F37},
	{0x44AD, 0x82339F38},
	{0x44AE, 0x82339F39},
	{0x44AF, 0x8233A030},
	{0x44B0, 0x8233A031},
	{0x44B1, 0x8233A032},
	{0x44B2, 0x8233A033},
	{0x44B3, 0x8233A034},
	{0x44B4, 0x8233A035},
	{0x44B5, 0x8233A036},
	{0x44B6, 0x8233A037},
	{0x44B7, 0x8233A038},
	{0x44B8, 0x8233A039},
	{0x44B9, 0x8233A130},
	{0x44BA, 0x8233A131},
	{0x44BB, 0x8233A132},
	{0x44BC, 0x8233A133},
	{0x44BD, 0x8233A134},
	{0x44BE, 0x8233A135},
	{0x44BF, 0x8233A136},
	{0x44C0, 0x8233A137},
	{0x44C1, 0x8233A138},
	{0x44C2, 0x8233A139},
	{0x44C3, 0x8233A230},
	{0x44C4, 0x8233A231},
	{0x44C5, 0x8233A232},
	{0x44C6, 0x8233A233},
	{0x44C7, 0x8233A234},
	{0x44C8, 0x8233A235},
	{0x44C9, 0x8233A236},
	{0x44CA, 0x8233A237},
	{0x44CB, 0x8233A238},
	{0x44CC, 0x8233A239},
	{0x44CD, 0x8233A330},
	{0x44CE, 0x8233A331},
	{0x44CF, 0x8233A332},
	{0x44D0, 0x8233A333},
	{0x44D1, 0x8233A334},
	{0x44D2, 0x8233A335},
	{0x44D3, 0x8233A336},
	{0x44D4, 0x8233A337},
	{0x44D5, 0x8233A338},
	{0x464D, 0x8233C932},
	{0x464E, 0x8233C933},
	{0x464F, 0x8233C934},
	{0x4650, 0x8233C935},
	{0x4651, 0x8233C936},
	{0x4652, 0x8233C937},
	{0x4653, 0x8233C938},
	{0x4654, 0x8233C939},
	{0x4655, 0x8233CA30},
	{0x4656, 0x8233CA31},
	{0x4657, 0x8233CA32},
	{0x4658, 0x8233CA33},
	{0x4659, 0x8233CA34},
	{0x465A, 0x8233CA35},
	{0x465B, 0x8233CA36},
	{0x465C, 0x8233CA37},
	{0x465D, 0x8233CA38},
	{0x465E, 0x8233CA39},
	{0x465F, 0x8233CB30},
	{0x4660, 0x8233CB31},
	{0x4662, 0x8233CB32},
	{0x4663, 0x8233CB33},
	{0x4664, 0x8233CB34},
	{0x4665, 0x8233CB35},
	{0x4666, 0x8233CB36},
	{0x4667, 0x8233CB37},
	{0x4668, 0x8233CB38},
	{0x4669, 0x8233CB39},
	{0x466A, 0x8233CC30},
	{0x466B, 0x8233CC31},
	{0x466C, 0x8233CC32},
	{0x466D, 0x8233CC33},
	{0x466E, 0x8233CC34},
	{0x466F, 0x8233CC35},
	{0x4670, 0x8233CC36},
	{0x4671, 0x8233CC37},
	{0x4672, 0x8233CC38},
	{0x4673, 0x8233CC39},
	{0x4674, 0x8233CD30},
	{0x4675, 0x8233CD31},
	{0x4676, 0x8233CD32},
	{0x4677, 0x8233CD33},
	{0x4678, 0x8233CD34},
	{0x4679, 0x8233CD35},
	{0x467A, 0x8233CD36},
	{0x467B, 0x8233CD37},
	{0x467C, 0x8233CD38},
	{0x467D, 0x8233CD39},
	{0x467E, 0x8233CE30},
	{0x467F, 0x8233CE31},
	{0x4680, 0x8233CE32},
	{0x4681, 0x8233CE33},
	{0x4682, 0x8233CE34},
	{0x4683, 0x8233CE35},
	{0x4684, 0x8233CE36},
	{0x4685, 0x8233CE37},
	{0x4686, 0x8233CE38},
	{0x4687, 0x8233CE39},
	{0x4688, 0x8233CF30},
	{0x4689, 0x8233CF31},
	{0x468A, 0x8233CF32},
	{0x468B, 0x8233CF33},
	{0x468C, 0x8233CF34},
	{0x468D, 0x8233CF35},
	{0x468E, 0x8233CF36},
	{0x468F, 0x8233CF37},
	{0x4690, 0x8233CF38},
	{0x4691, 0x8233CF39},
	{0x4692, 0x8233D030},
	{0x4693, 0x8233D031},
	{0x4694, 0x8233D032},
	{0x4695, 0x8233D033},
	{0x4696, 0x8233D034},
	{0x4697, 0x8233D035},
	{0x4698, 0x8233D036},
	{0x4699, 0x8233D037},
	{0x469A, 0x8233D038},
	{0x469B, 0x8233D039},
	{0x469C, 0x8233D130},
	{0x469D, 0x8233D131},
	{0x469E, 0x8233D132},
	{0x469F, 0x8233D133},
	{0x46A0, 0x8233D134},
	{0x46A1, 0x8233D135},
	{0x46A2, 0x8233D136},
	{0x46A3, 0x8233D137},
	{0x46A4, 0x8233D138},
	{0x46A5, 0x8233D139},
	{0x46A6, 0x8233D230},
	{0x46A7, 0x8233D231},
	{0x46A8, 0x8233D232},
	{0x46A9, 0x8233D233},
	{0x46AA, 0x8233D234},
	{0x46AB, 0x8233D235},
	{0x46AC, 0x8233D236},
	{0x46AD, 0x8233D237},
	{0x46AE, 0x8233D238},
	{0x46AF, 0x8233D239},
	{0x46B0, 0x8233D330},
	{0x46B1, 0x8233D331},
	{0x46B2, 0x8233D332},
	{0x46B3, 0x8233D333},
	{0x46B4, 0x8233D334},
	{0x46B5, 0x8233D335},
	{0x46B6, 0x8233D336},
	{0x46B7, 0x8233D337},
	{0x46B8, 0x8233D338},
	{0x46B9, 0x8233D339},
	{0x46BA, 0x8233D430},
	{0x46BB, 0x8233D431},
	{0x46BC, 0x8233D432},
	{0x46BD, 0x8233D433},
	{0x46BE, 0x8233D434},
	{0x46BF, 0x8233D435},
	{0x46C0, 0x8233D436},
	{0x46C1, 0x8233D437},
	{0x46C2, 0x8233D438},
	{0x46C3, 0x8233D439},
	{0x46C4, 0x8233D530},
	{0x46C5, 0x8233D531},
	{0x46C6, 0x8233D532},
	{0x46C7, 0x8233D533},
	{0x46C8, 0x8233D534},
	{0x46C9, 0x8233D535},
	{0x46CA, 0x8233D536},
	{0x46CB, 0x8233D537},
	{0x46CC, 0x8233D538},
	{0x46CD, 0x8233D539},
	{0x46CE, 0x8233D630},
	{0x46CF, 0x8233D631},
	{0x46D0, 0x8233D632},
	{0x46D1, 0x8233D633},
	{0x46D2, 0x8233D634},
	{0x46D3, 0x8233D635},
	{0x46D4, 0x8233D636},
	{0x46D5, 0x8233D637},
	{0x46D6, 0x8233D638},
	{0x46D7, 0x8233D639},
	{0x46D8, 0x8233D730},
	{0x46D9, 0x8233D731},
	{0x46DA, 0x8233D732},
	{0x46DB, 0x8233D733},
	{0x46DC, 0x8233D734},
	{0x46DD, 0x8233D735},
	{0x46DE, 0x8233D736},
	{0x46DF, 0x8233D737},
	{0x46E0, 0x8233D738},
	{0x46E1, 0x8233D739},
	{0x46E2, 0x8233D830},
	{0x46E3, 0x8233D831},
	{0x46E4, 0x8233D832},
	{0x46E5, 0x8233D833},
	{0x46E6, 0x8233D834},
	{0x46E7, 0x8233D835},
	{0x46E8, 0x8233D836},
	{0x46E9, 0x8233D837},
	{0x46EA, 0x8233D838},
	{0x46EB, 0x8233D839},
	{0x46EC, 0x8233D930},
	{0x46ED, 0x8233D931},
	{0x46EE, 0x8233D932},
	{0x46EF, 0x8233D933},
	{0x46F0, 0x8233D934},
	{0x46F1, 0x8233D935},
	{0x46F2, 0x8233D936},
	{0x46F3, 0x8233D937},
	{0x46F4, 0x8233D938},
	{0x46F5, 0x8233D939},
	{0x46F6, 0x8233DA30},
	{0x46F7, 0x8233DA31},
	{0x46F8, 0x8233DA32},
	{0x46F9, 0x8233DA33},
	{0x46FA, 0x8233DA34},
	{0x46FB, 0x8233DA35},
	{0x46FC, 0x8233DA36},
	{0x46FD, 0x8233DA37},
	{0x46FE, 0x8233DA38},
	{0x46FF, 0x8233DA39},
	{0x4700, 0x8233DB30},
	{0x4701, 0x8233DB31},
	{0x4702, 0x8233DB32},
	{0x4703, 0x8233DB33},
	{0x4704, 0x8233DB34},
	{0x4705, 0x8233DB35},
	{0x4706, 0x8233DB36},
	{0x4707, 0x8233DB37},
	{0x4708, 0x8233DB38},
	{0x4709, 0x8233DB39},
	{0x470A, 0x8233DC30},
	{0x470B, 0x8233DC31},
	{0x470C, 0x8233DC32},
	{0x470D, 0x8233DC33},
	{0x470E, 0x8233DC34},
	{0x470F, 0x8233DC35},
	{0x4710, 0x8233DC36},
	{0x4711, 0x8233DC37},
	{0x4712, 0x8233DC38},
	{0x4713, 0x8233DC39},
	{0x4714, 0x8233DD30},
	{0x4715, 0x8233DD31},
	{0x4716, 0x8233DD32},
	{0x4717, 0x8233DD33},
	{0x4718, 0x8233DD34},
	{0x4719, 0x8233DD35},
	{0x471A, 0x8233DD36},
	{0x471B, 0x8233DD37},
	{0x471C, 0x8233DD38},
	{0x471D, 0x8233DD39},
	{0x471E, 0x8233DE30},
	{0x471F, 0x8233DE31},
	{0x4720, 0x8233DE32},
	{0x4721, 0x8233DE33},
	{0x4722, 0x8233DE34},
	{0x4724, 0x8233DE35},
	{0x4725, 0x8233DE36},
	{0x4726, 0x8233DE37},
	{0x4727, 0x8233DE38},
	{0x4728, 0x8233DE39},
	{0x472A, 0x8233DF30},
	{0x472B, 0x8233DF31},
	{0x472C, 0x8233DF32},
	{0x472D, 0x8233DF33},
	{0x472E, 0x8233DF34},
	{0x472F, 0x8233DF35},
	{0x4730, 0x8233DF36},
	{0x4731, 0x8233DF37},
	{0x4732, 0x8233DF38},
	{0x4733, 0x8233DF39},
	{0x4734, 0x8233E030},
	{0x4735, 0x8233E031},
	{0x4736, 0x8233E032},
	{0x4737, 0x8233E033},
	{0x4738, 0x8233E034},
	{0x4739, 0x8233E035},
	{0x473A, 0x8233E036},
	{0x473B, 0x8233E037},
	{0x473C, 0x8233E038},
	{0x473D, 0x8233E039},
	{0x473E, 0x8233E130},
	{0x473F, 0x8233E131},
	{0x4740, 0x8233E132},
	{0x4741, 0x8233E133},
	{0x4742, 0x8233E134},
	{0x4743, 0x8233E135},
	{0x4744, 0x8233E136},
	{0x4745, 0x8233E137},
	{0x4746, 0x8233E138},
	{0x4747, 0x8233E139},
	{0x4748, 0x8233E230},
	{0x4749, 0x8233E231},
	{0x474A, 0x8233E232},
	{0x474B, 0x8233E233},
	{0x474C, 0x8233E234},
	{0x474D, 0x8233E235},
	{0x474E, 0x8233E236},
	{0x474F, 0x8233E237},
	{0x4750, 0x8233E238},
	{0x4751, 0x8233E239},
	{0x4752, 0x8233E330},
	{0x4753, 0x8233E331},
	{0x4754, 0x8233E332},
	{0x4755, 0x8233E333},
	{0x4756, 0x8233E334},
	{0x4757, 0x8233E335},
	{0x4758, 0x8233E336},
	{0x4759, 0x8233E337},
	{0x475A, 0x8233E338},
	{0x475B, 0x8233E339},
	{0x475C, 0x8233E430},
	{0x475D, 0x8233E431},
	{0x475E, 0x8233E432},
	{0x475F, 0x8233E433},
	{0x4760, 0x8233E434},
	{0x4761, 0x8233E435},
	{0x4762, 0x8233E436},
	{0x4763, 0x8233E437},
	{0x4764, 0x8233E438},
	{0x4765, 0x8233E439},
	{0x4766, 0x8233E530},
	{0x4767, 0x8233E531},
	{0x4768, 0x8233E532},
	{0x4769, 0x8233E533},
	{0x476A, 0x8233E534},
	{0x476B, 0x8233E535},
	{0x476C, 0x8233E536},
	{0x476D, 0x8233E537},
	{0x476E, 0x8233E538},
	{0x476F, 0x8233E539},
	{0x4770, 0x8233E630},
	{0x4771, 0x8233E631},
	{0x4772, 0x8233E632},
	{0x4773, 0x8233E633},
	{0x4774, 0x8233E634},
	{0x4775, 0x8233E635},
	{0x4776, 0x8233E636},
	{0x4777, 0x8233E637},
	{0x4778, 0x8233E638},
	{0x4779, 0x8233E639},
	{0x477A, 0x8233E730},
	{0x477B, 0x8233E731},
	{0x477D, 0x8233E732},
	{0x477E, 0x8233E733},
	{0x477F, 0x8233E734},
	{0x4780, 0x8233E735},
	{0x4781, 0x8233E736},
	{0x4782, 0x8233E737},
	{0x4783, 0x8233E738},
	{0x4784, 0x8233E739},
	{0x4785, 0x8233E830},
	{0x4786, 0x8233E831},
	{0x4787, 0x8233E832},
	{0x4788, 0x8233E833},
	{0x4789, 0x8233E834},
	{0x478A, 0x8233E835},
	{0x478B, 0x8233E836},
	{0x478C, 0x8233E837},
	{0x4948, 0x82349639},
	{0x4949, 0x82349730},
	{0x494A, 0x82349731},
	{0x494B, 0x82349732},
	{0x494C, 0x82349733},
	{0x494D, 0x82349734},
	{0x494E, 0x82349735},
	{0x494F, 0x82349736},
	{0x4950, 0x82349737},
	{0x4951, 0x82349738},
	{0x4952, 0x82349739},
	{0x4953, 0x82349830},
	{0x4954, 0x82349831},
	{0x4955, 0x82349832},
	{0x4956, 0x82349833},
	{0x4957, 0x82349834},
	{0x4958, 0x82349835},
	{0x4959, 0x82349836},
	{0x495A, 0x82349837},
	{0x495B, 0x82349838},
	{0x495C, 0x82349839},
	{0x495D, 0x82349930},
	{0x495E, 0x82349931},
	{0x495F, 0x82349932},
	{0x4960, 0x82349933},
	{0x4961, 0x82349934},
	{0x4962, 0x82349935},
	{0x4963, 0x82349936},
	{0x4964, 0x82349937},
	{0x4965, 0x82349938},
	{0x4966, 0x82349939},
	{0x4967, 0x82349A30},
	{0x4968, 0x82349A31},
	{0x4969, 0x82349A32},
	{0x496A, 0x82349A33},
	{0x496B, 0x82349A34},
	{0x496C, 0x82349A35},
	{0x496D, 0x82349A36},
	{0x496E, 0x82349A37},
	{0x496F, 0x82349A38},
	{0x4970, 0x82349A39},
	{0x4971, 0x82349B30},
	{0x4972, 0x82349B31},
	{0x4973, 0x82349B32},
	{0x4974, 0x82349B33},
	{0x4975, 0x82349B34},
	{0x4976, 0x82349B35},
	{0x4977, 0x82349B36},
	{0x4978, 0x82349B37},
	{0x4979, 0x82349B38},
	{0x497B, 0x82349B39},
	{0x497C, 0x82349C30},
	{0x497E, 0x82349C31},
	{0x497F, 0x82349C32},
	{0x4980, 0x82349C33},
	{0x4981, 0x82349C34},
	{0x4984, 0x82349C35},
	{0x4987, 0x82349C36},
	{0x4988, 0x82349C37},
	{0x4989, 0x82349C38},
	{0x498A, 0x82349C39},
	{0x498B, 0x82349D30},
	{0x498C, 0x82349D31},
	{0x498D, 0x82349D32},
	{0x498E, 0x82349D33},
	{0x498F, 0x82349D34},
	{0x4990, 0x82349D35},
	{0x4991, 0x82349D36},
	{0x4992, 0x82349D37},
	{0x4993, 0x82349D38},
	{0x4994, 0x82349D39},
	{0x4995, 0x82349E30},
	{0x4996, 0x82349E31},
	{0x4997, 0x82349E32},
	{0x4998, 0x82349E33},
	{0x4999, 0x82349E34},
	{0x499A, 0x82349E35},
	{0x499C, 0x82349E36},
	{0x499D, 0x82349E37},
	{0x499E, 0x82349E38},
	{0x49A0, 0x82349E39},
	{0x49A1, 0x82349F30},
	{0x49A2, 0x82349F31},
	{0x49A3, 0x82349F32},
	{0x49A4, 0x82349F33},
	{0x49A5, 0x82349F34},
	{0x49A6, 0x82349F35},
	{0x49A7, 0x82349F36},
	{0x49A8, 0x82349F37},
	{0x49A9, 0x82349F38},
	{0x49AA, 0x82349F39},
	{0x49AB, 0x8234A030},
	{0x49AC, 0x8234A031},
	{0x49AD, 0x8234A032},
	{0x49AE, 0x8234A033},
	{0x49AF, 0x8234A034},
	{0x49B0, 0x8234A035},
	{0x49B1, 0x8234A036},
	{0x49B2, 0x8234A037},
	{0x49B3, 0x8234A038},
	{0x49B4, 0x8234A039},
	{0x49B5, 0x8234A130},
	{0x4C78, 0x8234E734},
	{0x4C79, 0x8234E735},
	{0x4C7A, 0x8234E736},
	{0x4C7B, 0x8234E737},
	{0x4C7C, 0x8234E738},
	{0x4C7D, 0x8234E739},
	{0x4C7E, 0x8234E830},
	{0x4C7F, 0x8234E831},
	{0x4C80, 0x8234E832},
	{0x4C81, 0x8234E833},
	{0x4C82, 0x8234E834},
	{0x4C83, 0x8234E835},
	{0x4C84, 0x8234E836},
	{0x4C85, 0x8234E837},
	{0x4C86, 0x8234E838},
	{0x4C87, 0x8234E839},
	{0x4C88, 0x8234E930},
	{0x4C89, 0x8234E931},
	{0x4C8A, 0x8234E932},
	{0x4C8B, 0x8234E933},
	{0x4C8C, 0x8234E934},
	{0x4C8D, 0x8234E935},
	{0x4C8E, 0x8234E936},
	{0x4C8F, 0x8234E937},
	{0x4C90, 0x8234E938},
	{0x4C91, 0x8234E939},
	{0x4C92, 0x8234EA30},
	{0x4C93, 0x8234EA31},
	{0x4C94, 0x8234EA32},
	{0x4C95, 0x8234EA33},
	{0x4C96, 0x8234EA34},
	{0x4C97, 0x8234EA35},
	{0x4C98, 0x8234EA36},
	{0x4C99, 0x8234EA37},
	{0x4C9A, 0x8234EA38},
	{0x4C9B, 0x8234EA39},
	{0x4C9C, 0x8234EB30},
	{0x4C9D, 0x8234EB31},
	{0x4C9E, 0x8234EB32},
	{0x4CA4, 0x8234EB33},
	{0x4CA5, 0x8234EB34},
	{0x4CA6, 0x8234EB35},
	{0x4CA7, 0x8234EB36},
	{0x4CA8, 0x8234EB37},
	{0x4CA9, 0x8234EB38},
	{0x4CAA, 0x8234EB39},
	{0x4CAB, 0x8234EC30},
	{0x4CAC, 0x8234EC31},
	{0x4CAD, 0x8234EC32},
	{0x4CAE, 0x8234EC33},
	{0x4CAF, 0x8234EC34},
	{0x4CB0, 0x8234EC35},
	{0x4CB1, 0x8234EC36},
	{0x4CB2, 0x8234EC37},
	{0x4CB3, 0x8234EC38},
	{0x4CB4, 0x8234EC39},
	{0x4CB5, 0x8234ED30},
	{0x4CB6, 0x8234ED31},
	{0x4CB7, 0x8234ED32},
	{0x4CB8, 0x8234ED33},
	{0x4CB9, 0x8234ED34},
	{0x4CBA, 0x8234ED35},
	{0x4CBB, 0x8234ED36},
	{0x4CBC, 0x8234ED37},
	{0x4CBD, 0x8234ED38},
	{0x4CBE, 0x8234ED39},
	{0x4CBF, 0x8234EE30},
	{0x4CC0, 0x8234EE31},
	{0x4CC1, 0x8234EE32},
	{0x4CC2, 0x8234EE33},
	{0x4CC3, 0x8234EE34},
	{0x4CC4, 0x8234EE35},
	{0x4CC5, 0x8234EE36},
	{0x4CC6, 0x8234EE37},
	{0x4CC7, 0x8234EE38},
	{0x4CC8, 0x8234EE39},
	{0x4CC9, 0x8234EF30},
	{0x4CCA, 0x8234EF31},
	{0x4CCB, 0x8234EF32},
	{0x4CCC, 0x8234EF33},
	{0x4CCD, 0x8234EF34},
	{0x4CCE, 0x8234EF35},
	{0x4CCF, 0x8234EF36},
	{0x4CD0, 0x8234EF37},
	{0x4CD1, 0x8234EF38},
	{0x4CD2, 0x8234EF39},
	{0x4CD3, 0x8234F030},
	{0x4CD4, 0x8234F031},
	{0x4CD5, 0x8234F032},
	{0x4CD6, 0x8234F033},
	{0x4CD7, 0x8234F034},
	{0x4CD8, 0x8234F035},
	{0x4CD9, 0x8234F036},
	{0x4CDA, 0x8234F037},
	{0x4CDB, 0x8234F038},
	{0x4CDC, 0x8234F039},
	{0x4CDD, 0x8234F130},
	{0x4CDE, 0x8234F131},
	{0x4CDF, 0x8234F132},
	{0x4CE0, 0x8234F133},
	{0x4CE1, 0x8234F134},
	{0x4CE2, 0x8234F135},
	{0x4CE3, 0x8234F136},
	{0x4CE4, 0x8234F137},
	{0x4CE5, 0x8234F138},
	{0x4CE6, 0x8234F139},
	{0x4CE7, 0x8234F230},
	{0x4CE8, 0x8234F231},
	{0x4CE9, 0x8234F232},
	{0x4CEA, 0x8234F233},
	{0x4CEB, 0x8234F234},
	{0x4CEC, 0x8234F235},
	{0x4CED, 0x8234F236},
	{0x4CEE, 0x8234F237},
	{0x4CEF, 0x8234F238},
	{0x4CF0, 0x8234F239},
	{0x4CF1, 0x8234F330},
	{0x4CF2, 0x8234F331},
	{0x4CF3, 0x8234F332},
	{0x4CF4, 0x8234F333},
	{0x4CF5, 0x8234F334},
	{0x4CF6, 0x8234F335},
	{0x4CF7, 0x8234F336},
	{0x4CF8, 0x8234F337},
	{0x4CF9, 0x8234F338},
	{0x4CFA, 0x8234F339},
	{0x4CFB, 0x8234F430},
	{0x4CFC, 0x8234F431},
	{0x4CFD, 0x8234F432},
	{0x4CFE, 0x8234F433},
	{0x4CFF, 0x8234F434},
	{0x4D00, 0x8234F435},
	{0x4D01, 0x8234F436},
	{0x4D02, 0x8234F437},
	{0x4D03, 0x8234F438},
	{0x4D04, 0x8234F439},
	{0x4D05, 0x8234F530},
	{0x4D06, 0x8234F531},
	{0x4D07, 0x8234F532},
	{0x4D08, 0x8234F533},
	{0x4D09, 0x8234F534},
	{0x4D0A, 0x8234F535},
	{0x4D0B, 0x8234F536},
	{0x4D0C, 0x8234F537},
	{0x4D0D, 0x8234F538},
	{0x4D0E, 0x8234F539},
	{0x4D0F, 0x8234F630},
	{0x4D10, 0x8234F631},
	{0x4D11, 0x8234F632},
	{0x4D12, 0x8234F633},
	{0x4D1A, 0x8234F634},
	{0x4D1B, 0x8234F635},
	{0x4D1C, 0x8234F636},
	{0x4D1D, 0x8234F637},
	{0x4D1E, 0x8234F638},
	{0x4D1F, 0x8234F639},
	{0x4D20, 0x8234F730},
	{0x4D21, 0x8234F731},
	{0x4D22, 0x8234F732},
	{0x4D23, 0x8234F733},
	{0x4D24, 0x8234F734},
	{0x4D25, 0x8234F735},
	{0x4D26, 0x8234F736},
	{0x4D27, 0x8234F737},
	{0x4D28, 0x8234F738},
	{0x4D29, 0x8234F739},
	{0x4D2A, 0x8234F830},
	{0x4D2B, 0x8234F831},
	{0x4D2C, 0x8234F832},
	{0x4D2D, 0x8234F833},
	{0x4D2E, 0x8234F834},
	{0x4D2F, 0x8234F835},
	{0x4D30, 0x8234F836},
	{0x4D31, 0x8234F837},
	{0x4D32, 0x8234F838},
	{0x4D33, 0x8234F839},
	{0x4D34, 0x8234F930},
	{0x4D35, 0x8234F931},
	{0x4D36, 0x8234F932},
	{0x4D37, 0x8234F933},
	{0x4D38, 0x8234F934},
	{0x4D39, 0x8234F935},
	{0x4D3A, 0x8234F936},
	{0x4D3B, 0x8234F937},
	{0x4D3C, 0x8234F938},
	{0x4D3D, 0x8234F939},
	{0x4D3E, 0x8234FA30},
	{0x4D3F, 0x8234FA31},
	{0x4D40, 0x8234FA32},
	{0x4D41, 0x8234FA33},
	{0x4D42, 0x8234FA34},
	{0x4D43, 0x8234FA35},
	{0x4D44, 0x8234FA36},
	{0x4D45, 0x8234FA37},
	{0x4D46, 0x8234FA38},
	{0x4D47, 0x8234FA39},
	{0x4D48, 0x8234FB30},
	{0x4D49, 0x8234FB31},
	{0x4D4A, 0x8234FB32},
	{0x4D4B, 0x8234FB33},
	{0x4D4C, 0x8234FB34},
	{0x4D4D, 0x8234FB35},
	{0x4D4E, 0x8234FB36},
	{0x4D4F, 0x8234FB37},
	{0x4D50, 0x8234FB38},
	{0x4D51, 0x8234FB39},
	{0x4D52, 0x8234FC30},
	{0x4D53, 0x8234FC31},
	{0x4D54, 0x8234FC32},
	{0x4D55, 0x8234FC33},
	{0x4D56, 0x8234FC34},
	{0x4D57, 0x8234FC35},
	{0x4D58, 0x8234FC36},
	{0x4D59, 0x8234FC37},
	{0x4D5A, 0x8234FC38},
	{0x4D5B, 0x8234FC39},
	{0x4D5C, 0x8234FD30},
	{0x4D5D, 0x8234FD31},
	{0x4D5E, 0x8234FD32},
	{0x4D5F, 0x8234FD33},
	{0x4D60, 0x8234FD34},
	{0x4D61, 0x8234FD35},
	{0x4D62, 0x8234FD36},
	{0x4D63, 0x8234FD37},
	{0x4D64, 0x8234FD38},
	{0x4D65, 0x8234FD39},
	{0x4D66, 0x8234FE30},
	{0x4D67, 0x8234FE31},
	{0x4D68, 0x8234FE32},
	{0x4D69, 0x8234FE33},
	{0x4D6A, 0x8234FE34},
	{0x4D6B, 0x8234FE35},
	{0x4D6C, 0x8234FE36},
	{0x4D6D, 0x8234FE37},
	{0x4D6E, 0x8234FE38},
	{0x4D6F, 0x8234FE39},
	{0x4D70, 0x82358130},
	{0x4D71, 0x82358131},
	{0x4D72, 0x82358132},
	{0x4D73, 0x82358133},
	{0x4D74, 0x82358134},
	{0x4D75, 0x82358135},
	{0x4D76, 0x82358136},
	{0x4D77, 0x82358137},
	{0x4D78, 0x82358138},
	{0x4D79, 0x82358139},
	{0x4D7A, 0x82358230},
	{0x4D7B, 0x82358231},
	{0x4D7C, 0x82358232},
	{0x4D7D, 0x82358233},
	{0x4D7E, 0x82358234},
	{0x4D7F, 0x82358235},
	{0x4D80, 0x82358236},
	{0x4D81, 0x82358237},
	{0x4D82, 0x82358238},
	{0x4D83, 0x82358239},
	{0x4D84, 0x82358330},
	{0x4D85, 0x82358331},
	{0x4D86, 0x82358332},
	{0x4D87, 0x82358333},
	{0x4D88, 0x82358334},
	{0x4D89, 0x82358335},
	{0x4D8A, 0x82358336},
	{0x4D8B, 0x82358337},
	{0x4D8C, 0x82358338},
	{0x4D8D, 0x82358339},
	{0x4D8E, 0x82358430},
	{0x4D8F, 0x82358431},
	{0x4D90, 0x82358432},
	{0x4D91, 0x82358433},
	{0x4D92, 0x82358434},
	{0x4D93, 0x82358435},
	{0x4D94, 0x82358436},
	{0x4D95, 0x82358437},
	{0x4D96, 0x82358438},
	{0x4D97, 0x82358439},
	{0x4D98, 0x82358530},
	{0x4D99, 0x82358531},
	{0x4D9A, 0x82358532},
	{0x4D9B, 0x82358533},
	{0x4D9C, 0x82358534},
	{0x4D9D, 0x82358535},
	{0x4D9E, 0x82358536},
	{0x4D9F, 0x82358537},
	{0x4DA0, 0x82358538},
	{0x4DA1, 0x82358539},
	{0x4DA2, 0x82358630},
	{0x4DA3, 0x82358631},
	{0x4DA4, 0x82358632},
	{0x4DA5, 0x82358633},
	{0x4DA6, 0x82358634},
	{0x4DA7, 0x82358635},
	{0x4DA8, 0x82358636},
	{0x4DA9, 0x82358637},
	{0x4DAA, 0x82358638},
	{0x4DAB, 0x82358639},
	{0x4DAC, 0x82358730},
	{0x4DAD, 0x82358731},
	{0x4DAF, 0x82358732},
	{0x4DB0, 0x82358733},
	{0x4DB1, 0x82358734},
	{0x4DB2, 0x82358735},
	{0x4DB3, 0x82358736},
	{0x4DB4, 0x82358737},
	{0x4DB5, 0x82358738},
	{0x4DB6, 0x82358739},
	{0x4DB7, 0x82358830},
	{0x4DB8, 0x82358831},
	{0x4DB9, 0x82358832},
	{0x4DBA, 0x82358833},
	{0x4DBB, 0x82358834},
	{0x4DBC, 0x82358835},
	{0x4DBD, 0x82358836},
	{0x4DBE, 0x82358837},
	{0x4DBF, 0x82358838},
	{0x4DC0, 0x82358839},
	{0x4DC1, 0x82358930},
	{0x4DC2, 0x82358931},
	{0x4DC3, 0x82358932},
	{0x4DC4, 0x82358933},
	{0x4DC5, 0x82358934},
	{0x4DC6, 0x82358935},
	{0x4DC7, 0x82358936},
	{0x4DC8, 0x82358937},
	{0x4DC9, 0x82358938},
	{0x4DCA, 0x82358939},
	{0x4DCB, 0x82358A30},
	{0x4DCC, 0x82358A31},
	{0x4DCD, 0x82358A32},
	{0x4DCE, 0x82358A33},
	{0x4DCF, 0x82358A34},
	{0x4DD0, 0x82358A35},
	{0x4DD1, 0x82358A36},
	{0x4DD2, 0x82358A37},
	{0x4DD3, 0x82358A38},
	{0x4DD4, 0x82358A39},
	{0x4DD5, 0x82358B30},
	{0x4DD6, 0x82358B31},
	{0x4DD7, 0x82358B32},
	{0x4DD8, 0x82358B33},
	{0x4DD9, 0x82358B34},
	{0x4DDA, 0x82358B35},
	{0x4DDB, 0x82358B36},
	{0x4DDC, 0x82358B37},
	{0x4DDD, 0x82358B38},
	{0x4DDE, 0x82358B39},
	{0x4DDF, 0x82358C30},
	{0x4DE0, 0x82358C31},
	{0x4DE1, 0x82358C32},
	{0x4DE2, 0x82358C33},
	{0x4DE3, 0x82358C34},
	{0x4DE4, 0x82358C35},
	{0x4DE5, 0x82358C36},
	{0x4DE6, 0x82358C37},
	{0x4DE7, 0x82358C38},
	{0x4DE8, 0x82358C39},
	{0x4DE9, 0x82358D30},
	{0x4DEA, 0x82358D31},
	{0x4DEB, 0x82358D32},
	{0x4DEC, 0x82358D33},
	{0x4DED, 0x82358D34},
	{0x4DEE, 0x82358D35},
	{0x4DEF, 0x82358D36},
	{0x4DF0, 0x82358D37},
	{0x4DF1, 0x82358D38},
	{0x4DF2, 0x82358D39},
	{0x4DF3, 0x82358E30},
	{0x4DF4, 0x82358E31},
	{0x4DF5, 0x82358E32},
	{0x4DF6, 0x82358E33},
	{0x4DF7, 0x82358E34},
	{0x4DF8, 0x82358E35},
	{0x4DF9, 0x82358E36},
	{0x4DFA, 0x82358E37},
	{0x4DFB, 0x82358E38},
	{0x4DFC, 0x82358E39},
	{0x4DFD, 0x82358F30},
	{0x4DFE, 0x82358F31},
	{0x4DFF, 0x82358F32},
	{0xE76C, 0x8336C739},
	{0xE7C7, 0x8135F437},
	{0xE7C8, 0x8336C830},
	{0xE7E7, 0x8336C831},
	{0xE7E8, 0x8336C832},
	{0xE7E9, 0x8336C833},
	{0xE7EA, 0x8336C834},
	{0xE7EB, 0x8336C835},
	{0xE7EC, 0x8336C836},
	{0xE7ED, 0x8336C837},
	{0xE7EE, 0x8336C838},
	{0xE7EF, 0x8336C839},
	{0xE7F0, 0x8336C930},
	{0xE7F1, 0x8336C931},
	{0xE7F2, 0x8336C932},
	{0xE7F3, 0x8336C933},
	{0xE815, 0x8336C934},
	{0xE819, 0x8336C935},
	{0xE81A, 0x8336C936},
	{0xE81B, 0x8336C937},
	{0xE81C, 0x8336C938},
	{0xE81D, 0x8336C939},
	{0xE81F, 0x8336CA30},
	{0xE820, 0x8336CA31},
	{0xE821, 0x8336CA32},
	{0xE822, 0x8336CA33},
	{0xE823, 0x8336CA34},
	{0xE824, 0x8336CA35},
	{0xE825, 0x8336CA36},
	{0xE827, 0x8336CA37},
	{0xE828, 0x8336CA38},
	{0xE829, 0x8336CA39},
	{0xE82A, 0x8336CB30},
	{0xE82D, 0x8336CB31},
	{0xE82E, 0x8336CB32},
	{0xE82F, 0x8336CB33},
	{0xE830, 0x8336CB34},
	{0xE833, 0x8336CB35},
	{0xE834, 0x8336CB36},
	{0xE835, 0x8336CB37},
	{0xE836, 0x8336CB38},
	{0xE837, 0x8336CB39},
	{0xE838, 0x8336CC30},
	{0xE839, 0x8336CC31},
	{0xE83A, 0x8336CC32},
	{0xE83C, 0x8336CC33},
	{0xE83D, 0x8336CC34},
	{0xE83E, 0x8336CC35},
	{0xE83F, 0x8336CC36},
	{0xE840, 0x8336CC37},
	{0xE841, 0x8336CC38},
	{0xE842, 0x8336CC39},
	{0xE844, 0x8336CD30},
	{0xE845, 0x8336CD31},
	{0xE846, 0x8336CD32},
	{0xE847, 0x8336CD33},
	{0xE848, 0x8336CD34},
	{0xE849, 0x8336CD35},
	{0xE84A, 0x8336CD36},
	{0xE84B, 0x8336CD37},
	{0xE84C, 0x8336CD38},
	{0xE84D, 0x8336CD39},
	{0xE84E, 0x8336CE30},
	{0xE84F, 0x8336CE31},
	{0xE850, 0x8336CE32},
	{0xE851, 0x8336CE33},
	{0xE852, 0x8336CE34},
	{0xE853, 0x8336CE35},
	{0xE856, 0x8336CE36},
	{0xE857, 0x8336CE37},
	{0xE858, 0x8336CE38},
	{0xE859, 0x8336CE39},
	{0xE85A, 0x8336CF30},
	{0xE85B, 0x8336CF31},
	{0xE85C, 0x8336CF32},
	{0xE85D, 0x8336CF33},
	{0xE85E, 0x8336CF34},
	{0xE85F, 0x8336CF35},
	{0xE860, 0x8336CF36},
	{0xE861, 0x8336CF37},
	{0xE862, 0x8336CF38},
	{0xE863, 0x8336CF39},
	{0xF92D, 0x84308535},
	{0xF92E, 0x84308536},
	{0xF92F, 0x84308537},
	{0xF930, 0x84308538},
	{0xF931, 0x84308539},
	{0xF932, 0x84308630},
	{0xF933, 0x84308631},
	{0xF934, 0x84308632},
	{0xF935, 0x84308633},
	{0xF936, 0x84308634},
	{0xF937, 0x84308635},
	{0xF938, 0x84308636},
	{0xF939, 0x84308637},
	{0xF93A, 0x84308638},
	{0xF93B, 0x84308639},
	{0xF93C, 0x84308730},
	{0xF93D, 0x84308731},
	{0xF93E, 0x84308732},
	{0xF93F, 0x84308733},
	{0xF940, 0x84308734},
	{0xF941, 0x84308735},
	{0xF942, 0x84308736},
	{0xF943, 0x84308737},
	{0xF944, 0x84308738},
	{0xF945, 0x84308739},
	{0xF946, 0x84308830},
	{0xF947, 0x84308831},
	{0xF948, 0x84308832},
	{0xF949, 0x84308833},
	{0xF94A, 0x84308834},
	{0xF94B, 0x84308835},
	{0xF94C, 0x84308836},
	{0xF94D, 0x84308837},
	{0xF94E, 0x84308838},
	{0xF94F, 0x84308839},
	{0xF950, 0x84308930},
	{0xF951, 0x84308931},
	{0xF952, 0x84308932},
	{0xF953, 0x84308933},
	{0xF954, 0x84308934},
	{0xF955, 0x84308935},
	{0xF956, 0x84308936},
	{0xF957, 0x84308937},
	{0xF958, 0x84308938},
	{0xF959, 0x84308939},
	{0xF95A, 0x84308A30},
	{0xF95B, 0x84308A31},
	{0xF95C, 0x84308A32},
	{0xF95D, 0x84308A33},
	{0xF95E, 0x84308A34},
	{0xF95F, 0x84308A35},
	{0xF960, 0x84308A36},
	{0xF961, 0x84308A37},
	{0xF962, 0x84308A38},
	{0xF963, 0x84308A39},
	{0xF964, 0x84308B30},
	{0xF965, 0x84308B31},
	{0xF966, 0x84308B32},
	{0xF967, 0x84308B33},
	{0xF968, 0x84308B34},
	{0xF969, 0x84308B35},
	{0xF96A, 0x84308B36},
	{0xF96B, 0x84308B37},
	{0xF96C, 0x84308B38},
	{0xF96D, 0x84308B39},
	{0xF96E, 0x84308C30},
	{0xF96F, 0x84308C31},
	{0xF970, 0x84308C32},
	{0xF971, 0x84308C33},
	{0xF972, 0x84308C34},
	{0xF973, 0x84308C35},
	{0xF974, 0x84308C36},
	{0xF975, 0x84308C37},
	{0xF976, 0x84308C38},
	{0xF977, 0x84308C39},
	{0xF978, 0x84308D30},
	{0xF97A, 0x84308D31},
	{0xF97B, 0x84308D32},
	{0xF97C, 0x84308D33},
	{0xF97D, 0x84308D34},
	{0xF97E, 0x84308D35},
	{0xF97F, 0x84308D36},
	{0xF980, 0x84308D37},
	{0xF981, 0x84308D38},
	{0xF982, 0x84308D39},
	{0xF983, 0x84308E30},
	{0xF984, 0x84308E31},
	{0xF985, 0x84308E32},
	{0xF986, 0x84308E33},
	{0xF987, 0x84308E34},
	{0xF988, 0x84308E35},
	{0xF989, 0x84308E36},
	{0xF98A, 0x84308E37},
	{0xF98B, 0x84308E38},
	{0xF98C, 0x84308E39},
	{0xF98D, 0x84308F30},
	{0xF98E, 0x84308F31},
	{0xF98F, 0x84308F32},
	{0xF990, 0x84308F33},
	{0xF991, 0x84308F34},
	{0xF992, 0x84308F35},
	{0xF993, 0x84308F36},
	{0xF994, 0x84308F37},
	{0xF996, 0x84308F38},
	{0xF997, 0x84308F39},
	{0xF998, 0x84309030},
	{0xF999, 0x84309031},
	{0xF99A, 0x84309032},
	{0xF99B, 0x84309033},
	{0xF99C, 0x84309034},
	{0xF99D, 0x84309035},
	{0xF99E, 0x84309036},
	{0xF99F, 0x84309037},
	{0xF9A0, 0x84309038},
	{0xF9A1, 0x84309039},
	{0xF9A2, 0x84309130},
	{0xF9A3, 0x84309131},
	{0xF9A4, 0x84309132},
	{0xF9A5, 0x84309133},
	{0xF9A6, 0x84309134},
	{0xF9A7, 0x84309135},
	{0xF9A8, 0x84309136},
	{0xF9A9, 0x84309137},
	{0xF9AA, 0x84309138},
	{0xF9AB, 0x84309139},
	{0xF9AC, 0x84309230},
	{0xF9AD, 0x84309231},
	{0xF9AE, 0x84309232},
	{0xF9AF, 0x84309233},
	{0xF9B0, 0x84309234},
	{0xF9B1, 0x84309235},
	{0xF9B2, 0x84309236},
	{0xF9B3, 0x84309237},
	{0xF9B4, 0x84309238},
	{0xF9B5, 0x84309239},
	{0xF9B6, 0x84309330},
	{0xF9B7, 0x84309331},
	{0xF9B8, 0x84309332},
	{0xF9B9, 0x84309333},
	{0xF9BA, 0x84309334},
	{0xF9BB, 0x84309335},
	{0xF9BC, 0x84309336},
	{0xF9BD, 0x84309337},
	{0xF9BE, 0x84309338},
	{0xF9BF, 0x84309339},
	{0xF9C0, 0x84309430},
	{0xF9C1, 0x84309431},
	{0xF9C2, 0x84309432},
	{0xF9C3, 0x84309433},
	{0xF9C4, 0x84309434},
	{0xF9C5, 0x84309435},
	{0xF9C6, 0x84309436},
	{0xF9C7, 0x84309437},
	{0xF9C8, 0x84309438},
	{0xF9C9, 0x84309439},
	{0xF9CA, 0x84309530},
	{0xF9CB, 0x84309531},
	{0xF9CC, 0x84309532},
	{0xF9CD, 0x84309533},
	{0xF9CE, 0x84309534},
	{0xF9CF, 0x84309535},
	{0xF9D0, 0x84309536},
	{0xF9D1, 0x84309537},
	{0xF9D2, 0x84309538},
	{0xF9D3, 0x84309539},
	{0xF9D4, 0x84309630},
	{0xF9D5, 0x84309631},
	{0xF9D6, 0x84309632},
	{0xF9D7, 0x84309633},
	{0xF9D8, 0x84309634},
	{0xF9D9, 0x84309635},
	{0xF9DA, 0x84309636},
	{0xF9DB, 0x84309637},
	{0xF9DC, 0x84309638},
	{0xF9DD, 0x84309639},
	{0xF9DE, 0x84309730},
	{0xF9DF, 0x84309731},
	{0xF9E0, 0x84309732},
	{0xF9E1, 0x84309733},
	{0xF9E2, 0x84309734},
	{0xF9E3, 0x84309735},
	{0xF9E4, 0x84309736},
	{0xF9E5, 0x84309737},
	{0xF9E6, 0x84309738},
	{0xF9E8, 0x84309739},
	{0xF9E9, 0x84309830},
	{0xF9EA, 0x84309831},
	{0xF9EB, 0x84309832},
	{0xF9EC, 0x84309833},
	{0xF9ED, 0x84309834},
	{0xF9EE, 0x84309835},
	{0xF9EF, 0x84309836},
	{0xF9F0, 0x84309837},
	{0xF9F2, 0x84309838},
	{0xF9F3, 0x84309839},
	{0xF9F4, 0x84309930},
	{0xF9F5, 0x84309931},
	{0xF9F6, 0x84309932},
	{0xF9F7, 0x84309933},
	{0xF9F8, 0x84309934},
	{0xF9F9, 0x84309935},
	{0xF9FA, 0x84309936},
	{0xF9FB, 0x84309937},
	{0xF9FC, 0x84309938},
	{0xF9FD, 0x84309939},
	{0xF9FE, 0x84309A30},
	{0xF9FF, 0x84309A31},
	{0xFA00, 0x84309A32},
	{0xFA01, 0x84309A33},
	{0xFA02, 0x84309A34},
	{0xFA03, 0x84309A35},
	{0xFA04, 0x84309A36},
	{0xFA05, 0x84309A37},
	{0xFA06, 0x84309A38},
	{0xFA07, 0x84309A39},
	{0xFA08, 0x84309B30},
	{0xFA09, 0x84309B31},
	{0xFA0A, 0x84309B32},
	{0xFA0B, 0x84309B33},
	{0xFA10, 0x84309B34},
	{0xFA12, 0x84309B35},
	{0xFA15, 0x84309B36},
	{0xFA16, 0x84309B37},
	{0xFA17, 0x84309B38},
	{0xFA19, 0x84309B39},
	{0xFA1A, 0x84309C30},
	{0xFA1B, 0x84309C31},
	{0xFA1C, 0x84309C32},
	{0xFA1D, 0x84309C33},
	{0xFA1E, 0x84309C34},
	{0xFA22, 0x84309C35},
	{0xFA25, 0x84309C36},
	{0xFA26, 0x84309C37},
	{0xFE32, 0x84318538},
	{0xFE45, 0x84318539},
	{0xFE46, 0x84318630},
	{0xFE47, 0x84318631},
	{0xFE48, 0x84318632},
	{0xFE53, 0x84318633},
	{0xFE58, 0x84318634},
	{0xFE67, 0x84318635},
	{0xFE6C, 0x84318636},
	{0xFE6D, 0x84318637},
	{0xFE6E, 0x84318638},
	{0xFE6F, 0x84318639},
	{0xFE70, 0x84318730},
	{0xFE71, 0x84318731},
	{0xFE72, 0x84318732},
	{0xFE73, 0x84318733},
	{0xFE74, 0x84318734},
	{0xFE75, 0x84318735},
	{0xFE76, 0x84318736},
	{0xFE77, 0x84318737},
	{0xFE78, 0x84318738},
	{0xFE79, 0x84318739},
	{0xFE7A, 0x84318830},
	{0xFE7B, 0x84318831},
	{0xFE7C, 0x84318832},
	{0xFE7D, 0x84318833},
	{0xFE7E, 0x84318834},
	{0xFE7F, 0x84318835},
	{0xFE80, 0x84318836},
	{0xFE81, 0x84318837},
	{0xFE82, 0x84318838},
	{0xFE83, 0x84318839},
	{0xFE84, 0x84318930},
	{0xFE85, 0x84318931},
	{0xFE86, 0x84318932},
	{0xFE87, 0x84318933},
	{0xFE88, 0x84318934},
	{0xFE89, 0x84318935},
	{0xFE8A, 0x84318936},
	{0xFE8B, 0x84318937},
	{0xFE8C, 0x84318938},
	{0xFE8D, 0x84318939},
	{0xFE8E, 0x84318A30},
	{0xFE8F, 0x84318A31},
	{0xFE90, 0x84318A32},
	{0xFE91, 0x84318A33},
	{0xFE92, 0x84318A34},
	{0xFE93, 0x84318A35},
	{0xFE94, 0x84318A36},
	{0xFE95, 0x84318A37},
	{0xFE96, 0x84318A38},
	{0xFE97, 0x84318A39},
	{0xFE98, 0x84318B30},
	{0xFE99, 0x84318B31},
	{0xFE9A, 0x84318B32},
	{0xFE9B, 0x84318B33},
	{0xFE9C, 0x84318B34},
	{0xFE9D, 0x84318B35},
	{0xFE9E, 0x84318B36},
	{0xFE9F, 0x84318B37},
	{0xFEA0, 0x84318B38},
	{0xFEA1, 0x84318B39},
	{0xFEA2, 0x84318C30},
	{0xFEA3, 0x84318C31},
	{0xFEA4, 0x84318C32},
	{0xFEA5, 0x84318C33},
	{0xFEA6, 0x84318C34},
	{0xFEA7, 0x84318C35},
	{0xFEA8, 0x84318C36},
	{0xFEA9, 0x84318C37},
	{0xFEAA, 0x84318C38},
	{0xFEAB, 0x84318C39},
	{0xFEAC, 0x84318D30},
	{0xFEAD, 0x84318D31},
	{0xFEAE, 0x84318D32},
	{0xFEAF, 0x84318D33},
	{0xFEB0, 0x84318D34},
	{0xFEB1, 0x84318D35},
	{0xFEB2, 0x84318D36},
	{0xFEB3, 0x84318D37},
	{0xFEB4, 0x84318D38},
	{0xFEB5, 0x84318D39},
	{0xFEB6, 0x84318E30},
	{0xFEB7, 0x84318E31},
	{0xFEB8, 0x84318E32},
	{0xFEB9, 0x84318E33},
	{0xFEBA, 0x84318E34},
	{0xFEBB, 0x84318E35},
	{0xFEBC, 0x84318E36},
	{0xFEBD, 0x84318E37},
	{0xFEBE, 0x84318E38},
	{0xFEBF, 0x84318E39},
	{0xFEC0, 0x84318F30},
	{0xFEC1, 0x84318F31},
	{0xFEC2, 0x84318F32},
	{0xFEC3, 0x84318F33},
	{0xFEC4, 0x84318F34},
	{0xFEC5, 0x84318F35},
	{0xFEC6, 0x84318F36},
	{0xFEC7, 0x84318F37},
	{0xFEC8, 0x84318F38},
	{0xFEC9, 0x84318F39},
	{0xFECA, 0x84319030},
	{0xFECB, 0x84319031},
	{0xFECC, 0x84319032},
	{0xFECD, 0x84319033},
	{0xFECE, 0x84319034},
	{0xFECF, 0x84319035},
	{0xFED0, 0x84319036},
	{0xFED1, 0x84319037},
	{0xFED2, 0x84319038},
	{0xFED3, 0x84319039},
	{0xFED4, 0x84319130},
	{0xFED5, 0x84319131},
	{0xFED6, 0x84319132},
	{0xFED7, 0x84319133},
	{0xFED8, 0x84319134},
	{0xFED9, 0x84319135},
	{0xFEDA, 0x84319136},
	{0xFEDB, 0x84319137},
	{0xFEDC, 0x84319138},
	{0xFEDD, 0x84319139},
	{0xFEDE, 0x84319230},
	{0xFEDF, 0x84319231},
	{0xFEE0, 0x84319232},
	{0xFEE1, 0x84319233},
	{0xFEE2, 0x84319234},
	{0xFEE3, 0x84319235},
	{0xFEE4, 0x84319236},
	{0xFEE5, 0x84319237},
	{0xFEE6, 0x84319238},
	{0xFEE7, 0x84319239},
	{0xFEE8, 0x84319330},
	{0xFEE9, 0x84319331},
	{0xFEEA, 0x84319332},
	{0xFEEB, 0x84319333},
	{0xFEEC, 0x84319334},
	{0xFEED, 0x84319335},
	{0xFEEE, 0x84319336},
	{0xFEEF, 0x84319337},
	{0xFEF0, 0x84319338},
	{0xFEF1, 0x84319339},
	{0xFEF2, 0x84319430},
	{0xFEF3, 0x84319431},
	{0xFEF4, 0x84319432},
	{0xFEF5, 0x84319433},
	{0xFEF6, 0x84319434},
	{0xFEF7, 0x84319435},
	{0xFEF8, 0x84319436},
	{0xFEF9, 0x84319437},
	{0xFEFA, 0x84319438},
	{0xFEFB, 0x84319439},
	{0xFEFC, 0x84319530},
	{0xFEFD, 0x84319531},
	{0xFEFE, 0x84319532},
	{0xFEFF, 0x84319533},
	{0xFF00, 0x84319534},
	{0xFF5F, 0x84319535},
	{0xFF60, 0x84319536},
	{0xFF61, 0x84319537},
	{0xFF62, 0x84319538},
	{0xFF63, 0x84319539},
	{0xFF64, 0x84319630},
	{0xFF65, 0x84319631},
	{0xFF66, 0x84319632},
	{0xFF67, 0x84319633},
	{0xFF68, 0x84319634},
	{0xFF69, 0x84319635},
	{0xFF6A, 0x84319636},
	{0xFF6B, 0x84319637},
	{0xFF6C, 0x84319638},
	{0xFF6D, 0x84319639},
	{0xFF6E, 0x84319730},
	{0xFF6F, 0x84319731},
	{0xFF70, 0x84319732},
	{0xFF71, 0x84319733},
	{0xFF72, 0x84319734},
	{0xFF73, 0x84319735},
	{0xFF74, 0x84319736},
	{0xFF75, 0x84319737},
	{0xFF76, 0x84319738},
	{0xFF77, 0x84319739},
	{0xFF78, 0x84319830},
	{0xFF79, 0x84319831},
	{0xFF7A, 0x84319832},
	{0xFF7B, 0x84319833},
	{0xFF7C, 0x84319834},
	{0xFF7D, 0x84319835},
	{0xFF7E, 0x84319836},
	{0xFF7F, 0x84319837},
	{0xFF80, 0x84319838},
	{0xFF81, 0x84319839},
	{0xFF82, 0x84319930},
	{0xFF83, 0x84319931},
	{0xFF84, 0x84319932},
	{0xFF85, 0x84319933},
	{0xFF86, 0x84319934},
	{0xFF87, 0x84319935},
	{0xFF88, 0x84319936},
	{0xFF89, 0x84319937},
	{0xFF8A, 0x84319938},
	{0xFF8B, 0x84319939},
	{0xFF8C, 0x84319A30},
	{0xFF8D, 0x84319A31},
	{0xFF8E, 0x84319A32},
	{0xFF8F, 0x84319A33},
	{0xFF90, 0x84319A34},
	{0xFF91, 0x84319A35},
	{0xFF92, 0x84319A36},
	{0xFF93, 0x84319A37},
	{0xFF94, 0x84319A38},
	{0xFF95, 0x84319A39},
	{0xFF96, 0x84319B30},
	{0xFF97, 0x84319B31},
	{0xFF98, 0x84319B32},
	{0xFF99, 0x84319B33},
	{0xFF9A, 0x84319B34},
	{0xFF9B, 0x84319B35},
	{0xFF9C, 0x84319B36},
	{0xFF9D, 0x84319B37},
	{0xFF9E, 0x84319B38},
	{0xFF9F, 0x84319B39},
	{0xFFA0, 0x84319C30},
	{0xFFA1, 0x84319C31},
	{0xFFA2, 0x84319C32},
	{0xFFA3, 0x84319C33},
	{0xFFA4, 0x84319C34},
	{0xFFA5, 0x84319C35},
	{0xFFA6, 0x84319C36},
	{0xFFA7, 0x84319C37},
	{0xFFA8, 0x84319C38},
	{0xFFA9, 0x84319C39},
	{0xFFAA, 0x84319D30},
	{0xFFAB, 0x84319D31},
	{0xFFAC, 0x84319D32},
	{0xFFAD, 0x84319D33},
	{0xFFAE, 0x84319D34},
	{0xFFAF, 0x84319D35},
	{0xFFB0, 0x84319D36},
	{0xFFB1, 0x84319D37},
	{0xFFB2, 0x84319D38},
	{0xFFB3, 0x84319D39},
	{0xFFB4, 0x84319E30},
	{0xFFB5, 0x84319E31},
	{0xFFB6, 0x84319E32},
	{0xFFB7, 0x84319E33},
	{0xFFB8, 0x84319E34},
	{0xFFB9, 0x84319E35},
	{0xFFBA, 0x84319E36},
	{0xFFBB, 0x84319E37},
	{0xFFBC, 0x84319E38},
	{0xFFBD, 0x84319E39},
	{0xFFBE, 0x84319F30},
	{0xFFBF, 0x84319F31},
	{0xFFC0, 0x84319F32},
	{0xFFC1, 0x84319F33},
	{0xFFC2, 0x84319F34},
	{0xFFC3, 0x84319F35},
	{0xFFC4, 0x84319F36},
	{0xFFC5, 0x84319F37},
	{0xFFC6, 0x84319F38},
	{0xFFC7, 0x84319F39},
	{0xFFC8, 0x8431A030},
	{0xFFC9, 0x8431A031},
	{0xFFCA, 0x8431A032},
	{0xFFCB, 0x8431A033},
	{0xFFCC, 0x8431A034},
	{0xFFCD, 0x8431A035},
	{0xFFCE, 0x8431A036},
	{0xFFCF, 0x8431A037},
	{0xFFD0, 0x8431A038},
	{0xFFD1, 0x8431A039},
	{0xFFD2, 0x8431A130},
	{0xFFD3, 0x8431A131},
	{0xFFD4, 0x8431A132},
	{0xFFD5, 0x8431A133},
	{0xFFD6, 0x8431A134},
	{0xFFD7, 0x8431A135},
	{0xFFD8, 0x8431A136},
	{0xFFD9, 0x8431A137},
	{0xFFDA, 0x8431A138},
	{0xFFDB, 0x8431A139},
	{0xFFDC, 0x8431A230},
	{0xFFDD, 0x8431A231},
	{0xFFDE, 0x8431A232},
	{0xFFDF, 0x8431A233},
}
